--[[
StoreManagerExtension.lua
	
Author: 	Ifko[nator]
Date:		03.10.2022
Version:	2.2

History:	v1.0 @04.07.2020 - initial implementation in FS 19
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.1 @11.07.2020 - added support for all ingame languages
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.2 @11.08.2020 - default seed is now more expensive
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.3 @27.08.2020 - reworkt function to load fill types, so it is not mess up with other mods
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.4 @29.08.2020 - store images for english and french players will now load correctly
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.5 @28.10.2020 - some script optimation
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.6 @10.02.2021 - removed a print call
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.7 @14.02.2021 - reworkt function to remove seed pallets/bigBags from shop wich fill type is not supportet by the current map
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.8 @05.06.2021 - some script optimation
            --------------------------------------------------------------------------------------------------------------------------------------------------
            v1.9 @13.06.2021 - minior adjustments
            --------------------------------------------------------------------------------------------------------------------------------------------------
			v2.0 @17.03.2022 - convert to FS 22
            --------------------------------------------------------------------------------------------------------------------------------------------------
			v2.1 @24.05.2022 - added posibility to change the category images for different languages
            --------------------------------------------------------------------------------------------------------------------------------------------------
			v2.2 @03.10.2022 - added posibility to load mod category via modDesc for the Realistic Seeder ( <realisticSeeder addCategoryFromThisMod="true"/> )
]]

StoreManagerExtension = {};

StoreManagerExtension.BASE_KEY = "modDesc.storeItems";
StoreManagerExtension.STORE_ITEM_IMAGE_KEY = StoreManagerExtension.BASE_KEY .. ".storeImages.storeImage_";
StoreManagerExtension.CATEGORY_IMAGE_KEY = StoreManagerExtension.BASE_KEY .. ".categoryImages.categoryImage_";

function StoreManagerExtension:onStartMission()
    if RealisticSeederUtil == nil then
		return;
	end;
    
    StoreManagerExtension.DEBUG_PRIORITY = RealisticSeederUtil.getDebugPriority(RealisticSeederUtil.modDesc, "modDesc.realisticSeeder#debugPriority");

    if RealisticSeederUtil.languageShort ~= "de" then
        local newStoreImages = {};
        local storeImageNumber = 0;

        while true do
            local storeImageKey = StoreManagerExtension.STORE_ITEM_IMAGE_KEY .. RealisticSeederUtil.languageShort .. "(" .. tostring(storeImageNumber) .. ")";

            if not hasXMLProperty(RealisticSeederUtil.modDesc, storeImageKey) then
                storeImageKey = StoreManagerExtension.STORE_ITEM_IMAGE_KEY .. "en(" .. tostring(storeImageNumber) .. ")";

                if not hasXMLProperty(RealisticSeederUtil.modDesc, storeImageKey) then
                    break;
                end;
            end;

            local storeImage_de = Utils.getNoNil(getXMLString(RealisticSeederUtil.modDesc, StoreManagerExtension.STORE_ITEM_IMAGE_KEY .. "de(" .. tostring(storeImageNumber) .. ")#file"), "");
            local storeImage_currentLanguageShort = Utils.getNoNil(getXMLString(RealisticSeederUtil.modDesc, storeImageKey .. "#file"), "");

            local newStoreImage = {};

            if storeImage_de ~= "" and storeImage_currentLanguageShort ~= "" then
                newStoreImage.de = storeImage_de;
                newStoreImage.currentLanguageShort = storeImage_currentLanguageShort;

                table.insert(newStoreImages, newStoreImage);
            end;

            storeImageNumber = storeImageNumber + 1;
        end;

        for _, storeItem in pairs(g_storeManager.items) do
            if storeItem.imageFilename ~= nil and string.find(storeItem.imageFilename, RealisticSeederUtil.currentModName) then
                for _, newStoreImage in pairs(newStoreImages) do 
                    if string.find(storeItem.imageFilename, newStoreImage.de) then 
                        --## swap store image
                        storeItem.imageFilename = Utils.getFilename(newStoreImage.currentLanguageShort, RealisticSeederUtil.currentModDirectory);

                        RealisticSeederUtil.printDebug("Swaped store image  " .. newStoreImage.de .. " with " .. storeItem.imageFilename .. ".", StoreManagerExtension.DEBUG_PRIORITY, true, "StoreManagerExtension");
                    end;
                end;
            end;
        end;
    end;

    for _, storeItem in pairs(g_storeManager:getItems()) do
        if string.find(storeItem.xmlFilename, "data/objects/bigBagContainer/bigBagContainerSeeds.xml") or string.find(storeItem.xmlFilename, "data/objects/pallets/pioneer/pioneerPallet.xml") then
            --## make default seed more expensive
            storeItem.price = storeItem.price * 1.5;
        end;

        if storeItem.isMod then
            local modDesc = loadXMLFile("modsDesc", storeItem.baseDir .. "modDesc.xml");
            local xmlFile = loadXMLFile("vehicle", storeItem.xmlFilename);
        
            RealisticSeederUtil.loadSupportedModCategories(getXMLString(xmlFile, "vehicle.storeData.category"), Utils.getNoNil(getXMLBool(modDesc, "modDesc.realisticSeeder#addCategoryFromThisMod"), false));
        
            delete(modDesc);
            delete(xmlFile);
        end;
    end;

    local fillTypes = RealisticSeederUtil.getSeedFillTypes();
    
    if fillTypes ~= nil then
        for _, fillType in pairs(fillTypes) do
            --## check if selected fruit is available in current map
            for _, storeItem in pairs(g_storeManager.items) do
                if not RealisticSeederUtil.getCanFruitBePlanted(fillType, nil) and string.find(string.upper(storeItem.xmlFilename), g_fillTypeManager:getFillTypeByIndex(fillType).name) then
                    storeItem.showInStore = false;

                    RealisticSeederUtil.printDebug("Removed store item " .. storeItem.xmlFilename .. " from the shop due to not supportet fillType (" .. g_fillTypeManager:getFillTypeByIndex(fillType).title .. ")!", StoreManagerExtension.DEBUG_PRIORITY, true, "StoreManagerExtension");
                end;
            end;
        end;
    end;

    if g_fillTypeManager:getFillTypeByName("SEEDS").palletFilename ~= nil then
        if g_fillTypeManager:getFillTypeByName("SEEDS").palletFilename == "data/objects/bigBagContainer/bigBagContainerSeeds.xml" then
           
            --## change default seed big bag with pioneer pallet for unloading
            g_fillTypeManager:getFillTypeByName("SEEDS").palletFilename = "data/objects/pallets/pioneer/pioneerPallet.xml";
        end;
    end;
end;

FSBaseMission.onStartMission = Utils.prependedFunction(FSBaseMission.onStartMission, StoreManagerExtension.onStartMission);

function StoreManagerExtension:load(superFunc)
	if not self.isInitialized then
		local foundBrands = {};
		local foundCategory = {};
		local foundDLCs = {};

		for _, storeItem in ipairs(self.storeManager:getItems()) do
			if storeItem.categoryName ~= "" and storeItem.showInStore and (storeItem.extraContentId == nil or g_extraContentSystem:getIsItemIdUnlocked(storeItem.extraContentId)) then
				local brand = self.brandManager:getBrandByIndex(storeItem.brandIndex);

				if brand ~= nil and not foundBrands[storeItem.brandIndex] and storeItem.species ~= "placeable" then
					foundBrands[storeItem.brandIndex] = true;

					if brand.name ~= "NONE" then
						self:addBrandForDisplay(brand);
					end;
				end;

				local category = self.storeManager:getCategoryByName(storeItem.categoryName);

				if category ~= nil and not foundCategory[storeItem.categoryName] then
					if RealisticSeederUtil ~= nil and RealisticSeederUtil.languageShort ~= "de" then
                        StoreManagerExtension.DEBUG_PRIORITY = RealisticSeederUtil.getDebugPriority(RealisticSeederUtil.modDesc, "modDesc.realisticSeeder#debugPriority");
                        
                        local categoryImageNumber = 0;
        
                        while true do
                            local categoryImageKey = StoreManagerExtension.CATEGORY_IMAGE_KEY .. RealisticSeederUtil.languageShort .. "(" .. tostring(categoryImageNumber) .. ")";
            
                            if not hasXMLProperty(RealisticSeederUtil.modDesc, categoryImageKey) then
                                categoryImageKey = StoreManagerExtension.CATEGORY_IMAGE_KEY .. "en(" .. tostring(categoryImageNumber) .. ")";

                                if not hasXMLProperty(RealisticSeederUtil.modDesc, categoryImageKey) then
                                    break;
                                end;
                            end;
            
                            local categoryName = string.upper(Utils.getNoNil(getXMLString(RealisticSeederUtil.modDesc, categoryImageKey .. "#name"), ""));
                            local newCategoryImage = Utils.getNoNil(getXMLString(RealisticSeederUtil.modDesc, categoryImageKey .. "#file"), "");
            
                            if categoryName ~= "" and newCategoryImage ~= "" then
                                if string.upper(category.name) == categoryName then
                                    category.oldImage = category.image;
                                    category.image = Utils.getFilename(newCategoryImage, RealisticSeederUtil.currentModDirectory);

                                    RealisticSeederUtil.printDebug("Swaped category image  " .. category.oldImage .. " with " .. category.image .. ".", StoreManagerExtension.DEBUG_PRIORITY, true, "StoreManagerExtension");
                                end;
                            end;
            
                            categoryImageNumber = categoryImageNumber + 1;
                        end;
                    end;
                    
                    foundCategory[storeItem.categoryName] = true;

					self:addCategoryForDisplay(category);
				end;

				if storeItem.customEnvironment ~= nil and foundDLCs[storeItem.customEnvironment] == nil and storeItem.species ~= "placeable" then
					local info = {
						id = storeItem.customEnvironment,
						isMod = storeItem.isMod,
						label = storeItem.dlcTitle,
						iconFilename = g_modManager:getModByName(storeItem.customEnvironment).iconFilename,
						sortValue = (not storeItem.isMod and "    " or "") .. storeItem.dlcTitle
					};

					foundDLCs[storeItem.customEnvironment] = true;

					table.insert(self.displayDLCs, info);
				end;
			end;
		end;

		for _, pack in pairs(self.storeManager:getPacks()) do
			table.insert(self.displayPacks, {
				id = pack.name,
				iconFilename = pack.image,
				label = pack.title,
				sortValue = pack.orderId
			});
		end;

		if Platform.hasInAppPurchases then
			self:addCategoryForDisplay(self.storeManager:getCategoryByName(ShopController.COINS_CATEGORY));
		end;

		table.sort(self.displayBrands, ShopController.brandSortFunction);
		table.sort(self.displayToolCategories, ShopController.categorySortFunction);
		table.sort(self.displayObjectCategories, ShopController.categorySortFunction);
		table.sort(self.displayPlaceableCategories, ShopController.categorySortFunction);
		table.sort(self.displayVehicleCategories, ShopController.categorySortFunction);
		table.sort(self.displayPacks, ShopController.categorySortFunction);
		table.sort(self.displayDLCs, ShopController.brandSortFunction);

		self.isInitialized = true;
	end;
end;

ShopController.load = Utils.overwrittenFunction(ShopController.load, StoreManagerExtension.load);

addModEventListener(StoreManagerExtension);