--[[
RealisticSeederUtil

Utility for my Realistic Seeder

Author:		Ifko[nator]
Datum:		22.11.2022
Version:	v3.0

History:	v1.0 @12.09.2020 - initial implementation in FS 19
			-----------------------------------------------------------------------------------------------------------
			v1.1 @15.02.2021 - reworkt changeMaterial function
			-----------------------------------------------------------------------------------------------------------
			v1.2 @06.06.2021 - added "getFertilizerFillUnitIndex" and "getSeedFillTypes" functions
			-----------------------------------------------------------------------------------------------------------
			v1.3 @13.06.2021 - minior adjustments
			-----------------------------------------------------------------------------------------------------------
			v1.4 @15.06.2021 - added "getIsValidPalletName" function
			-----------------------------------------------------------------------------------------------------------
			v1.5 @21.06.2021 - added "getLoadedFillTypes" and removed "getFertilizerFillUnitIndex" functions
			-----------------------------------------------------------------------------------------------------------
			v1.6 @23.06.2021 - added "fixFillTypeName" function
			-----------------------------------------------------------------------------------------------------------
			v1.7 @07.07.2021 - added "getIsUnSupportetValue" function
							   added "unSupportetHofBergmannVersions" table
			-----------------------------------------------------------------------------------------------------------
			v1.8 @31.07.2021 - added "getIsValidStoreCategoryName" function
			-----------------------------------------------------------------------------------------------------------
			v1.9 @30.09.2021 - added "getConfigText" function
			-----------------------------------------------------------------------------------------------------------
			v2.0 @27.05.2022 - removed unnecessary's functions
			-----------------------------------------------------------------------------------------------------------
			v2.1 @09.06.2022 - added "fillTypeCategoriesForSeeds" table
			-----------------------------------------------------------------------------------------------------------
			v2.2 @12.06.2022 - added "getCurrentLanguage", "getStorePrefix" and "getCurrentOverlayName" functions
							   added "palletTypes" table
			-----------------------------------------------------------------------------------------------------------
			v2.3 @18.06.2022 - added "isValidFillTypeName" and "getForceBagOverlay" functions
			-----------------------------------------------------------------------------------------------------------
			v2.4 @19.06.2022 - added "inputs" table and "foundFillTypeNameInTable" function
			-----------------------------------------------------------------------------------------------------------
			v2.5 @22.06.2022 - added "getNumberOfFillUnits" and "getFillTypeNameDoNotSupportThreeUnloadModes" functions
			-----------------------------------------------------------------------------------------------------------
			v2.6 @17.09.2022 - added "getForcedFillUnitIndex" function
			-----------------------------------------------------------------------------------------------------------
			v2.6 @27.09.2022 - added "getIsAllowedToUnloadInBigBags" and "getIsRealisticSeederBigBagFillType" functions
			-----------------------------------------------------------------------------------------------------------
			v2.7 @29.09.2022 - added "getPalletFilename" function
			-----------------------------------------------------------------------------------------------------------
			v2.8 @03.10.2022 - added "getIsUnloadActionEventAllowed" and "loadSupportedModCategories" functions
			-----------------------------------------------------------------------------------------------------------
			v2.9 @18.11.2022 - added "SELECT_FERTILIZER_MODE" input
			-----------------------------------------------------------------------------------------------------------
			v3.0 @22.11.2022 - added "SELECT_UNLOAD_SIDE" input
]]

RealisticSeederUtil = {};
RealisticSeederUtil.currentModName = g_currentModName;
RealisticSeederUtil.languageShort = g_languageShort;
RealisticSeederUtil.currentModDirectory = g_currentModDirectory;
RealisticSeederUtil.categoryPrefix = "REALISTICSEEDER";

RealisticSeederUtil.validStoreCategories = {
    "seeders",
    "planters",
	"potatoHarvesting",
	"weeders",
	"discHarrows",
	"cultivators",
	"subsoilers",
	"AmazoneTOOL",
	"KvernelandTOOL",
	"lemkenTOOL",
	"KuhnTOOL"
};

RealisticSeederUtil.unSupportetHofBergmannVersions = {
	
};

RealisticSeederUtil.fillTypeCategoriesForSeeds = {
	"BULK",
	"SHOVEL",
	"AUGERWAGON",
	"TRAINWAGON",
	"LOADINGVEHICLE"
};

RealisticSeederUtil.palletTypes = {
	"bigBags",
	"bigBags_2",
	"pallets",
	"bags"
};

RealisticSeederUtil.inputs = {
	"UNLOAD_BUTTON",
	"SELECT_UNLOAD_MODE",
	"SELECT_FILL_UNIT",
	"SELECT_FERTILIZER_MODE",
	"SELECT_UNLOAD_SIDE"
};

RealisticSeederUtil.l10nTexts = {};

local l10nTextsToLoad = {
	"input_UNLOAD_BUTTON",
	"input_SELECT_UNLOAD_MODE",
	"input_SELECT_FILL_UNIT",
	"input_SELECT_FERTILIZER_MODE",
	"input_SELECT_UNLOAD_SIDE",
	"category_pallets",
	"action_bags",
	"action_unloadXToX",
	"action_workMode_right",
	"action_workMode_left",
	"action_switchUnloadSideToX",
	"action_selectedFillUnitX",
	"action_activateFerilizing",
	"action_deactivateFerilizing",
	"action_dynamicPalletAttach",
	"action_dynamicPalletDetach"
};

local function getL10nText(text)
	if text ~= nil then
		if text:sub(1, 6) == "$l10n_" then
			text = g_i18n:getText(text:sub(7), RealisticSeederUtil.currentModName);
		elseif g_i18n:hasText(text, RealisticSeederUtil.currentModName) then
			text = g_i18n:getText(text, RealisticSeederUtil.currentModName);
		end;
	end;

	return text;
end;

for _, l10nTextToLoad in pairs(l10nTextsToLoad) do
	RealisticSeederUtil.l10nTexts[l10nTextToLoad] = getL10nText(l10nTextToLoad);
end;

if RealisticSeederUtil.modDesc == nil then
	RealisticSeederUtil.modDesc = loadXMLFile("modDesc", RealisticSeederUtil.currentModDirectory .. "modDesc.xml");
end;

RealisticSeederUtil.fillTypesPath = RealisticSeederUtil.currentModDirectory .. getXMLString(RealisticSeederUtil.modDesc, "modDesc.fillTypes#filename");
	
if fileExists(RealisticSeederUtil.fillTypesPath) then
	RealisticSeederUtil.fillTypesXML = loadXMLFile("fillTypes", RealisticSeederUtil.fillTypesPath);
end;

getfenv(0)["RealisticSeederUtil"] = RealisticSeederUtil;

function RealisticSeederUtil.getDebugPriority(xmlFile, path)
	return Utils.getNoNil(getXMLInt(xmlFile, path), 0);
end;

function RealisticSeederUtil.printError(errorMessage, isWarning, isInfo, luaName)
	local prefix = "::ERROR:: ";
	
	if isWarning then
		prefix = "::WARNING:: ";
	elseif isInfo then
		prefix = "::INFO:: ";
	end;
	
	print(prefix .. "from the " .. luaName .. ".lua: " .. tostring(errorMessage));
end;

function RealisticSeederUtil.printDebug(debugMessage, priority, addString, luaName)
	if priority >= 1 then
		local prefix = "";
		
		if addString then
			prefix = "::DEBUG:: from the " .. luaName .. ".lua: ";
		end;
		
		print(prefix .. tostring(debugMessage));
	end;
end;

function RealisticSeederUtil.getSpecByName(self, specName, currentModName)
    local spec = self["spec_" .. Utils.getNoNil(currentModName, RealisticSeederUtil.currentModName) .. "." .. specName];

	if spec ~= nil then
        return spec;
    end;

    return self["spec_" .. specName];
end;

function RealisticSeederUtil.getIsValidStoreCategory(storeCategory)
    for _, validStoreCategory in pairs(RealisticSeederUtil.validStoreCategories) do
        if storeCategory == validStoreCategory then
            return true;
        end;
    end;

    return false;
end;

function RealisticSeederUtil.getIsValidPalletName(palletName)
	return (not string.find(palletName, "PALLETFORK") and not string.find(palletName, "TRIGGER")) and (string.find(palletName, "PALLET") or string.find(palletName, "BIGBAG") or string.find(palletName, "TANK"));
end;

function RealisticSeederUtil.foundFunctionAdditionalTank(xmlFile)
    local foundFunctionAdditionalTank = false;
	local additionalTankNumber = 0;

	while true do
		local functionAditionalTankKey = "vehicle.storeData.functions.function(" .. additionalTankNumber .. ")";
		
		if not hasXMLProperty(xmlFile, functionAditionalTankKey) then
			break;
		end;

		if getXMLString(xmlFile, functionAditionalTankKey) == "$l10n_function_additionalTank" then
			foundFunctionAdditionalTank = true;

			break;
		end;

		additionalTankNumber = additionalTankNumber + 1;
    end;
    
    return foundFunctionAdditionalTank;
end;

function RealisticSeederUtil.getSeedFillUnitIndex(xmlFile, object, specFillUnit)
	local seedFillUnitIndex = 1;
	
	local baseKey = string.format("vehicle.fillUnit.fillUnitConfigurations.fillUnitConfiguration(%d).fillUnits", Utils.getNoNil(object.configurations["fillUnit"], 1) - 1);

	local fillUnitNumber = 0;

	while true do
		local fillUnitKey = string.format("%s.fillUnit(%d)", baseKey, fillUnitNumber);
		
        if not hasXMLProperty(xmlFile, fillUnitKey) then
            break;
		end;

		for fillTypeIndex, _ in pairs(specFillUnit.fillUnits[fillUnitNumber + 1].supportedFillTypes) do
			if string.find(g_fillTypeManager:getFillTypeByIndex(fillTypeIndex).name, "SEED") then
				seedFillUnitIndex = fillUnitNumber + 1;

				break;
			end;
		end;

		fillUnitNumber = fillUnitNumber + 1;
    end;
    
    return seedFillUnitIndex;
end;

function RealisticSeederUtil.getLoadedFillTypes(object, getIsActiveForInput, specFillUnit, numFillUnits)
	local isSeedFillType, isFertilizerFillType, isLimeFillType, allowDefaultUnload, fillUnitFillType, seedFillTypeIndex, fillUnitIndex = false, false, false, false, g_fillTypeManager:getFillTypeByIndex(FillType.UNKNOWN), 0, 0;

	if getIsActiveForInput then
		for fillUnitNumber = 1, numFillUnits do
			local fillUnit = specFillUnit.fillUnits[fillUnitNumber];

			if fillUnit ~= nil and fillUnit.showOnHud then
				fillUnitFillType = g_fillTypeManager:getFillTypeByIndex(object:getFillUnitFillType(fillUnitNumber));

				if fillUnitFillType ~= nil then
					fillUnitIndex = fillUnitNumber;
					
					if string.find(fillUnitFillType.name, "SEED") then
						isSeedFillType = true;

						seedFillTypeIndex = object:getFillUnitFillType(fillUnitNumber);
					end;
					
					if fillUnitFillType.name == "FERTILIZER" then
						isFertilizerFillType = true;
					end;
					
					if fillUnitFillType.name == "LIME" then
						isLimeFillType = true;
					end;
					
					if fillUnitFillType.name == "HERBICIDE" or fillUnitFillType.name == "LIQUIDFERTILIZER" then
						allowDefaultUnload = true;
					end;
				end;
			end;
		end;
	end;

	return isSeedFillType, isFertilizerFillType, isLimeFillType, allowDefaultUnload, fillUnitFillType, seedFillTypeIndex, fillUnitIndex;
end;

function RealisticSeederUtil.fixFillTypeName(fillTypeName)
	if fillTypeName == "seed_sugarbeet" then
		fillTypeName = "seed_sugarBeet";
	elseif fillTypeName == "sugarbeet" then
		fillTypeName = "sugarBeet";
	elseif fillTypeName == "seed_oilseedradish" then
		fillTypeName = "seed_oilseedRadish";
	elseif fillTypeName == "oilseedradish" then
		fillTypeName = "oilseedRadish";
	end;

	return fillTypeName;
end;

function RealisticSeederUtil.changeMaterial(object, materialNode, translateNode)
	if object.components == nil then
		local newMaterial = getMaterial(translateNode, 0);
		local numChildren = getNumOfChildren(materialNode);

		if numChildren > 0 then
			for childNumber = 0, numChildren - 1 do
				local oldMaterial = getMaterial(getChildAt(materialNode, childNumber), 0);

				ConfigurationUtil.replaceMaterialRec(object, getChildAt(materialNode, childNumber), oldMaterial, newMaterial);
			end;
		else
			local oldMaterial = getMaterial(materialNode, 0);

			ConfigurationUtil.replaceMaterialRec(object, materialNode, oldMaterial, newMaterial);
		end;
	else
		local oldMaterial = getMaterial(materialNode, 0);
		local newMaterial = getMaterial(translateNode, 0);
	
		for _, component in pairs(object.components) do
			ConfigurationUtil.replaceMaterialRec(object, component.node, oldMaterial, newMaterial);
		end;
	end;
end;

function RealisticSeederUtil.getSeedFillTypes()
	local fillTypes = nil;

	if RealisticSeederUtil.fillTypesXML ~= nil then
		local fillTypeNames = "";
    	local fillTypeNumber = 0;

    	while true do
			local fillTypeKey = "map.fillTypes.fillType(" .. tostring(fillTypeNumber) .. ")";

			if not hasXMLProperty(RealisticSeederUtil.fillTypesXML, fillTypeKey) then
				break;
			end;

    	    local fillType = Utils.getNoNil(getXMLString(RealisticSeederUtil.fillTypesXML, fillTypeKey .. "#name"), "");

    	    if fillType ~= "" then
				if fillTypeNames ~= "" then 
					fillType = " " .. fillType;
				end;

				fillTypeNames = fillTypeNames .. fillType;
			end;
		
    	    fillTypeNumber = fillTypeNumber + 1;
    	end;

		fillTypes = g_fillTypeManager:getFillTypesByNames(fillTypeNames, "Warning: invalid fillType '%s'.");
	end;

	return fillTypes;
end;

function RealisticSeederUtil.getModByTitle(germanTitle, englishTitle)
	local mod, modName, modDir = nil, "", "";

	if englishTitle == nil then
		englishTitle = germanTitle;
	end;

	if germanTitle == nil then
		return nil, "", "";
	end;
	
	for _, mod in pairs(g_modManager.mods) do
		if mod.title == germanTitle or mod.title == englishTitle then		
			if g_modIsLoaded[tostring(mod.modName)] then	
				mod = mod;
				modName = mod.modName;
				modDir = mod.modDir;

				break;
			end;
		end;
	end;

	return mod, modName, modDir;
end;

function RealisticSeederUtil.getIsUnSupportetValue(unSupportetValues, unSupportetValue)
	for unSupportetValueNumber = 1, #unSupportetValues do
		if unSupportetValues[unSupportetValueNumber] == unSupportetValue then
			return true;
		end;
	end;
	
	return false;
end;

function RealisticSeederUtil.getCanFruitBePlanted(fillType, fruitType)
	local seedsFillType = g_fillTypeManager:getFillTypeByIndex(fillType);
	local seedsFruitType = g_fruitTypeManager:getFruitTypeByName(string.sub(seedsFillType.name, 6));

	if fruitType == nil then
		fruitType = seedsFruitType;
	end;
	
	return fruitType ~= nil and fruitType.terrainDataPlaneId ~= nil;
end;

function RealisticSeederUtil.foundFillTypeNameInTable(fillTypeNameTable, fillTypeName)
	for fillTypeNumber = 1, #fillTypeNameTable do
		if string.find(RealisticSeederUtil.fixFillTypeName(fillTypeNameTable[fillTypeNumber]:lower()), RealisticSeederUtil.fixFillTypeName(fillTypeName)) then
			return true;
		end;
	end;
	
	return false;
end;

function RealisticSeederUtil.getCurrentLanguage()
	local currentLanguage = "english";

	if RealisticSeederUtil.languageShort == "de" then
		currentLanguage = "german";
	elseif RealisticSeederUtil.languageShort == "fr" then
		currentLanguage = "french";
	elseif RealisticSeederUtil.languageShort == "pl" then
		currentLanguage = "polish";
	elseif RealisticSeederUtil.languageShort == "cz" then
		currentLanguage = "czech";
	end;

	return currentLanguage;
end;

function RealisticSeederUtil.getStorePrefix(palletType)
	local storePrefix = palletType;

	if palletType == "bigBags" or palletType == "bigBags_2" then
		storePrefix = "bigBag";
	elseif palletType == "pallets" then
		storePrefix = "pallet";
	elseif palletType == "bags" then
		storePrefix = "bag";
	end;

	return "store_" .. storePrefix .. "_";
end;

function RealisticSeederUtil.getCurrentOverlayName(palletType)
	local currentOverlayName = palletType;

	if currentOverlayName == "bigBags" then
		currentOverlayName = "bigBag";
	elseif palletType == "bigBags_2" then
		currentOverlayName = "bigBag2";
	elseif palletType == "pallets" then
		currentOverlayName = "pallet";
	elseif palletType == "bags" then
		currentOverlayName = "bag";
	end;

	return currentOverlayName;
end;

function RealisticSeederUtil.isValidFillTypeName(seedFillTypeName)
	return string.find(seedFillTypeName, "seed") or string.find(seedFillTypeName, "fertilizer") or string.find(seedFillTypeName, "lime");
end;

function RealisticSeederUtil.getForceBagOverlay(seedFillTypeName, fillLevel, bagUnloadThreshold)
	return seedFillTypeName ~= "seed_potato" and fillLevel > 0 and fillLevel <= bagUnloadThreshold;
end;

function RealisticSeederUtil.getFillTypeNameDoNotSupportThreeUnloadModes(fillTypeName)
	return fillTypeName == "FERTILIZER" or fillTypeName == "LIME" or fillTypeName == "SEEDS";
end;

function RealisticSeederUtil.getNumberOfFillUnits(fillUnits)
	local numberOfFillUnits = 0;
	
	for fillUnitNumber, fillUnit in pairs(fillUnits) do
		if fillUnit.showOnHud then
			numberOfFillUnits = numberOfFillUnits + 1;
		end;
	end;
	
	return numberOfFillUnits;
end;

function RealisticSeederUtil.getForcedFillUnitIndex(fillUnits)
	local numberOfFillUnits, fillUnitIndex = 0, 1;

	for fillUnitNumber, fillUnit in pairs(fillUnits) do
		if fillUnit.showOnHud and fillUnit.fillLevel > 0 then
			numberOfFillUnits = numberOfFillUnits + 1;
		end;

		if numberOfFillUnits == 1 then
			if fillUnit.showOnHud and fillUnit.fillLevel > 0 then
				fillUnitIndex = fillUnitNumber;
			end;
		end;
	end;

	return numberOfFillUnits, fillUnitIndex;
end;

function RealisticSeederUtil.getIsAllowedToUnloadInBigBags(specRealisticSeeder, unloadModePallets, fixedFillTypeName, fillLevel)
	return specRealisticSeeder.currentUnloadMode ~= unloadModePallets and fixedFillTypeName ~= "seed_potato" and fillLevel > specRealisticSeeder.bagUnloadThreshold;
end;

function RealisticSeederUtil.getIsRealisticSeederBigBagFillType(fixedFillTypeName)
	return fixedFillTypeName ~= "fertilizer" and fixedFillTypeName ~= "lime" and fixedFillTypeName ~= "seeds";
end;

function RealisticSeederUtil.getPalletFilename(specRealisticSeeder, xmlPathPreFix, unloadModePallets, unloadModeBigBags2, palletPath, fixedFillTypeName, fillLevel)
	--## realistic seeder seed fruit types pallets
	local palletFilename = xmlPathPreFix .. palletPath .. fixedFillTypeName .. ".xml";

	if fixedFillTypeName == "seeds" and fillLevel > specRealisticSeeder.bagUnloadThreshold then
		--## default seeds pallet for default seeds
		palletFilename = "data/objects/pallets/pioneer/pioneerPallet.xml";
	end;
	
	if RealisticSeederUtil.getIsAllowedToUnloadInBigBags(specRealisticSeeder, unloadModePallets, fixedFillTypeName, fillLevel) then
		if RealisticSeederUtil.getIsRealisticSeederBigBagFillType(fixedFillTypeName) then
			--## realistic seeder seed fruit types big bags
			palletFilename = xmlPathPreFix .. "bigBags/bigBag_" .. fixedFillTypeName .. ".xml";

			if specRealisticSeeder.currentUnloadMode == unloadModeBigBags2 then
				palletFilename = xmlPathPreFix .. "bigBags_2/bigBag_" .. fixedFillTypeName .. ".xml";
			end;
		else
			if specRealisticSeeder.bigBagModDir ~= "" then
				--## big bag from the big bag pack or the hof bergmann map
				palletFilename = specRealisticSeeder.bigBagModDir .. fixedFillTypeName .. "/bigBag_" .. fixedFillTypeName .. ".xml";
			else
				--## default big bag, when the lsfm big bag pack or the hof bergmann map is not loaded
				palletFilename = "data/objects/bigBag/" .. fixedFillTypeName .. "/bigBag_" .. fixedFillTypeName .. ".xml";
			end;
		end;
	end;

	return palletFilename;
end;

function RealisticSeederUtil.getIsUnloadActionEventAllowed(seedName)
	return seedName:find("SEED") or seedName == "FERTILIZER" or seedName == "LIME";
end;

function RealisticSeederUtil.loadSupportedModCategories(categoryToAdd, addCategory)
	if addCategory then
		local isValidStoreCategory = RealisticSeederUtil.getIsValidStoreCategory(categoryToAdd);

		if not isValidStoreCategory then
			table.insert(RealisticSeederUtil.validStoreCategories, categoryToAdd);
		end;
	end;
end;