--[[
RealisticSeeder

Specialization for an realistic seeder

Author: 	Ifko[nator]
Date: 		23.11.2022
Version:	4.1

History:	v1.0 @04.07.2020 - initial implementation in FS 19
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.1 @30.07.2020 - changed the loading logic of the seed fillTypes
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.2 @16.08.2020 - fix for sugar cane planters and for additional tank fill volumes
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.3 @27.08.2020 - reworkt function to load fill types, so it is not mess up with other mods
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.4 @12.09.2020 - added support for trailers and shovels they use the fill type category 'bulk' or 'augerwagon'
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.5 @16.02.2021 - added possibility to unload seed in pallets from a trailer/shovel
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.6 @12.06.2021 - reworkt function to unload seed in pallets from a trailer/shovel
				             - added possibility to unload seed and fertilizer seperatly
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.7 @13.06.2021 - minior adjustments
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.8 @14.06.2021 - Fix following Error when try to join an multiplayer game:

			Error: Event initialization only allowed at compile time
			LUA call stack:
			  dataS/scripts/network/EventIds.lua (28) : printCallstack
			  =E:/Eigene Dokumente/My Games/FarmingSimulator2019/mods/FS19_realisticSeeder/specializations/RealisticSeeder.lua (1048) : InitStaticEventClass
			  =[C] (4294967295)
			  dataS/scripts/vehicles/SpecializationManager.lua (85) : source
			  dataS/scripts/main.lua (2631) : addSpecialization
			  =E:/Eigene Dokumente/My Games/FarmingSimulator2019/mods/FS19_realisticSeeder/specializations/RegisterSpecialization.lua (80) : addSpecialization
			  dataS/scripts/utils/Utils.lua (404) : newFunc
			  dataS/scripts/main.lua (2980) : loadVehicleTypeFromXML
			  dataS/scripts/gui/MPLoadingScreen.lua (677) : loadMod
			  dataS/scripts/misc/AsyncManager.lua (42) : lambda
			  dataS/scripts/misc/AsyncManager.lua (119) : runLambda
			  dataS/scripts/misc/DeferredLoadingManager.lua (39) : runTopTask
			  dataS/scripts/main.lua (1904) : update
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v1.9 @28.06.2021 - it's possible again to unload herbicide
							 - added possibility to switch between pallets and big bags for unload seed, fertilizer and lime
							 - on the hof bergmann map the default liquid fertilizer and herbicide tanks will replaced with the ibc tanks from the map for unloading
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v2.0 @09.07.2021 - fixed Multiplayer Errors
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v2.1 @13.07.2021 - added function to create bags, if fill level is 100 liter or smaller
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v2.2 @24.07.2021 - it's possible again to unlaod to discharge triggers
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v2.3 @31.07.2021 - fix for the pronto 6 AS tank by Agrartechnik Nordeifel
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v2.4 @09.08.2021 - bug fix for the additional tanks: hatzenbichler th 1400, seedhawk 980 air cart and from the bourgault dlc
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.0 @19.11.2021 - convert to FS 22
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.1 @21.05.2022 - added compatibilty with the LSFM BigBag Pack and the Hof Bergmann
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.2 @06.06.2022 - added possibility to switch between the two big bags for unloading
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.3 @09.06.2022 - added "assignItemAttributeData" overwrite
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.4 @19.06.2022 - added function to render store image for the current unload mode to show witch is active
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.5 @22.06.2022 - added function to switch fill units for unloading
							   added save function for "currentUnloadMode" and "currentFillUnitNumber"
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.6 @18.08.2022 - added timer function, to show the icons for the current unload mode only an certian time
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.7 @17.09.2022 - fixed some Multiplayer issues
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.8 @03.10.2022 - with the universal tank pack the default liquid fertilizer and herbicide tanks will replaced with the ibc tank from the mod for unloading
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v3.9 @08.10.2022 - it is no longer possible to unload non seed fill types in pallets from an trailer. Exception: lime, fertilizer, herbicide and liquid fertilizer
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v4.0 @18.11.2022 - added possibility to turn off/on the fertilizing function from sowingmachines
			------------------------------------------------------------------------------------------------------------------------------------------------------------------
			v4.1 @23.11.2022 - added possibility to switch the unload side between left and right for the pallets, big bags and bags
]]

RealisticSeeder = {};

RealisticSeeder.XML_PATH_PREFIX = RealisticSeederUtil.currentModDirectory .. "xml/";

RealisticSeeder.UNLOAD_MODE_BIG_BAG = 1;
RealisticSeeder.UNLOAD_MODE_PALLETS = 2;
RealisticSeeder.UNLOAD_MODE_BIG_BAG_2 = 3;
RealisticSeeder.UNLOAD_MODE_MAX = 3;

RealisticSeeder.PRO_SEED_MOD_NAME = "";

RealisticSeeder.BASE_FILL_UNITS_KEY = "vehicle.fillUnit.fillUnitConfigurations.fillUnitConfiguration(%d).fillUnits";
RealisticSeeder.FILL_UNIT_KEY = "%s.fillUnit(%d)";

for _, mod in pairs(g_modManager.mods) do
	if _G[tostring(mod.modName)].ProSeed ~= nil then
		if g_modIsLoaded[tostring(mod.modName)] then	
			RealisticSeeder.PRO_SEED_MOD_NAME = mod.modName;

			break;
		end;
	end;
end;

RealisticSeeder.SAVE_PARAMETERS = {
	"currentUnloadMode",
	"currentFillUnitNumber",
	"isUnloadingToLeft",
	"allowFertilizer"
};

RealisticSeeder.SAVEGAME_PARAMETERS = {};

RealisticSeeder.SAVEGAME_PARAMETERS.currentUnloadMode = {
	description = "Current selected unload Mode.",
	valueType = "INT",
	defaultValue = 1,
	allowRegisterSave = true
};

RealisticSeeder.SAVEGAME_PARAMETERS.currentFillUnitNumber = {
	description = "Current selected fill unit.",
	valueType = "INT",
	defaultValue = 1,
	allowRegisterSave = true
};

RealisticSeeder.SAVEGAME_PARAMETERS.isUnloadingToLeft = {
	description = "Current selected unload side.",
	valueType = "BOOL",
	defaultValue = true,
	allowRegisterSave = true
};

RealisticSeeder.SAVEGAME_PARAMETERS.allowFertilizer = {
	description = "Allow/Disallow fertilizing.",
	valueType = "BOOL";
	defaultValue = true,
	allowRegisterSave = RealisticSeeder.PRO_SEED_MOD_NAME == ""
};

function RealisticSeeder.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame;

	for _, saveParameter in pairs(RealisticSeeder.SAVE_PARAMETERS) do
		if RealisticSeeder.SAVEGAME_PARAMETERS[saveParameter].allowRegisterSave then
			schemaSavegame:register(
				XMLValueType[RealisticSeeder.SAVEGAME_PARAMETERS[saveParameter].valueType],
				"vehicles.vehicle(?).realisticSeeder#" .. saveParameter,
				RealisticSeeder.SAVEGAME_PARAMETERS[saveParameter].description,
				RealisticSeeder.SAVEGAME_PARAMETERS[saveParameter].defaultValue
			);
		end;
	end;
end;

function RealisticSeeder.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(FillUnit, specializations);
end;

function RealisticSeeder.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onPostLoad",
		"onUpdate",
		"onUpdateTick",
		"onDraw",
		"onReadStream",
		"onWriteStream",
		"onReadUpdateStream",
		"onWriteUpdateStream",
		"onRegisterActionEvents"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, RealisticSeeder);
	end;
end;

function RealisticSeeder.registerOverwrittenFunctions(vehicleType)
    if RealisticSeeder.PRO_SEED_MOD_NAME == "" then
		SpecializationUtil.registerOverwrittenFunction(vehicleType, "processSowingMachineArea", RealisticSeeder.processSowingMachineArea);
	end;
end;

function RealisticSeeder.registerFunctions(vehicleType)
	local newFunctions = {
		"loadSeedFillTypes",
		"getAttachedAdditionalTanks",
		"addFillTypesToBulkAndAugerwagon",
		"unloadFillUnitsToPallets",
		"switchUnloadMode",
		"switchFillUnit",
		"switchFertilizerMode",
		"switchUnloadSide",
		"loadStoreImages"
	};
	
	for _, newFunction in ipairs(newFunctions) do
		SpecializationUtil.registerFunction(vehicleType, newFunction, RealisticSeeder[newFunction]);
	end;
end;

function RealisticSeeder:onLoad(savegame)
	if RealisticSeederUtil == nil then
		return;
	end;
	
	RealisticSeeder.debugPriority = RealisticSeederUtil.getDebugPriority(RealisticSeederUtil.modDesc, "modDesc.realisticSeeder#debugPriority");
	
	if fileExists(RealisticSeederUtil.fillTypesPath) then
		self:addFillTypesToBulkAndAugerwagon(self.xmlFile.handle, getXMLString(self.xmlFile.handle, "vehicle.storeData.category"), Utils.getNoNil(getXMLString(self.xmlFile.handle, "vehicle.base.size#width"), 3), RealisticSeederUtil.getSeedFillTypes());
	else
		RealisticSeederUtil.printError("Can't load fillTypesXML file (path: " .. RealisticSeederUtil.fillTypesPath .. ")! Aborting adding fill types to the fill type categories 'BULK' and 'AUGERWAGON'!", false, false, "RealisticSeeder");
	end;
	
	local _, _, bigBagModDir = RealisticSeederUtil.getModByTitle("LSFM BigBag Pack");
	local _, _, hofBergmannModDir = RealisticSeederUtil.getModByTitle("Hof Bergmann");
	
	local bigBagStoreDir = bigBagModDir;
	
	if bigBagModDir ~= "" then
		--## set filename to the big bags from the LSFM big bag pack
		bigBagModDir = bigBagModDir .. "objects/bigBag/";
	end;
	
	if hofBergmannModDir ~= "" then
		--## set filename to the big bags from the hof bergmann map
		bigBagModDir = hofBergmannModDir .. "objects/bigBag/";
		bigBagStoreDir = hofBergmannModDir;
	end;
	
	for _, nonSeedFillType in pairs({"FERTILIZER", "HERBICIDE"}) do
		local palletFilename = RealisticSeeder.XML_PATH_PREFIX .. "ibcTank_fillable.xml";
			
		if nonSeedFillType == "FERTILIZER" then
			nonSeedFillType = "LIQUID" .. nonSeedFillType;
		end;
			
		local fillType = g_fillTypeManager:getFillTypeByName(nonSeedFillType);
			
		if fillType.palletFilename ~= nil 
			and fileExists(palletFilename) 
			and fillType.palletFilename ~= palletFilename 
		then
			--## switch default tanks for liquid fertilizer and herbicide with the ibc tank from Farmer_Andy
			fillType.palletFilename = palletFilename;
		end;
	end;

	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	
	specRealisticSeeder.bigBagModDir = bigBagModDir;
	specRealisticSeeder.bagUnloadThreshold = Utils.getNoNil(getXMLInt(RealisticSeederUtil.modDesc, "modDesc.realisticSeeder#bagUnloadThreshold"), 500);

	self:loadStoreImages(specRealisticSeeder, bigBagStoreDir, RealisticSeederUtil.getSeedFillTypes(), g_gameSettings:getValue("uiScale"));
	
	if not RealisticSeederUtil.getIsValidStoreCategory(getXMLString(self.xmlFile.handle, "vehicle.storeData.category")) then
		return;
	end;
	
	if RealisticSeederUtil.fillTypesPath ~= nil and fileExists(RealisticSeederUtil.fillTypesPath) then
		self:loadSeedFillTypes(self.xmlFile.handle, getXMLString(self.xmlFile.handle, "vehicle.storeData.category"), Utils.getNoNil(getXMLString(self.xmlFile.handle, "vehicle.base.size#width"), 3), RealisticSeederUtil.getSeedFillTypes());
	else
		RealisticSeederUtil.printError("Can't load fillTypesXML file (path: " .. RealisticSeederUtil.fillTypesPath .. ")! Aborting loading fill types for sowing machines!", false, false, "RealisticSeeder");
	end;
end;

function RealisticSeeder:loadStoreImages(specRealisticSeeder, bigBagStoreDir, seedFillTypes, uiScale)
	RealisticSeeder.OVERLAY_POSITIONS = {
		g_currentMission.inGameMenu.hud.speedMeter.gaugeCenterX + g_currentMission.inGameMenu.hud.speedMeter.speedIndicatorRadiusY * 0.05,
		g_currentMission.inGameMenu.hud.speedMeter.gaugeCenterY + g_currentMission.inGameMenu.hud.speedMeter.speedIndicatorRadiusY * 2.85
	};

	RealisticSeeder.OVERLAY_SIZES = {
		0.12 * uiScale,
		0.14 * uiScale
	};
	
	specRealisticSeeder.storeImageTimer = 0;
	specRealisticSeeder.storeImageTimerMax = 1000;
	specRealisticSeeder.storeImages = {};

	for _, nonSeedFillType in pairs({"FERTILIZER", "LIME"}) do
		table.insert(seedFillTypes, FillType[nonSeedFillType]);
	end;

	for _, seedFillType in pairs(seedFillTypes) do
		local seedFillTypeName = RealisticSeederUtil.fixFillTypeName(g_fillTypeManager:getFillTypeByIndex(seedFillType).name:lower());

		if RealisticSeederUtil.getCanFruitBePlanted(seedFillType, nil) or seedFillTypeName == "fertilizer" or seedFillTypeName == "lime" then
			local currentLanguage = RealisticSeederUtil.getCurrentLanguage();
			local storeImage = {};
			
			storeImage.overlay = {};

			for _, palletType in pairs(RealisticSeederUtil.palletTypes) do
				local storePrefix = RealisticSeederUtil.getStorePrefix(palletType);
				local currentOverlayName = RealisticSeederUtil.getCurrentOverlayName(palletType);

				local storeImagePath = Utils.getFilename("store/" .. currentLanguage .. "/" .. palletType .. "/" .. storePrefix .. seedFillTypeName .. ".dds", RealisticSeederUtil.currentModDirectory);

				storeImage.overlay[palletType] = {};
				storeImage.seedFillTypeName = seedFillTypeName;

				storeImage[palletType] = "noImage.png";
				
				if fileExists(storeImagePath) then
					storeImage[palletType] = storeImagePath;
				else
					if seedFillTypeName == "fertilizer" or seedFillTypeName == "lime" then
						if bigBagStoreDir ~= "" then
							storeImage[palletType] = Utils.getFilename("store/store_bigBag_" .. seedFillTypeName .. ".dds", bigBagStoreDir);
						else
							storeImage[palletType] = "data/objects/bigBag/" .. seedFillTypeName .. "/store_bigBag_" .. seedFillTypeName .. ".dds";
						end;
					end;
				end;

				if fileExists(storeImage[palletType]) then
					storeImage.overlay[palletType][currentOverlayName] = Overlay.new(
						storeImage[palletType],
						RealisticSeeder.OVERLAY_POSITIONS[1],
						RealisticSeeder.OVERLAY_POSITIONS[2],
						RealisticSeeder.OVERLAY_SIZES[1],
						RealisticSeeder.OVERLAY_SIZES[2]
					);
				end;

				table.insert(specRealisticSeeder.storeImages, storeImage);
			end;
		end;
	end;
end;

function RealisticSeeder:onPostLoad(savegame)
	if RealisticSeederUtil == nil then
		return;
	end;
	
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	

	specRealisticSeeder.currentUnloadMode = RealisticSeeder.UNLOAD_MODE_BIG_BAG;
	specRealisticSeeder.currentUnloadModeOld = RealisticSeeder.UNLOAD_MODE_BIG_BAG;
	specRealisticSeeder.currentUnloadModeSync = RealisticSeeder.UNLOAD_MODE_BIG_BAG;
	specRealisticSeeder.currentFillUnitNumber = 1;
	specRealisticSeeder.currentFillUnitNumberSync = 1;

	specRealisticSeeder.isUnloadingToLeft = true;

	if savegame ~= nil then
		specRealisticSeeder.currentUnloadMode = savegame.xmlFile:getValue(savegame.key .. ".realisticSeeder#currentUnloadMode", specRealisticSeeder.currentUnloadMode);
		specRealisticSeeder.currentFillUnitNumber = savegame.xmlFile:getValue(savegame.key .. ".realisticSeeder#currentFillUnitNumber", specRealisticSeeder.currentFillUnitNumber);
		specRealisticSeeder.isUnloadingToLeft = savegame.xmlFile:getValue(savegame.key .. ".realisticSeeder#isUnloadingToLeft", specRealisticSeeder.isUnloadingToLeft);
	end;
	
	self:switchUnloadMode(specRealisticSeeder.currentUnloadMode, false);
	self:switchFillUnit(specRealisticSeeder.currentFillUnitNumber, false);
	self:switchUnloadSide(specRealisticSeeder.isUnloadingToLeft, false);
	
	if not RealisticSeederUtil.getIsValidStoreCategory(getXMLString(self.xmlFile.handle, "vehicle.storeData.category")) then
		return;
	end;
	
	local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");
	local specFillVolume = RealisticSeederUtil.getSpecByName(self, "fillVolume");
	local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
	local specSprayer = RealisticSeederUtil.getSpecByName(self, "sprayer");

	if RealisticSeeder.PRO_SEED_MOD_NAME == "" and specSowingMachine ~= nil and specSprayer ~= nil then
		specRealisticSeeder.allowFertilizer = true;

		if savegame ~= nil then
			specRealisticSeeder.allowFertilizer = savegame.xmlFile:getValue(savegame.key .. ".realisticSeeder#allowFertilizer", specRealisticSeeder.allowFertilizer);
		end;

		self:switchFertilizerMode(specRealisticSeeder.allowFertilizer, false);
	end;
	
	specRealisticSeeder.oldSeedFruitType = FillType.UNKNOWN;
	specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
	specRealisticSeeder.additionalTankFillLevel = 0;
	specRealisticSeeder.additionalTank = nil;
	
	if specSowingMachine ~= nil and g_fillTypeManager:getFillTypeByIndex(specFillUnit.fillUnits[specSowingMachine.fillUnitIndex].fillType).name == "SEED_POTATO" then
		g_animationManager:setFillType(specSowingMachine.animationNodes, FillType.SEEDS);
	end;
end;

function RealisticSeeder:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
	if RealisticSeederUtil == nil then
		return;
	end;
	
	if self.isClient then
		local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
		local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
		local specSprayer = RealisticSeederUtil.getSpecByName(self, "sprayer");
		
		self:clearActionEventsTable(specRealisticSeeder.actionEvents);
		
		if isActiveForInput then
			for _, input in pairs(RealisticSeederUtil.inputs) do
				local allowAddButton = true;
				
				if input == "SELECT_FERTILIZER_MODE" then
					allowAddButton = specSowingMachine ~= nil and specSprayer ~= nil and RealisticSeeder.PRO_SEED_MOD_NAME == "";
				end;

				if allowAddButton then
					local _, actionEventId = self:addActionEvent(
						specRealisticSeeder.actionEvents, --## actionEventsTable
						InputAction[input], --## inputAction
						self, --## target
						RealisticSeeder.actionEventCallback, --## callback function
						false, --## triggerUp
						true, --## triggerDown
						false, --## triggerAlways
						true, --## startActive
						nil --## callbackState
					);

					g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH);
					g_inputBinding:setActionEventTextVisibility(actionEventId, true);
					g_inputBinding:setActionEventActive(actionEventId, true);
					g_inputBinding:setActionEventText(actionEventId, RealisticSeederUtil.l10nTexts["input_" .. input]);
				end;
			end;
		end;
	end;
end;

function RealisticSeeder:actionEventCallback(actionName, inputValue, callbackState, isAnalog)
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	
	if actionName == "SELECT_UNLOAD_MODE" then
		if specRealisticSeeder.currentUnloadMode < RealisticSeeder.UNLOAD_MODE_MAX then
			specRealisticSeeder.currentUnloadModeSync = specRealisticSeeder.currentUnloadModeSync + 1;
		else
			specRealisticSeeder.currentUnloadModeSync = RealisticSeeder.UNLOAD_MODE_BIG_BAG;
		end;
		
		self:switchUnloadMode(specRealisticSeeder.currentUnloadModeSync, false);
	elseif actionName == "SELECT_FILL_UNIT" then
		local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");

		if specFillUnit ~= nil then
			local numberOfFillUnits = RealisticSeederUtil.getNumberOfFillUnits(specFillUnit.fillUnits);

			if numberOfFillUnits > 1 then
				if specRealisticSeeder.currentFillUnitNumber < numberOfFillUnits then
					specRealisticSeeder.currentFillUnitNumberSync = specRealisticSeeder.currentFillUnitNumberSync + 1;
				else
					specRealisticSeeder.currentFillUnitNumberSync = 1;
				end;
			else
				specRealisticSeeder.currentFillUnitNumberSync = 1;
			end;

			self:switchFillUnit(specRealisticSeeder.currentFillUnitNumberSync, false);
		end;
	elseif actionName == "SELECT_FERTILIZER_MODE" then
		local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
		local specSprayer = RealisticSeederUtil.getSpecByName(self, "sprayer");

		if specSowingMachine ~= nil and specSprayer ~= nil then
			self:switchFertilizerMode(not specRealisticSeeder.allowFertilizer, false);
		end;
	elseif actionName == "SELECT_UNLOAD_SIDE" then
		self:switchUnloadSide(not specRealisticSeeder.isUnloadingToLeft, false);
	else
		self:unloadFillUnitsToPallets(true);
	end;
end;

function RealisticSeeder:switchUnloadMode(currentUnloadMode, noEventSend)
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	
	if currentUnloadMode ~= specRealisticSeeder.currentUnloadMode then
		specRealisticSeeder.currentUnloadMode = currentUnloadMode;
		
		RealisticSeederSwitchUnloadModeEvent.sendEvent(self, currentUnloadMode, noEventSend);
	end;
end;

function RealisticSeeder:switchFillUnit(currentFillUnitNumber, noEventSend)
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	
	if currentFillUnitNumber ~= specRealisticSeeder.currentFillUnitNumber then
		specRealisticSeeder.currentFillUnitNumber = currentFillUnitNumber;

		RealisticSeederSwitchFillUnitNumberEvent.sendEvent(self, currentFillUnitNumber, noEventSend);
	end;
end;

function RealisticSeeder:switchFertilizerMode(allowFertilizer, noEventSend)
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	
	if allowFertilizer ~= specRealisticSeeder.allowFertilizer then
		specRealisticSeeder.allowFertilizer = allowFertilizer;

		RealisticSeederSwitchFertilizerModeEvent.sendEvent(self, allowFertilizer, noEventSend);
	end;
end;

function RealisticSeeder:switchUnloadSide(isUnloadingToLeft, noEventSend)
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	
	if isUnloadingToLeft ~= specRealisticSeeder.isUnloadingToLeft then
		specRealisticSeeder.isUnloadingToLeft = isUnloadingToLeft;

		RealisticSeederSwitchUnloadSideEvent.sendEvent(self, isUnloadingToLeft, noEventSend);
	end;
end;

function RealisticSeeder:addFillTypesToBulkAndAugerwagon(xmlFile, storeCategory, width, seedFillTypes)
	if xmlFile == nil then
		RealisticSeederUtil.printError("Missing xmlFile file! Aborting adding fill types for " .. self:getFullName() .. "!", false, false, "RealisticSeeder");
		
		return;
	elseif storeCategory == nil then
		RealisticSeederUtil.printError("Missing storeCategory! Aborting adding types for " .. self:getFullName() .. "!", false, false, "RealisticSeeder");
		
		return;
	elseif RealisticSeederUtil.foundFunctionAdditionalTank(xmlFile) then
		--## stop for additional tanks!
		
		return;
	end;
	
	local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");
	local baseKey = RealisticSeeder.BASE_FILL_UNITS_KEY:format(Utils.getNoNil(self.configurations["fillUnit"], 1) - 1);
	
	local fillUnitNumber = 0;
	
    while true do
		local fillUnitKey = RealisticSeeder.FILL_UNIT_KEY:format(baseKey, fillUnitNumber);
		
        if not hasXMLProperty(xmlFile, fillUnitKey) then
            break;
		end;
		
		if not RealisticSeederUtil.getIsValidStoreCategory(storeCategory) and 
		(
			specFillUnit.fillUnits[fillUnitNumber + 1].supportedFillTypes[FillType.SEEDS] 
			or specFillUnit.fillUnits[fillUnitNumber + 1].supportedFillTypes[FillType.FERTILIZER] 
			or specFillUnit.fillUnits[fillUnitNumber + 1].supportedFillTypes[FillType.LIQUID_FERTILIZER]
		)
		then
			--## create table for unloading
			specFillUnit.unloading = {};

			local offset = getXMLString(xmlFile, fillUnitKey .. "#offset") or "0 0 0";
			
			--## set new values for table!
			unloading = {};
			unloading.node = I3DUtil.indexToObject(self.components, getXMLString(xmlFile, fillUnitKey .. "#node")) or self.rootNode;
			unloading.width = width + 18;
			unloading.offset = offset:getVectorN(3);
			
			--## insert new values in table!
			table.insert(specFillUnit.unloading, unloading);
		end;
		
		if seedFillTypes ~= nil then
			for _, fillType in pairs(seedFillTypes) do
				if RealisticSeederUtil.getCanFruitBePlanted(fillType, nil) then
					for _, category in pairs(RealisticSeederUtil.fillTypeCategoriesForSeeds) do
						if not g_fillTypeManager:getIsFillTypeInCategory(fillType, category) then
							g_fillTypeManager:addFillTypeToCategory(fillType, g_fillTypeManager.nameToCategoryIndex[category]);
							
							RealisticSeederUtil.printDebug("Insert fill type " .. g_fillTypeManager:getFillTypeByIndex(fillType).name .. " to category " .. category, RealisticSeeder.debugPriority, true, "RealisticSeeder");
						end;
					end;
				end;
			end;
		end;
		
		fillUnitNumber = fillUnitNumber + 1;
	end;
end;

function RealisticSeeder:loadSeedFillTypes(xmlFile, storeCategory, width, seedFillTypes)
	if xmlFile == nil then
		RealisticSeederUtil.printError("Missing xmlFile file! Aborting loading seed fill types for " .. self:getFullName() .. "!", false, false, "RealisticSeeder");
		
		return;
	elseif storeCategory == nil then
		RealisticSeederUtil.printError("Missing storeCategory! Aborting loading seed fill types for " .. self:getFullName() .. "!", false, false, "RealisticSeeder");
		
		return;
	end;
	
	local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
	local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");
	
	RealisticSeederUtil.printDebug("Loading seed fill types for " .. self:getFullName() .. "!", RealisticSeeder.debugPriority, true, "RealisticSeeder");
	
	local seedFillUnitIndex = RealisticSeederUtil.getSeedFillUnitIndex(xmlFile, self, specFillUnit);
	local baseKey = RealisticSeeder.BASE_FILL_UNITS_KEY:format(Utils.getNoNil(self.configurations["fillUnit"], 1) - 1);

	local fillUnitNumber = 0;
	
    while true do
		local fillUnitKey = RealisticSeeder.FILL_UNIT_KEY:format(baseKey, fillUnitNumber);
		
        if not hasXMLProperty(xmlFile, fillUnitKey) then
			break;
		end;
		
		local defaultFillType = Utils.getNoNil(getXMLString(xmlFile, fillUnitKey .. "#fillTypes"), "");

		defaultFillType = defaultFillType:lower();
		
		--## fix unloading width for all sowing machines! Thanks to GIANTS for this issue!
		if defaultFillType:find("seeds") and RealisticSeederUtil.getIsValidStoreCategory(storeCategory) then
			--## clear GIANTS shit out of table or create it, if it is missing! e.g.: potato planters!
			specFillUnit.unloading = {};

			local offset = getXMLString(xmlFile, fillUnitKey .. "#offset") or "0 0 0";
			
			--## set new values for table!
			unloading = {};
			unloading.node = I3DUtil.indexToObject(self.components, getXMLString(xmlFile, fillUnitKey .. "#node")) or self.rootNode;
			unloading.width = width + 18;
			unloading.offset = offset:getVectorN(3);
			
			--## insert new values in table!
			table.insert(specFillUnit.unloading, unloading);
		end;
		
		if seedFillTypes ~= nil then
			for _, fillType in pairs(seedFillTypes) do
				--## check if the fruit type is avaiable on the current map
				if RealisticSeederUtil.getCanFruitBePlanted(fillType, nil) then
					local seedsFillType = g_fillTypeManager:getFillTypeByIndex(fillType);
					
					if specSowingMachine ~= nil and specSowingMachine.seeds ~= nil and defaultFillType:find("seeds") then	
						for _, fruitType in pairs(specSowingMachine.seeds) do
							local seedsFruitType = g_fillTypeManager:getFillTypeByIndex(g_fruitTypeManager:getFillTypeIndexByFruitTypeIndex(fruitType));
							
							if seedsFruitType ~= nil then
								--## check if the fruit type is avaiable
								if seedsFillType.name:sub(6) == seedsFruitType.name and not specFillUnit.fillUnits[fillUnitNumber + 1].supportedFillTypes[fillType] then
									specFillUnit.fillUnits[fillUnitNumber + 1].supportedFillTypes[fillType] = true;
									
									RealisticSeederUtil.printDebug("Insert fill type " .. seedsFillType.name .. " to " .. self:getFullName(), RealisticSeeder.debugPriority, true, "RealisticSeeder");
								end;
							end;
						end;
					else
						if RealisticSeederUtil.foundFunctionAdditionalTank(xmlFile) then
							local seedsFruitType = g_fruitTypeManager:getFruitTypeByName(seedsFillType.name:sub(6));
							
							if seedsFruitType ~= nil then
								--## difference between the store categories
								if RealisticSeederUtil.getIsValidStoreCategory(storeCategory) then
									--## check if fertilizer and seeds is avaiable in the additional tank, if not, add it!
									for _, fillTypeToCheck in pairs({"FERTILIZER", "SEEDS"}) do
										if not specFillUnit.fillUnits[seedFillUnitIndex].supportedFillTypes[FillType[fillTypeToCheck]] then
											specFillUnit.fillUnits[seedFillUnitIndex].supportedFillTypes[FillType[fillTypeToCheck]] = true;
											
											RealisticSeederUtil.printDebug("Insert fill type " .. g_fillTypeManager:getFillTypeByIndex(FillType[fillTypeToCheck]).name .. " to " .. self:getFullName(), RealisticSeeder.debugPriority, true, "RealisticSeeder");
										end;
									end;
									
									for _, category in pairs({"SOWINGMACHINE", "PLANTER"}) do
										local categoryIndex = g_fruitTypeManager.categories[category];
										local categoryFruitTypes = g_fruitTypeManager.categoryToFruitTypes[categoryIndex];
										
        								if categoryFruitTypes ~= nil then
        								    for _, fruitType in ipairs(categoryFruitTypes) do
												if seedsFillType.name:sub(6) == g_fillTypeManager:getFillTypeByIndex(g_fruitTypeManager:getFillTypeIndexByFruitTypeIndex(fruitType)).name and not specFillUnit.fillUnits[seedFillUnitIndex].supportedFillTypes[fillType] then
													specFillUnit.fillUnits[seedFillUnitIndex].supportedFillTypes[fillType] = true;
													
													RealisticSeederUtil.printDebug("Insert fill type " .. seedsFillType.name .. " to " .. self:getFullName(), RealisticSeeder.debugPriority, true, "RealisticSeeder");
												end;
        								    end;
										end;
									end;
								end;
							end;
						end;
					end;
				end;
			end;
		end;
		
		fillUnitNumber = fillUnitNumber + 1;
	end;
end;

function RealisticSeeder:getAttachedAdditionalTanks(rootVehicle)
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	
	if rootVehicle ~= nil then
		local specAttacherJoints = RealisticSeederUtil.getSpecByName(rootVehicle, "attacherJoints");
		
		if specAttacherJoints ~= nil then
			for _, implement in pairs(specAttacherJoints.attachedImplements) do
				local additionalTank = implement.object;
				local additionalTank_specAttachable = RealisticSeederUtil.getSpecByName(additionalTank, "attachable");
				
				if additionalTank_specAttachable.attacherVehicle ~= nil then
					local attacherJointVehicleSpec = RealisticSeederUtil.getSpecByName(additionalTank_specAttachable.attacherVehicle, "attacherJoints");
					
					if RealisticSeederUtil.foundFunctionAdditionalTank(additionalTank.xmlFile.handle) then
						local additionalTank_specFillUnit = RealisticSeederUtil.getSpecByName(additionalTank, "fillUnit");
						local additionalTank_specFillVolume = RealisticSeederUtil.getSpecByName(additionalTank, "fillVolume");
						local seedFillUnitIndex = RealisticSeederUtil.getSeedFillUnitIndex(additionalTank.xmlFile.handle, additionalTank, additionalTank_specFillUnit);
						
						if specRealisticSeeder.additionalTankFillType == FillType.UNKNOWN then
							specRealisticSeeder.additionalTankFillType = additionalTank_specFillUnit.fillUnits[seedFillUnitIndex].fillType;
						end;
						
						if g_fillTypeManager:getFillTypeByIndex(additionalTank:getFillUnitFillType(seedFillUnitIndex)).name:find("SEED") then
							specRealisticSeeder.additionalTank = additionalTank;
							specRealisticSeeder.additionalTankFillLevel = additionalTank_specFillUnit.fillUnits[seedFillUnitIndex].fillLevel;
							
							local seedsFillType = g_fillTypeManager:getFillTypeByName(g_fillTypeManager:getFillTypeByIndex(additionalTank:getFillUnitFillType(seedFillUnitIndex)).name:sub(6));
							
							if seedsFillType ~= nil then
								additionalTank:setFillUnitFillTypeToDisplay(seedFillUnitIndex, seedsFillType.index, true);
							end;
							
							if specRealisticSeeder.additionalTankFillLevel <= 0 then
								specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
								specRealisticSeeder.additionalTank = nil;
							end;
							
							break;
						else
							specRealisticSeeder.additionalTank = nil;
							specRealisticSeeder.additionalTankFillLevel = 0;
							
							additionalTank:setFillUnitFillTypeToDisplay(seedFillUnitIndex, additionalTank:getFillUnitFillType(seedFillUnitIndex), true);
						end;
					else
						specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
						specRealisticSeeder.additionalTank = nil;
						specRealisticSeeder.additionalTankFillLevel = 0;
					end;
					
					if additionalTank.getAttachedAdditionalTanks ~= nil then
						additionalTank:getAttachedAdditionalTanks(additionalTank);
					end;
				end;
			end;
		end;
	else
		specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
		specRealisticSeeder.additionalTank = nil;
		specRealisticSeeder.additionalTankFillLevel = 0;
	end;
end;

function RealisticSeeder:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	if RealisticSeederUtil == nil then
		return;
	end;
	
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");
	local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
	local specSprayer = RealisticSeederUtil.getSpecByName(self, "sprayer");
	
	local actionEventUnload = specRealisticSeeder.actionEvents[InputAction.UNLOAD_BUTTON];
	local actionEventSelectUnloadMode = specRealisticSeeder.actionEvents[InputAction.SELECT_UNLOAD_MODE];
	local actionEventSelectFillUnit = specRealisticSeeder.actionEvents[InputAction.SELECT_FILL_UNIT];
	local actionEventSelectFertilizerMode = specRealisticSeeder.actionEvents[InputAction.SELECT_FERTILIZER_MODE];
	local actionEventSelectUnloadSide = specRealisticSeeder.actionEvents[InputAction.SELECT_UNLOAD_SIDE];
	
	local selectedUnloadMode = "BigBags";
	local allowSwitchUnloadMode = true;
	
	local seedName = "UNKNOWN";
	local fillLevel = 0;
	
	local fillUnitFillType = g_fillTypeManager:getFillTypeByIndex(self:getFillUnitFillType(specRealisticSeeder.currentFillUnitNumber));
		
	if fillUnitFillType ~= nil then
		seedName = fillUnitFillType.name;
		fillLevel = self:getFillUnitFillLevel(specRealisticSeeder.currentFillUnitNumber);
	end;
	
	if RealisticSeederUtil.getFillTypeNameDoNotSupportThreeUnloadModes(seedName) then
		if specRealisticSeeder.currentUnloadMode == RealisticSeeder.UNLOAD_MODE_BIG_BAG_2 then
			self:switchUnloadMode(RealisticSeeder.UNLOAD_MODE_BIG_BAG, false);
		end;
		
		RealisticSeeder.UNLOAD_MODE_MAX = 2;
	else
		RealisticSeeder.UNLOAD_MODE_MAX = 3;
	end;
	
	if specRealisticSeeder.currentUnloadMode == RealisticSeeder.UNLOAD_MODE_BIG_BAG_2 then
		selectedUnloadMode = "BigBags 2";
	elseif specRealisticSeeder.currentUnloadMode == RealisticSeeder.UNLOAD_MODE_PALLETS then
		selectedUnloadMode = RealisticSeederUtil.l10nTexts.category_pallets;
	end;

	if fillLevel <= specRealisticSeeder.bagUnloadThreshold then
		selectedUnloadMode = RealisticSeederUtil.l10nTexts.action_bags;
		
		allowSwitchUnloadMode = false;
	end;

	if specRealisticSeeder.currentUnloadMode ~= specRealisticSeeder.currentUnloadModeOld then
		if specRealisticSeeder.storeImageTimer < specRealisticSeeder.storeImageTimerMax then
			specRealisticSeeder.storeImageTimer = specRealisticSeeder.storeImageTimer + dt;
		else
			specRealisticSeeder.storeImageTimer = 0;
			specRealisticSeeder.currentUnloadModeOld = specRealisticSeeder.currentUnloadMode;
		end;
	end;
	
	if actionEventSelectUnloadMode ~= nil then
		local getIsUnloadActionEventAllowed = RealisticSeederUtil.getIsUnloadActionEventAllowed(seedName);
		
		if actionEventUnload ~= nil then
			if getIsUnloadActionEventAllowed then
				local seedTitle = g_fillTypeManager:getFillTypeByName(seedName).title;
				
				if seedName == "SEED_POTATO" then
					selectedUnloadMode = RealisticSeederUtil.l10nTexts.category_pallets;
					
					allowSwitchUnloadMode = false;
				end;
				
				g_inputBinding:setActionEventText(actionEventUnload.actionEventId, RealisticSeederUtil.l10nTexts.action_unloadXToX:format(seedTitle, selectedUnloadMode));
			end;
			
			g_inputBinding:setActionEventActive(actionEventUnload.actionEventId, getIsUnloadActionEventAllowed);

			if actionEventSelectUnloadSide ~= nil then
				local currentSide = RealisticSeederUtil.l10nTexts.action_workMode_right;
	
				if specRealisticSeeder.isUnloadingToLeft then
					currentSide = RealisticSeederUtil.l10nTexts.action_workMode_left;
				end;
	
				g_inputBinding:setActionEventActive(actionEventSelectUnloadSide.actionEventId, getIsUnloadActionEventAllowed);
				g_inputBinding:setActionEventText(actionEventSelectUnloadSide.actionEventId, RealisticSeederUtil.l10nTexts.action_switchUnloadSideToX:format(currentSide));
			end;
		end;

		if actionEventSelectFillUnit ~= nil then
			local numberOfFillUnits, fillUnitIndex = RealisticSeederUtil.getForcedFillUnitIndex(specFillUnit.fillUnits);

			if numberOfFillUnits == 1 and specRealisticSeeder.currentFillUnitNumber ~= fillUnitIndex then
				self:switchFillUnit(fillUnitIndex, false);
			end;

			g_inputBinding:setActionEventActive(actionEventSelectFillUnit.actionEventId, numberOfFillUnits > 1);
			g_inputBinding:setActionEventText(actionEventSelectFillUnit.actionEventId, RealisticSeederUtil.l10nTexts.action_selectedFillUnitX:format(RealisticSeederUtil.l10nTexts.input_SELECT_FILL_UNIT, specRealisticSeeder.currentFillUnitNumber));
		end;

		if actionEventSelectFertilizerMode ~= nil and RealisticSeeder.PRO_SEED_MOD_NAME == "" then
			local currentText = RealisticSeederUtil.l10nTexts.action_activateFerilizing;

			if specRealisticSeeder.allowFertilizer then
				currentText = RealisticSeederUtil.l10nTexts.action_deactivateFerilizing;
			end;

			g_inputBinding:setActionEventActive(actionEventSelectFertilizerMode.actionEventId, specSowingMachine ~= nil and specSprayer ~= nil);
			g_inputBinding:setActionEventText(actionEventSelectFertilizerMode.actionEventId, currentText);
		end;
		
		g_inputBinding:setActionEventActive(actionEventSelectUnloadMode.actionEventId, getIsUnloadActionEventAllowed and allowSwitchUnloadMode);
	end;
end;

function RealisticSeeder:onUpdateTick(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	if RealisticSeederUtil == nil or not RealisticSeederUtil.getIsValidStoreCategory(getXMLString(self.xmlFile.handle, "vehicle.storeData.category")) then
		return;
	end;
	
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
	local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");
	
	self:getAttachedAdditionalTanks(self:getRootVehicle());
	
	if specRealisticSeeder.additionalTankFillLevel <= 0 then
		specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
	end;
	
	if specSowingMachine ~= nil then
		local selectedSeedFruitType = specFillUnit.fillUnits[specSowingMachine.fillUnitIndex].fillType;

		if specRealisticSeeder.additionalTankFillType ~= FillType.UNKNOWN then
			selectedSeedFruitType = specRealisticSeeder.additionalTankFillType;
		end;

		if selectedSeedFruitType ~= nil then
			specSowingMachine.allowsSeedChanging = selectedSeedFruitType == FillType.SEEDS; --## with default seeds, you can switch the seed fruit types as in the base game
			
			if selectedSeedFruitType ~= FillType.UNKNOWN and not specSowingMachine.allowsSeedChanging and specRealisticSeeder.oldSeedFruitType ~= selectedSeedFruitType then
				local fillTypeDesc = g_fillTypeManager:getFillTypeByIndex(selectedSeedFruitType);
				
				if fillTypeDesc ~= nil then
					local fruitTypeDesc = g_fruitTypeManager:getFruitTypeByName(fillTypeDesc.name:sub(6));
					
					if fruitTypeDesc ~= nil then
						self:setSeedFruitType(fruitTypeDesc.index, true);

						specRealisticSeeder.oldSeedFruitType = selectedSeedFruitType;
					end;
				end;
			end;

			if g_fillTypeManager:getFillTypeByIndex(selectedSeedFruitType).name == "SEED_POTATO" then
				g_animationManager:setFillType(specSowingMachine.animationNodes, FillType.SEEDS);
			end;
		end;
	end;
end;

function RealisticSeeder:onDraw()
	if RealisticSeederUtil == nil then
		return;
	end;

	if self:getIsActiveForInput() then
		local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");
		local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");

		local currentOverlay = "bigBag";

		if specRealisticSeeder.currentUnloadMode == RealisticSeeder.UNLOAD_MODE_BIG_BAG_2 then
			currentOverlay = "bigBag2";
		elseif specRealisticSeeder.currentUnloadMode == RealisticSeeder.UNLOAD_MODE_PALLETS then
			currentOverlay = "pallet";
		end;

		local fillUnitFillType = g_fillTypeManager:getFillTypeByIndex(self:getFillUnitFillType(specRealisticSeeder.currentFillUnitNumber));
		
		if fillUnitFillType ~= nil then
			local seedFillTypeName = RealisticSeederUtil.fixFillTypeName(fillUnitFillType.name:lower());
			local fillUnitFillLevel = self:getFillUnitFillLevel(specRealisticSeeder.currentFillUnitNumber);

			if RealisticSeederUtil.isValidFillTypeName(seedFillTypeName) then
				if RealisticSeederUtil.getForceBagOverlay(seedFillTypeName, fillUnitFillLevel, specRealisticSeeder.bagUnloadThreshold) then
					currentOverlay = "bag";
				end;

				for _, palletType in pairs(RealisticSeederUtil.palletTypes) do
					for _, storeImage in pairs(specRealisticSeeder.storeImages) do
						if storeImage.seedFillTypeName == seedFillTypeName then
							if storeImage.overlay[palletType][currentOverlay] ~= nil then
								if specRealisticSeeder.currentUnloadMode ~= specRealisticSeeder.currentUnloadModeOld then
									storeImage.overlay[palletType][currentOverlay]:render();
								end;
							end;
						end;
					end;
				end;
			end;
		end;
	end;
end;

function RealisticSeeder:saveToXMLFile(xmlFile, key)
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
	local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
	local specSprayer = RealisticSeederUtil.getSpecByName(self, "sprayer");

	for _, saveParameter in pairs(RealisticSeeder.SAVE_PARAMETERS) do
		local allowSaveParameter = true;

		if saveParameter == "allowFertilizer" then
			allowSaveParameter = RealisticSeeder.PRO_SEED_MOD_NAME == "" and specSowingMachine ~= nil and specSprayer ~= nil;
		end;

		if allowSaveParameter then
			xmlFile:setValue(key .. "#" .. saveParameter, specRealisticSeeder[saveParameter]);
		end;
	end;
end;

function RealisticSeeder:onReadStream(streamId, connection)
	if RealisticSeederUtil == nil then
		return;
	end;
	
	if connection:getIsServer() then
		local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");

		if specRealisticSeeder.additionalTankFillType == nil then	
			specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
		end;

		specRealisticSeeder.additionalTankFillType = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS);
	end;
end;

function RealisticSeeder:onWriteStream(streamId, connection)
	if RealisticSeederUtil == nil then
		return;
	end;
	
	if not connection:getIsServer() then
		local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");

		if specRealisticSeeder.additionalTankFillType == nil then	
			specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
		end;	
			
		streamWriteUIntN(streamId, specRealisticSeeder.additionalTankFillType, FillTypeManager.SEND_NUM_BITS);
	end;
end;

function RealisticSeeder:onReadUpdateStream(streamId, timestamp, connection)
	if RealisticSeederUtil == nil then
		return;
	end;

	if connection:getIsServer() then
		local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");

		if specRealisticSeeder.additionalTankFillType == nil then	
			specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
		end;

		specRealisticSeeder.additionalTankFillType = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS);
	end;
end;

function RealisticSeeder:onWriteUpdateStream(streamId, connection, dirtyMask)
	if RealisticSeederUtil == nil then
		return;
	end;
	
	if not connection:getIsServer() then
		local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");

		if specRealisticSeeder.additionalTankFillType == nil then	
			specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
		end;
		
		streamWriteUIntN(streamId, specRealisticSeeder.additionalTankFillType, FillTypeManager.SEND_NUM_BITS);
	end;
end;

function RealisticSeeder:processSowingMachineArea(superFunc, workArea, dt)
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");

    if not specRealisticSeeder.allowFertilizer then
        local specSprayer = RealisticSeederUtil.getSpecByName(self, "sprayer");

        if specSprayer ~= nil then
            specSprayer.workAreaParameters.sprayFillLevel = 0;
        end;
    end;

    local changedArea, totalArea = superFunc(self, workArea, dt);

    return changedArea, totalArea;
end;

RealisticSeeder.addFillUnitFillLevelBackup = SowingMachine.addFillUnitFillLevel; --## backup function for all sowingmachines they are not supportet by the realistic seeder

function RealisticSeeder:addFillUnitFillLevel(superFunc, farmId, fillUnitIndex, fillLevelDelta, fillType, toolType, fillInfo)
	if not RealisticSeederUtil.getIsValidStoreCategory(getXMLString(self.xmlFile.handle, "vehicle.storeData.category")) then
		return RealisticSeeder.addFillUnitFillLevelBackup(self, superFunc, farmId, fillUnitIndex, fillLevelDelta, fillType, toolType, fillInfo);
	end;
	
	local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
	local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");
	local specFillVolume = RealisticSeederUtil.getSpecByName(self, "fillVolume");

	if self:getFillUnitFillType(fillUnitIndex) ~= FillType.UNKNOWN and self:getFillUnitFillType(fillUnitIndex) ~= fillType then
		if specFillUnit.fillUnits[fillUnitIndex].fillLevel / math.max(specFillUnit.fillUnits[fillUnitIndex].capacity, 0.0001) > self:getFillTypeChangeThreshold() then
			return 0;
		end;
	end;

	if fillUnitIndex == specSowingMachine.fillUnitIndex then
		if self:getFillUnitSupportsFillType(fillUnitIndex, fillType) then
			self:setFillUnitForcedMaterialFillType(fillUnitIndex, fillType);
		end;

		local fruitType = specSowingMachine.seeds[specSowingMachine.currentSeed];

		if fruitType ~= nil then
			local seedsFillType = g_fruitTypeManager:getFillTypeIndexByFruitTypeIndex(fruitType);

			if seedsFillType ~= nil and self:getFillUnitSupportsFillType(fillUnitIndex, seedsFillType) then
				self:setFillUnitForcedMaterialFillType(fillUnitIndex, seedsFillType);
			end;
		end;
	end;

	return superFunc(self, farmId, fillUnitIndex, fillLevelDelta, fillType, toolType, fillInfo);
end;

SowingMachine.addFillUnitFillLevel = RealisticSeeder.addFillUnitFillLevel; --## need an hardcore overwrite... Otherwhise, the fill type will automaticlly set to "seeds"!

function RealisticSeeder:onStartWorkAreaProcessing(superFunc, dt)
	if RealisticSeederUtil == nil then
		return superFunc(dt);
	end;
	
	local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
    local specSowingMachine = RealisticSeederUtil.getSpecByName(self, "sowingMachine");
	
	specSowingMachine.isWorking = false;
    specSowingMachine.isProcessing = false;
	
	local seedsFruitType = specSowingMachine.seeds[specSowingMachine.currentSeed];
    local dx, _, dz = localDirectionToWorld(specSowingMachine.directionNode, 0, 0, 1);
    local angleRad = MathUtil.getYRotationFromDirection(dx, dz);
    local fruitType = g_fruitTypeManager:getFruitTypeByIndex(seedsFruitType);
	
	if fruitType ~= nil and fruitType.directionSnapAngle ~= 0 then
		angleRad = math.floor(angleRad / fruitType.directionSnapAngle + 0.5) * fruitType.directionSnapAngle;
    end;
	
	local angle = FSDensityMapUtil.convertToDensityMapAngle(angleRad, g_currentMission.fieldGroundSystem:getGroundAngleMaxValue());
	local seedsVehicle, seedsVehicleFillUnitIndex;

	if specRealisticSeeder.additionalTankFillType == nil then	
		specRealisticSeeder.additionalTankFillType = FillType.UNKNOWN;
	end;
	
	if specRealisticSeeder.additionalTankFillType == FillType.UNKNOWN then
		if self:getFillUnitFillLevel(specSowingMachine.fillUnitIndex) > 0 then
    	    seedsVehicle = self;
			seedsVehicleFillUnitIndex = specSowingMachine.fillUnitIndex;
		end;
	else
		if specRealisticSeeder.additionalTank ~= nil then
			local fillType = specRealisticSeeder.additionalTankFillType;
			local seedFillUnitIndex = 1;
			
			if specRealisticSeeder.additionalTank.spec_fillUnit ~= nil then
				seedFillUnitIndex = RealisticSeederUtil.getSeedFillUnitIndex(specRealisticSeeder.additionalTank.xmlFile.handle, specRealisticSeeder.additionalTank, specRealisticSeeder.additionalTank.spec_fillUnit);
			end;

			if specRealisticSeeder.additionalTank:getFillUnitFillLevel(seedFillUnitIndex) > 0 and specRealisticSeeder.additionalTank:getFillUnitFillType(seedFillUnitIndex) == fillType then
        		seedsVehicle = specRealisticSeeder.additionalTank;
				seedsVehicleFillUnitIndex = seedFillUnitIndex;
        	end;
		end;
	end;

	if seedsVehicle ~= nil and seedsVehicle ~= self then
		local fillType = g_fruitTypeManager:getFillTypeIndexByFruitTypeIndex(seedsFruitType);
		
       	if fillType ~= nil then 
			seedsVehicle:setFillUnitFillTypeToDisplay(seedsVehicleFillUnitIndex, fillType);
		end;
	end;
	
	local isTurnedOn = self:getIsTurnedOn();
	local canFruitBePlanted = false;
	
	if fruitType.terrainDataPlaneId ~= nil then
		--## check if selected fruit is available in current map
		canFruitBePlanted = true;
	end

	local isPlantingSeason = true;

	if not self:getCanPlantOutsideSeason() then
		isPlantingSeason = g_currentMission.growthSystem:canFruitBePlanted(seedsFruitType);
	end;
	
    if specSowingMachine.showWrongFruitForMissionWarning then
        specSowingMachine.showWrongFruitForMissionWarning = false;
	end;
	
    specSowingMachine.showFruitCanNotBePlantedWarning = not canFruitBePlanted;
	specSowingMachine.showWrongPlantingTimeWarning = not isPlantingSeason and (isTurnedOn or not specSowingMachine.needsActivation and self:getIsLowered());
	specSowingMachine.workAreaParameters.isActive = not specSowingMachine.needsActivation or isTurnedOn;
	specSowingMachine.workAreaParameters.canFruitBePlanted = canFruitBePlanted and isPlantingSeason;
	specSowingMachine.workAreaParameters.seedsFruitType = seedsFruitType;
	specSowingMachine.workAreaParameters.fieldGroundType = specSowingMachine.fieldGroundType;
	specSowingMachine.workAreaParameters.angle = angle;
	specSowingMachine.workAreaParameters.seedsVehicle = seedsVehicle;
	specSowingMachine.workAreaParameters.seedsVehicleFillUnitIndex = seedsVehicleFillUnitIndex;
	specSowingMachine.workAreaParameters.seedsVehicleUnloadInfoIndex = seedsVehicleUnloadInfoIndex;
	specSowingMachine.workAreaParameters.lastTotalArea = 0;
	specSowingMachine.workAreaParameters.lastChangedArea = 0;
	specSowingMachine.workAreaParameters.lastStatsArea = 0;
end;

SowingMachine.onStartWorkAreaProcessing = RealisticSeeder.onStartWorkAreaProcessing; --## need an hardcore overwrite... Otherwhise, the seed fill type will automaticlly set to "seeds"!

function RealisticSeeder:unloadFillUnitsToPallets(showWarning)
	if not self.isServer then
		g_client:getServerConnection():sendEvent(RealisticSeederUnloadFillUnitsToPalletsEvent.new(self));
	else
		local specRealisticSeeder = RealisticSeederUtil.getSpecByName(self, "realisticSeeder");
		local specFillUnit = RealisticSeederUtil.getSpecByName(self, "fillUnit");

		if specFillUnit.unloadingFillUnitsRunning then
			return;
		end;

		local currentUnloadSide = specRealisticSeeder.isUnloadingToLeft and 1 or -1;
		local unloadingPlaces = specFillUnit.unloading;
		local places = {};

		for _, unloading in ipairs(unloadingPlaces) do
			local node = unloading.node;
			local ox, oy, oz = unpack(unloading.offset);
			local x, y, z = localToWorld(node, ox - unloading.width * 0.1, oy, oz);

			local place = {
				startZ = z,
				startY = y,
				startX = x
			};

			place.rotX, place.rotY, place.rotZ = getWorldRotation(node);
			place.dirX, place.dirY, place.dirZ = localDirectionToWorld(node, currentUnloadSide, 0, 0);
			place.dirPerpX, place.dirPerpY, place.dirPerpZ = localDirectionToWorld(node, 0, 0, 1);
			place.yOffset = 1;
			place.maxWidth = math.huge;
			place.maxLength = math.huge;
			place.maxHeight = math.huge;
			place.width = unloading.width;

			table.insert(places, place);
		end;

		local usedPlaces = {};
		local success = true;
		local availablePallets = {};
		local unloadingTasks = {};

		local fillUnitNumber = specRealisticSeeder.currentFillUnitNumber;
		local fillLevel = self:getFillUnitFillLevel(fillUnitNumber);
		local fillTypeIndex = self:getFillUnitFillType(fillUnitNumber);
		local fillType = g_fillTypeManager:getFillTypeByIndex(fillTypeIndex);
		
		if fillType ~= nil then
			local fixedFillTypeName = RealisticSeederUtil.fixFillTypeName(fillType.name:lower()); --## fix invalid case warning's
				
			local palletPath = "pallets/pallet_";
				
			if fillLevel <= specRealisticSeeder.bagUnloadThreshold and fixedFillTypeName ~= "seed_potato" then
				palletPath = "bags/bag_";
			end;

			local palletFilename = RealisticSeederUtil.getPalletFilename(
				specRealisticSeeder,
				RealisticSeeder.XML_PATH_PREFIX,
				RealisticSeeder.UNLOAD_MODE_PALLETS,
				RealisticSeeder.UNLOAD_MODE_BIG_BAG_2,
				palletPath,
				fixedFillTypeName,
				fillLevel
			);
				
			if fillType.palletFilename ~= nil and fileExists(palletFilename) and fillType.palletFilename ~= palletFilename then
				--## switch pallet filename with big bag and vice versa
				fillType.palletFilename = palletFilename;
			end;
			
			if specFillUnit.fillUnits[fillUnitNumber].canBeUnloaded and fillLevel > 0 and fillType.palletFilename ~= nil then
				table.insert(unloadingTasks, {
					fillUnitIndex = fillUnitNumber,
					fillTypeIndex = fillTypeIndex,
					fillLevel = fillLevel,
					filename = fillType.palletFilename
				});
			end;
		end;

		local function unloadNext()
			local task = unloadingTasks[1];

			if task ~= nil then
				for pallet, _ in pairs(availablePallets) do
					local fillUnitIndex = pallet:getFirstValidFillUnitToFill(task.fillTypeIndex);

					if fillUnitIndex ~= nil then
						local appliedDelta = pallet:addFillUnitFillLevel(self:getOwnerFarmId(), fillUnitIndex, task.fillLevel, task.fillTypeIndex, ToolType.UNDEFINED, nil);

						self:addFillUnitFillLevel(self:getOwnerFarmId(), task.fillUnitIndex, -appliedDelta, task.fillTypeIndex, ToolType.UNDEFINED, nil);

						task.fillLevel = task.fillLevel - appliedDelta;

						if pallet:getFillUnitFreeCapacity(fillUnitIndex) <= 0 then
							availablePallets[pallet] = nil;
						end;
					end;
				end;

				if task.fillLevel > 0 then
					local function asyncCallback(_, vehicle, vehicleLoadState, arguments)
						if vehicleLoadState == VehicleLoadingUtil.VEHICLE_LOAD_OK then
							vehicle:emptyAllFillUnits(true);

							availablePallets[vehicle] = true;

							unloadNext();
						end;
					end;

					local size = StoreItemUtil.getSizeValues(task.filename, "vehicle", 0, {});
					local x, y, z, place, width, _ = PlacementUtil.getPlace(places, size, usedPlaces, true, true, true);

					if x == nil then
						success = false;

						if (ignoreWarning == nil or not ignoreWarning) and not success then
							g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO, specFillUnit.texts.unloadNoSpace);
						end;

						return;
					end;

					PlacementUtil.markPlaceUsed(usedPlaces, place, width);

					local location = {
						x = x,
						y = y,
						z = z,
						yRot = place.rotY
					};

					VehicleLoadingUtil.loadVehicle(task.filename, location, true, 0, Vehicle.PROPERTY_STATE_OWNED, self:getOwnerFarmId(), nil, nil, asyncCallback, nil);
				else
					table.remove(unloadingTasks, 1);

					unloadNext();
				end;
			end;
		end;

		unloadNext();
	end;
end;

function RealisticSeeder:updateUnloadActionDisplay(superFunc)
	if self.spec_fillUnit.unloading ~= nil then
		local isActive = self.spec_sprayer ~= nil;

		if not isActive then
			for fillUnitNumber, fillUnit in ipairs(self:getFillUnits()) do
				local fillLevel = self:getFillUnitFillLevel(fillUnitNumber);
				local fillTypeIndex = self:getFillUnitFillType(fillUnitNumber);
				local fillType = g_fillTypeManager:getFillTypeByIndex(fillTypeIndex);

				if fillUnit.canBeUnloaded and fillLevel > 0 and fillType.palletFilename ~= nil then
					isActive = fillType.name == "HERBICIDE" or fillType.name == "LIQUID_FERTILIZER";

					break;
				end;
			end;
		end;

		if self.getCurrentDischargeNode ~= nil then
			local dischargeNode = self:getCurrentDischargeNode();

			if dischargeNode ~= nil then
				local dischargeObject, _ = self:getDischargeTargetObject(dischargeNode);

				if dischargeObject ~= nil then
					isActive = false;
				end;
			end;
		end;
		
		g_inputBinding:setActionEventActive(self.spec_fillUnit.unloadActionEventId, isActive);
	end;
end;

FillUnit.updateUnloadActionDisplay = RealisticSeeder.updateUnloadActionDisplay;

function RealisticSeeder:updateFillLevelFrames(superFunc)
	local _, yOffset = self:getPosition();
	local isFirst = true

	for fillLevelBufferNumber = 1, #self.fillLevelBuffer do
		local fillLevelInformation = self.fillLevelBuffer[fillLevelBufferNumber];

		if fillLevelInformation.capacity > 0 or fillLevelInformation.fillLevel > 0 then
			if self.fillTypeFrames[fillLevelInformation.fillType] ~= nil then 
				local value = 0;

				if fillLevelInformation.capacity > 0 then
					value = fillLevelInformation.fillLevel / fillLevelInformation.capacity;
				end;

				local frame = self.fillTypeFrames[fillLevelInformation.fillType];

				frame:setVisible(true);

				local fillBar = self.fillTypeLevelBars[fillLevelInformation.fillType];

				fillBar:setValue(value);

				local baseX = self:getPosition();

				if isFirst then
					baseX = baseX + self.firstFillTypeOffset;
				end;

				frame:setPosition(baseX, yOffset);

				local precision = fillLevelInformation.precision or 0;
				local formattedNumber = nil;

				if precision > 0 then
					local rounded = MathUtil.round(fillLevelInformation.fillLevel, precision);
					
					formattedNumber = "%d%s%0" .. precision .. "d";
					formattedNumber = formattedNumber:format(math.floor(rounded), g_i18n.decimalSeparator, (rounded - math.floor(rounded)) * 10^precision);
				else
					formattedNumber = "%d";
					formattedNumber = formattedNumber:format(MathUtil.round(fillLevelInformation.fillLevel));
				end;

				self.weightFrames[fillLevelInformation.fillType]:setVisible(fillLevelInformation.maxReached);

				local fillTypeName, unitShort = nil;

				if fillLevelInformation.fillType ~= FillType.UNKNOWN then
					local fillTypeDesc = g_fillTypeManager:getFillTypeByIndex(fillLevelInformation.fillType);
					fillTypeName = fillTypeDesc.title;
					unitShort = fillTypeDesc.unitShort;
				end;

				local fillText = "%s%s (%d%%)";
				
				fillText = fillText:format(formattedNumber, unitShort or "", math.floor(100 * value));
				self.fillLevelTextBuffer[#self.fillLevelTextBuffer + 1] = fillText;

				if fillTypeName ~= nil then
					self.fillTypeTextBuffer[#self.fillLevelTextBuffer] = fillTypeName;
				end;

				yOffset = yOffset + self.frameHeight + self.frameOffsetY;
				isFirst = false;
			else
				--## fix, create fill type frame when nil, instead of spamming the log file full... Thanks again to GIANTS for this issue.
				local posX, posY = self:getPosition();
				local frame = self:createFillTypeFrame(self.hudAtlasPath, posX, posY, g_fillTypeManager:getFillTypeByIndex(fillLevelInformation.fillType));

				self.fillTypeFrames[fillLevelInformation.fillType] = frame;

				frame:setScale(self.uiScale, self.uiScale);

				self:addChild(frame);
			end;
		end;
	end;
end;

FillLevelsDisplay.updateFillLevelFrames = Utils.overwrittenFunction(FillLevelsDisplay.updateFillLevelFrames, RealisticSeeder.updateFillLevelFrames);

function RealisticSeeder:updateFillLevelBuffers()
	local function clearTable(table)
		for tableNumber = #table, 1, -1 do
			table[tableNumber] = nil;
		end;
	end;
	
	local function sortBuffer(a, b)
		return a.addIndex < b.addIndex;
	end;

	clearTable(self.fillLevelTextBuffer);
	clearTable(self.fillTypeTextBuffer);

	for fillLevelBufferNumber = 1, #self.fillLevelBuffer do
		local fillLevelInformation = self.fillLevelBuffer[fillLevelBufferNumber];
		local frame = self.fillTypeFrames[fillLevelInformation.fillType];

		if frame ~= nil and frame.setVisible ~= nil then	
			frame:setVisible(false);
		end;
	end;

	for fillLevelBufferNumber = 1, #self.fillLevelBuffer do
		self.fillLevelBuffer[fillLevelBufferNumber].fillLevel = 0;
		self.fillLevelBuffer[fillLevelBufferNumber].capacity = 0;
	end;

	self.addIndex = 0;
	self.needsSorting = false;

	self.vehicle:getFillLevelInformation(self);

	if self.needsSorting then
		table.sort(self.fillLevelBuffer, sortBuffer);
	end;
end;

FillLevelsDisplay.updateFillLevelBuffers = Utils.overwrittenFunction(FillLevelsDisplay.updateFillLevelBuffers, RealisticSeeder.updateFillLevelBuffers);

function RealisticSeeder:assignItemAttributeData(displayItem)
	if RealisticSeederUtil ~= nil then
		local vehicleXMLFile = loadXMLFile("vehicle", displayItem.storeItem.xmlFilename);

		if vehicleXMLFile ~= nil then
			if RealisticSeederUtil.getIsValidStoreCategory(getXMLString(vehicleXMLFile, "vehicle.storeData.category")) then
				local foundFunctionAdditionalTank = RealisticSeederUtil.foundFunctionAdditionalTank(vehicleXMLFile);
				local foundSeeds = RealisticSeederUtil.foundFillTypeNameInTable(displayItem.fillTypeIconFilenames, "seeds");
				local seedFruitTypeCategories = Utils.getNoNil(getXMLString(vehicleXMLFile, "vehicle.sowingmachine.seedFruitTypeCategories"), "");
				local seedFruitTypes = Utils.getNoNil(getXMLString(vehicleXMLFile, "vehicle.sowingmachine.seedFruitTypes"), "");
				local seedFruitTypeNames = {};
				
				if seedFruitTypes ~= "" then
					seedFruitTypeNames = seedFruitTypes:lower():split(" ");
				end;
			
				if seedFruitTypeCategories == "" and #seedFruitTypeNames == 0 and hasXMLProperty(vehicleXMLFile, "vehicle.storeData.bundleElements") then
					local bundleElementNumber = 0;
				
					while true do
						local bundleElementKey = "vehicle.storeData.bundleElements.bundleElement(" .. tostring(bundleElementNumber) .. ")";
					
						if not hasXMLProperty(vehicleXMLFile, bundleElementKey) then
							break;
						end;
					
						local bundleElementXMLFilename = Utils.getNoNil(getXMLString(vehicleXMLFile, bundleElementKey .. ".xmlFilename"), "");
					
						if bundleElementXMLFilename ~= "" then
							local bundleElementXMLFile = loadXMLFile("vehicle", Utils.getFilename(bundleElementXMLFilename, displayItem.storeItem.baseDir));
						
							if bundleElementXMLFile ~= nil then
								seedFruitTypeCategoriesCheck = Utils.getNoNil(getXMLString(bundleElementXMLFile, "vehicle.sowingmachine.seedFruitTypeCategories"), "");
								seedFruitTypeNamesCheck = Utils.getNoNil(getXMLString(bundleElementXMLFile, "vehicle.sowingmachine.seedFruitTypes"), "");
							
								if seedFruitTypeCategoriesCheck ~= "" then
									seedFruitTypeCategories = seedFruitTypeCategoriesCheck;
								
									break;
								elseif seedFruitTypeNamesCheck ~= "" then
									seedFruitTypeNames = seedFruitTypeNamesCheck:lower():split(" ");
								
									break;
								end;
							
								delete(bundleElementXMLFile);
							end;
						end;
					
						bundleElementNumber = bundleElementNumber + 1;
					end;
				end;
			
				if foundFunctionAdditionalTank then
					local foundFertilizer = RealisticSeederUtil.foundFillTypeNameInTable(displayItem.fillTypeIconFilenames, "fertilizer");
				
					if not foundFertilizer then
						--## add fertilizer icon to the shop overview for additional tanks, if not already added
						table.insert(displayItem.fillTypeIconFilenames, "dataS/menu/hud/fillTypes/hud_fill_fertilizer.png");
					end;
				
					if not foundSeeds then
						--## add seeds icon to the shop overview for additional tanks, if not already added
						table.insert(displayItem.fillTypeIconFilenames, "dataS/menu/hud/fillTypes/hud_fill_seeds.png");
					end;
				end;

				if foundSeeds or foundFunctionAdditionalTank then
					local seedFillTypes = RealisticSeederUtil.getSeedFillTypes();
				
					for _, seedFillType in pairs(seedFillTypes) do
						if RealisticSeederUtil.getCanFruitBePlanted(seedFillType, nil) then
							local seedFillTypeName = g_fillTypeManager:getFillTypeByIndex(seedFillType).name:lower();
							local foundSeedFillTypeName = RealisticSeederUtil.foundFillTypeNameInTable(displayItem.fillTypeIconFilenames, seedFillTypeName);

							if #seedFruitTypeNames == 0 then
								local seedFruitTypeCategory = g_fruitTypeManager:getFruitTypesByCategoryNames(seedFruitTypeCategories);
								seedFruitTypeNames = {};

								for _, fruitType in pairs(seedFruitTypeCategory) do
									table.insert(seedFruitTypeNames, g_fruitTypeManager:getFruitTypeByIndex(fruitType).name:lower());
								end;
							end;

							local seedFillTypeIsAvaiable = true;

							if #seedFruitTypeNames > 0 then
								seedFillTypeIsAvaiable = RealisticSeederUtil.foundFillTypeNameInTable(seedFruitTypeNames, seedFillTypeName:sub(6));
							end;

							if not foundSeedFillTypeName and seedFillTypeIsAvaiable then
								local fillTypePath = Utils.getFilename("hud/hud_fill_" .. RealisticSeederUtil.fixFillTypeName(seedFillTypeName) .. ".dds", RealisticSeederUtil.currentModDirectory);
							
								if fileExists(fillTypePath) then
									--## add fill type icons from my seed fill types to the shop overview, if avaiable and not already added
									table.insert(displayItem.fillTypeIconFilenames, fillTypePath);
								else
									RealisticSeederUtil.printError("Can't load image file (" .. fillTypePath .. ")! Aborting adding this imge to the shop view!", false, false, "RealisticSeeder");
								end;
							end;
						end;
					end;
				end;
			end;

			delete(vehicleXMLFile);
		end;
	end;
end;

ShopItemsFrame.assignItemAttributeData = Utils.prependedFunction(ShopItemsFrame.assignItemAttributeData, RealisticSeeder.assignItemAttributeData);

----------------------------------------------------------------------------------------------------------------------------------
---------------------------------------RealisticSeederUnloadFillUnitsToPalletsEvent-----------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------

RealisticSeederUnloadFillUnitsToPalletsEvent = {};

local RealisticSeederUnloadFillUnitsToPalletsEvent_mt = Class(RealisticSeederUnloadFillUnitsToPalletsEvent, Event);

InitEventClass(RealisticSeederUnloadFillUnitsToPalletsEvent, "RealisticSeederUnloadFillUnitsToPalletsEvent");

function RealisticSeederUnloadFillUnitsToPalletsEvent.emptyNew()
	local self = Event.new(RealisticSeederUnloadFillUnitsToPalletsEvent_mt);

	return self;
end;

function RealisticSeederUnloadFillUnitsToPalletsEvent.new(vehicle)
	local self = RealisticSeederUnloadFillUnitsToPalletsEvent.emptyNew();

	self.vehicle = vehicle;
	
	return self;
end;

function RealisticSeederUnloadFillUnitsToPalletsEvent:readStream(streamId, connection)
	self.vehicle = NetworkUtil.readNodeObject(streamId);
	
	self:run(connection);
end;

function RealisticSeederUnloadFillUnitsToPalletsEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.vehicle);
end;

function RealisticSeederUnloadFillUnitsToPalletsEvent:run(connection)
	if not connection:getIsServer() then
		if self.vehicle ~= nil and self.vehicle:getIsSynchronized() then
			local success = self.vehicle:unloadFillUnitsToPallets(true);

			if not success then
				connection:sendEvent(RealisticSeederUnloadFillUnitsToPalletsEvent.new(self.vehicle));
			end;
		end;
	else
		g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO, g_i18n:getText("fillUnit_unload_nospace"));
	end;
end;

--------------------------------------------------------------------------------------------------------------------------
---------------------------------------RealisticSeederSwitchUnloadModeEvent-----------------------------------------------
--------------------------------------------------------------------------------------------------------------------------

RealisticSeederSwitchUnloadModeEvent = {};

local RealisticSeederSwitchUnloadModeEvent_mt = Class(RealisticSeederSwitchUnloadModeEvent, Event);

InitEventClass(RealisticSeederSwitchUnloadModeEvent, "RealisticSeederSwitchUnloadModeEvent");

function RealisticSeederSwitchUnloadModeEvent.emptyNew()
	local self = Event.new(RealisticSeederSwitchUnloadModeEvent_mt);
    
	return self;
end;

function RealisticSeederSwitchUnloadModeEvent.new(sowingMachine, currentUnloadMode)
	local self = RealisticSeederSwitchUnloadModeEvent.emptyNew();
	
	self.sowingMachine = sowingMachine;
	self.currentUnloadMode = currentUnloadMode;

	return self;
end;

function RealisticSeederSwitchUnloadModeEvent:readStream(streamId, connection)
	self.sowingMachine = NetworkUtil.readNodeObject(streamId);
	self.currentUnloadMode = streamReadUInt8(streamId);
	
	self:run(connection);
end;

function RealisticSeederSwitchUnloadModeEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.sowingMachine);

	streamWriteUInt8(streamId, self.currentUnloadMode);
end;

function RealisticSeederSwitchUnloadModeEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(RealisticSeederSwitchUnloadModeEvent.new(self.sowingMachine, self.currentUnloadMode), nil, connection, self.sowingMachine);
	end;
	
    if self.sowingMachine ~= nil then
        self.sowingMachine:switchUnloadMode(self.currentUnloadMode, true);
	end;
end;

function RealisticSeederSwitchUnloadModeEvent.sendEvent(sowingMachine, currentUnloadMode, noEventSend)
	if not noEventSend then
		if g_server ~= nil then
			g_server:broadcastEvent(RealisticSeederSwitchUnloadModeEvent.new(sowingMachine, currentUnloadMode), nil, nil, sowingMachine);
		else
			g_client:getServerConnection():sendEvent(RealisticSeederSwitchUnloadModeEvent.new(sowingMachine, currentUnloadMode));
		end;
	end;
end;

--------------------------------------------------------------------------------------------------------------------------
-------------------------------------RealisticSeederSwitchFillUnitNumberEvent---------------------------------------------
--------------------------------------------------------------------------------------------------------------------------

RealisticSeederSwitchFillUnitNumberEvent = {};

local RealisticSeederSwitchFillUnitNumberEvent_mt = Class(RealisticSeederSwitchFillUnitNumberEvent, Event);

InitEventClass(RealisticSeederSwitchFillUnitNumberEvent, "RealisticSeederSwitchFillUnitNumberEvent");

function RealisticSeederSwitchFillUnitNumberEvent.emptyNew()
	local self = Event.new(RealisticSeederSwitchFillUnitNumberEvent_mt);
    
	return self;
end;

function RealisticSeederSwitchFillUnitNumberEvent.new(vehicle, fillUnitNumber)
	local self = RealisticSeederSwitchFillUnitNumberEvent.emptyNew();
	
	self.vehicle = vehicle;
	self.fillUnitNumber = fillUnitNumber;

	return self;
end;

function RealisticSeederSwitchFillUnitNumberEvent:readStream(streamId, connection)
	self.vehicle = NetworkUtil.readNodeObject(streamId);
	self.fillUnitNumber = streamReadUInt8(streamId);

	self:run(connection);
end;

function RealisticSeederSwitchFillUnitNumberEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.vehicle);

	streamWriteUInt8(streamId, self.fillUnitNumber);
end;

function RealisticSeederSwitchFillUnitNumberEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(RealisticSeederSwitchFillUnitNumberEvent.new(self.vehicle, self.fillUnitNumber), nil, connection, self.vehicle);
	end;
	
    if self.vehicle ~= nil then
		self.vehicle:switchFillUnit(self.fillUnitNumber, true);
	end;
end;

function RealisticSeederSwitchFillUnitNumberEvent.sendEvent(vehicle, fillUnitNumber, noEventSend)
	if not noEventSend then
		if g_server ~= nil then
			g_server:broadcastEvent(RealisticSeederSwitchFillUnitNumberEvent.new(vehicle, fillUnitNumber), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(RealisticSeederSwitchFillUnitNumberEvent.new(vehicle, fillUnitNumber));
		end;
	end;
end;

--------------------------------------------------------------------------------------------------------------------------
-------------------------------------RealisticSeederSwitchFertilizerModeEvent---------------------------------------------
--------------------------------------------------------------------------------------------------------------------------

RealisticSeederSwitchFertilizerModeEvent = {};

local RealisticSeederSwitchFertilizerModeEvent_mt = Class(RealisticSeederSwitchFertilizerModeEvent, Event);

InitEventClass(RealisticSeederSwitchFertilizerModeEvent, "RealisticSeederSwitchFertilizerModeEvent");

function RealisticSeederSwitchFertilizerModeEvent.emptyNew()
	local self = Event.new(RealisticSeederSwitchFertilizerModeEvent_mt);
    
	return self;
end;

function RealisticSeederSwitchFertilizerModeEvent.new(sowingMachine, allowFertilizer)
	local self = RealisticSeederSwitchFertilizerModeEvent.emptyNew();
	
	self.sowingMachine = sowingMachine;
	self.allowFertilizer = allowFertilizer;

	return self;
end;

function RealisticSeederSwitchFertilizerModeEvent:readStream(streamId, connection)
	self.sowingMachine = NetworkUtil.readNodeObject(streamId);
	self.allowFertilizer = streamReadBool(streamId);

	self:run(connection);
end;

function RealisticSeederSwitchFertilizerModeEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.sowingMachine);

	streamWriteBool(streamId, self.allowFertilizer);
end;

function RealisticSeederSwitchFertilizerModeEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(RealisticSeederSwitchFertilizerModeEvent.new(self.sowingMachine, self.allowFertilizer), nil, connection, self.sowingMachine);
	end;
	
    if self.sowingMachine ~= nil then
		self.sowingMachine:switchFertilizerMode(self.allowFertilizer, true);
	end;
end;

function RealisticSeederSwitchFertilizerModeEvent.sendEvent(sowingMachine, allowFertilizer, noEventSend)
	if not noEventSend then
		if g_server ~= nil then
			g_server:broadcastEvent(RealisticSeederSwitchFertilizerModeEvent.new(sowingMachine, allowFertilizer), nil, nil, sowingMachine);
		else
			g_client:getServerConnection():sendEvent(RealisticSeederSwitchFertilizerModeEvent.new(sowingMachine, allowFertilizer));
		end;
	end;
end;

--------------------------------------------------------------------------------------------------------------------------
---------------------------------------RealisticSeederSwitchUnloadSideEvent-----------------------------------------------
--------------------------------------------------------------------------------------------------------------------------

RealisticSeederSwitchUnloadSideEvent = {};

local RealisticSeederSwitchUnloadSideEvent_mt = Class(RealisticSeederSwitchUnloadSideEvent, Event);

InitEventClass(RealisticSeederSwitchUnloadSideEvent, "RealisticSeederSwitchUnloadSideEvent");

function RealisticSeederSwitchUnloadSideEvent.emptyNew()
	local self = Event.new(RealisticSeederSwitchUnloadSideEvent_mt);
    
	return self;
end;

function RealisticSeederSwitchUnloadSideEvent.new(vehicle, isUnloadingToLeft)
	local self = RealisticSeederSwitchUnloadSideEvent.emptyNew();
	
	self.vehicle = vehicle;
	self.isUnloadingToLeft = isUnloadingToLeft;

	return self;
end;

function RealisticSeederSwitchUnloadSideEvent:readStream(streamId, connection)
	self.vehicle = NetworkUtil.readNodeObject(streamId);
	self.isUnloadingToLeft = streamReadBool(streamId);

	self:run(connection);
end;

function RealisticSeederSwitchUnloadSideEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.vehicle);

	streamWriteBool(streamId, self.isUnloadingToLeft);
end;

function RealisticSeederSwitchUnloadSideEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(RealisticSeederSwitchUnloadSideEvent.new(self.vehicle, self.isUnloadingToLeft), nil, connection, self.vehicle);
	end;
	
    if self.vehicle ~= nil then
		self.vehicle:switchUnloadSide(self.isUnloadingToLeft, true);
	end;
end;

function RealisticSeederSwitchUnloadSideEvent.sendEvent(vehicle, isUnloadingToLeft, noEventSend)
	if not noEventSend then
		if g_server ~= nil then
			g_server:broadcastEvent(RealisticSeederSwitchUnloadSideEvent.new(vehicle, isUnloadingToLeft), nil, nil, vehicle);
		else
			g_client:getServerConnection():sendEvent(RealisticSeederSwitchUnloadSideEvent.new(vehicle, isUnloadingToLeft));
		end;
	end;
end;