----------------------------------------------------------------------------------
-- Copyright: ViperGTS96------------------------------------------------
----------------------------------------------------------------------------------
--------------------"The simplest design is the best design." --------------
----------------------------------------------------------------------------------
--- FS22/LS22 ------------------------------------------------------------------
-- date Jan1/2019 - Sept21/2022---------------------------------------------
----------------------------------------------------------------------------------

realDirtColor = {};

function realDirtColor.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Washable, specializations)
		and SpecializationUtil.hasSpecialization(AnimatedVehicle, specializations);
end;

function realDirtColor.registerEventListeners(vehicleType)
	for _, functionName in pairs( { "onLoadFinished", "onUpdate", "saveToXMLFile", "onReadStream", "onWriteStream" } ) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, realDirtColor);
	end;
end;

function realDirtColor.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "findWheelDirtColor", realDirtColor.findWheelDirtColor);
	SpecializationUtil.registerFunction(vehicleType, "getNewColorIndex", realDirtColor.getNewColorIndex);
	SpecializationUtil.registerFunction(vehicleType, "getFillTypeColorIndex", realDirtColor.getFillTypeColorIndex);
	SpecializationUtil.registerFunction(vehicleType, "getFruitTypeColorIndex", realDirtColor.getFruitTypeColorIndex);
	SpecializationUtil.registerFunction(vehicleType, "getSprayTypeColorIndex", realDirtColor.getSprayTypeColorIndex);
	SpecializationUtil.registerFunction(vehicleType, "getTextureColorIndex", realDirtColor.getTextureColorIndex);
	SpecializationUtil.registerFunction(vehicleType, "adjustColorSpeed", realDirtColor.adjustColorSpeed);
	SpecializationUtil.registerFunction(vehicleType, "colorChangeLerp", realDirtColor.colorChangeLerp);
	SpecializationUtil.registerFunction(vehicleType, "setChildrenDirtColor", realDirtColor.setChildrenDirtColor);
	SpecializationUtil.registerFunction(vehicleType, "thresholdCompareColor", realDirtColor.thresholdCompareColor);
	SpecializationUtil.registerFunction(vehicleType, "floorRDCColorIndex", realDirtColor.floorRDCColorIndex);
end;

function realDirtColor.registerOverwrittenFunctions(vehicleType)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "setNodeDirtColor", realDirtColor.setNodeDirtColor);
end;

function realDirtColor:setChildrenDirtColor(parentNode,r,g,b)
	if getNumOfChildren(parentNode) > 0 then
		for i=0, getNumOfChildren(parentNode)-1 do
			local childNode = getChildAt(parentNode, i);
			if getHasClassId(childNode, ClassIds.SHAPE) and getHasShaderParameter(childNode, "dirtColor") then
				setShaderParameter(childNode, "dirtColor", r,g,b, 0, false);
			end;
			self:setChildrenDirtColor(childNode,r,g,b);
		end;
	end;
end;

function realDirtColor:thresholdCompareColor(source,target,threshold)
	local sameValues = 0;
	if math.abs(source.r - target.r) <= threshold then sameValues = sameValues+1; end;
	if math.abs(source.g - target.g) <= threshold then sameValues = sameValues+1; end;
	if math.abs(source.b - target.b) <= threshold then sameValues = sameValues+1; end;
	return (sameValues == 3) ;
end;


function realDirtColor:floorRDCColorIndex(curIndex,prevIndex)
	if curIndex > 0.09 and curIndex < 0.11 then curIndex = 0.1;
	elseif curIndex > 0.49 and curIndex < 0.51 then curIndex = 0.5;
	elseif curIndex > 0.54 and curIndex < 0.56 then curIndex = 0.55; end;
	if prevIndex > 0.09 and prevIndex < 0.11 then prevIndex = 0.1;
	elseif prevIndex > 0.49 and prevIndex < 0.51 then prevIndex = 0.5;
	elseif prevIndex > 0.54 and prevIndex < 0.56 then prevIndex = 0.55; end;
	return curIndex,prevIndex;
end;

function realDirtColor:debugColorInfo(vehicle, text, isWheel)
	if isWheel == nil or not isWheel then
		if vehicle:getRootVehicle() == g_currentMission.controlledVehicle then
			g_currentMission:addExtraPrintText(text);
		end;
	end;
end;

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------PostLoad--------------------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

function realDirtColor:onLoadFinished(savegame)
  	self.spec_realDirtColor = {};
  	self.spec_realDirtColor.bodypart = {};
	self.spec_realDirtColor.useSimpleMode = false;

	local spec = self.spec_realDirtColor;
	if self.spec_washable ~= nil then
		for _, washableNodes in pairs(self.spec_washable.washableNodes) do
			for _, node in pairs(washableNodes.nodes) do
				table.insert(spec.bodypart,node);
			end;
		end;
	else
		self.spec_realDirtColor = nil;
		if realDirtColor.debugging then
			print(realDirtColor.title.." : No body parts found for "..FSBaseMission:getVehicleName(self));
		end;
		return; --abort script
	end;

	if spec.bodypart ~= nil and #spec.bodypart > 0 then
		local disableRDC = Utils.getNoNil(getXMLBool(self.xmlFile.handle, string.format("vehicle.realDirtColor#disable")), false);
		if disableRDC then
			self.spec_realDirtColor = nil;
			if realDirtColor.debugging then
				print(realDirtColor.title.." disabled for "..FSBaseMission:getVehicleName(self));
			end;
			return; --abort script
		end;
		local overrideToSimpleMode = Utils.getNoNil(getXMLBool(self.xmlFile.handle, string.format("vehicle.realDirtColor#useSimpleMode")), false);
		if overrideToSimpleMode or realDirtColor.simpleMode then
			spec.useSimpleMode = true;
			overrideToSimpleMode = nil;
		end;
		local ignoreRDCWhlLimit = Utils.getNoNil(getXMLBool(self.xmlFile.handle, string.format("vehicle.realDirtColor#ignoreRDCWheelLimit")), false);
		if not ignoreRDCWhlLimit then
			if not spec.useSimpleMode then
				local storeCat = g_storeManager:getItemByXMLFilename(self.configFileName)
				if storeCat ~= nil and storeCat.categoryName ~= nil then
					storeCat = storeCat.categoryName;
					for _, type in pairs(realDirtColor.excludedToolTypes) do
						if type == storeCat then
							spec.useSimpleMode = true;
							spec.detectPointAdj = Utils.getNoNil((self.size.length/2)-1.0, 0.5);
							if realDirtColor.debugging then
								print(realDirtColor.title.." detected a vehicle in "..storeCat.."; Vehicle/tool set to simple mode.");
							end;
						end;
					end;
				end;
			end;
			if not spec.useSimpleMode then
				if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels > realDirtColor.maxWheelLimit then
					spec.useSimpleMode = true; --Reduce CPU usage by shutting off independent wheel mode on vehicles (huge cultivators)
					if realDirtColor.debugging then
						print(realDirtColor.title.." detected more than "..realDirtColor.maxWheelLimit.." wheels; Vehicle/tool set to simple mode.");
					end;
				end;
			end;
		end;

		if not spec.useSimpleMode then
	   ---------------------------------------------------------------------------------------------------------------------------
	   ---------------------Remove Wheel Parts from Body Part Table for Independent Color Control---------------------------------
	   ---------------------------------------------------------------------------------------------------------------------------
			--[[if realDirtColor.debugging then print(#spec.bodypart); end;]]
			-------------Wheels-----------
			if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels > 0 then
				for _, wheel in pairs(self.spec_wheels.wheels) do
					wheel.bodypart = {};
					local wheelNodes = I3DUtil.getNodesByShaderParam(wheel.repr, "dirtColor", wheel)
					for _, wheelNode in pairs(wheelNodes) do
						for partIndex, bodypart in pairs(spec.bodypart) do
							local skipPart = false;
							for _, component in pairs(self.components) do
								if bodypart == component.node then skipPart = true; end;
							end;
							if string.find(string.lower(getName(bodypart)), "steering") then skipPart = true; end;
							if bodypart == wheelNode and not skipPart then
								table.remove(spec.bodypart, partIndex);
								table.insert(wheel.bodypart,wheelNode);
								if realDirtColor.debugging and realDirtColor.writeToLog then
									print("removed wheel.repr part : "..getName(bodypart).." / "..getName(wheelNode));
								end;
								wheel.rDC = true;
								break;
							end;
						end;
					end;
					if wheel.repr ~= wheel.driveNode then
						wheelNodes = I3DUtil.getNodesByShaderParam(wheel.driveNode, "dirtColor", wheel)
						for _, wheelNode in pairs(wheelNodes) do
							for partIndex, bodypart in pairs(spec.bodypart) do
								local skipPart = false;
								for _, component in pairs(self.components) do
									if bodypart == component.node then skipPart = true; end;
								end;
								if string.find(string.lower(getName(bodypart)), "steering") then skipPart = true; end;
								if bodypart == wheelNode and not skipPart then
									table.remove(spec.bodypart, partIndex);
									table.insert(wheel.bodypart,wheelNode);
									if realDirtColor.debugging and realDirtColor.writeToLog then
										print("removed wheel.driveNode part : "..getName(bodypart).." / "..getName(wheelNode));
									end;
									wheel.rDC = true;
									break;
								end;
							end;
						end;
					end;
					if #wheel.bodypart < 1 then wheel.bodypart = nil; end; 
				end;
				if realDirtColor.debugging and realDirtColor.writeToLog then 
					print("Body Parts : "..#spec.bodypart);
					for i, wheel in pairs(self.spec_wheels.wheels) do
						if wheel.bodypart ~= nil then print("Wheel #"..tostring(i).." Parts : "..#wheel.bodypart); end;
					end;
				end;
			end;
		end;
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--------------------------------------------------------------------------Initialize Colors to Components---------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
		spec.previousDirtColor = 0.1; 
		spec.currentDirtColor = 0.1; 
		spec.newDirtColor = 0;
		-------------------------------------------------------------------------------------	
		
		spec.rdWidth = Utils.getNoNil(self.size.width, 0);
		spec.rdLength = Utils.getNoNil(self.size.length, 0);

		if spec.rdWidth <= 0 then
			spec.rdWidth = 2;
		else
			spec.rdWidth = spec.rdWidth - (spec.rdWidth * 0.75) --Minus wheels
		end;
		if spec.rdLength <= 0 then
			spec.rdLength = 2;
		else
			spec.rdLength = spec.rdLength - (spec.rdLength * 0.85) -- Minus Front and rear links/weights
		end;

		spec.targetRDColor = {};
		spec.targetRDColor.r, spec.targetRDColor.g, spec.targetRDColor.b = realDirtColor.colors.dryDirt.r, realDirtColor.colors.dryDirt.g, realDirtColor.colors.dryDirt.b;

		if realDirtColor.colors.dryDirt.r == realDirtColor.colors.brownDirt.r then
			spec.dryDirtColor = 0;--to detect brown dirt terrain
		else
			spec.dryDirtColor = 1;--to detect red dirt terrain
		end;
		
		if not spec.useSimpleMode then
			if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels ~= 0 then
				for i, wheel in pairs(self.spec_wheels.wheels) do
					if wheel.rDC ~= nil and wheel.rDC then
						wheel.previousDirtColor = 0.1;
						wheel.currentDirtColor = 0.1;
						wheel.newDirtColor = 0;
						wheel.targetRDColor = {};
						wheel.targetRDColor.r, wheel.targetRDColor.g, wheel.targetRDColor.b = spec.targetRDColor.r, spec.targetRDColor.g, spec.targetRDColor.b;
					end;
				end;
			end;
		end;
		
		if realDirtColor.writeToLog then
			print(realDirtColor.title.." successfully applied to "..FSBaseMission:getVehicleName(self));
		end;
	else
		self.spec_realDirtColor = nil;
		if realDirtColor.debugging then
			Logging.error(realDirtColor.title.." : Error loading body parts for "..FSBaseMission:getVehicleName(self));
		end;
		return; --abort script
	end;
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------Load Save Game Stats-----------------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    if savegame ~= nil and self.isServer then
		if self.spec_realDirtColor ~= nil then
			local spec = self.spec_realDirtColor;
			local colorR = getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.body#r"));
			local colorG = getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.body#g"));
			local colorB = getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.body#b"));
			local bodyCurDirtColor = Utils.getNoNil(getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.body#curIndex")), 0.1);
			local bodyPrevDirtColor = Utils.getNoNil(getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.body#prevIndex")), 0.1);

			bodyCurDirtColor, bodyPrevDirtColor = self:floorRDCColorIndex(bodyCurDirtColor, bodyPrevDirtColor);

			if (colorR ~= nil and colorG ~= nil and colorB ~= nil) then
				spec.currentDirtColor = bodyCurDirtColor;
				spec.previousDirtColor = bodyPrevDirtColor;
				local _,_,targetRDColor = self:getNewColorIndex(0, 0, bodyCurDirtColor, spec.targetRDColor);
				spec.targetRDColor = targetRDColor;

				for _, part in pairs(spec.bodypart) do
					setShaderParameter(part, "dirtColor", colorR, colorG, colorB, 0, false);
				end;
			end;

			if not spec.useSimpleMode then
				if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels ~= 0 then
					for i, wheel in pairs(self.spec_wheels.wheels) do
						if wheel.rDC ~= nil and wheel.rDC then
					
							local wColorR = getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.wheel"..i.."#r"));
							local wColorG = getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.wheel"..i.."#g"));
							local wColorB = getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.wheel"..i.."#b"));
							local wCurDirtColor = Utils.getNoNil(getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.wheel"..i.."#curIndex")), 0.1);
							local wPrevDirtColor = Utils.getNoNil(getXMLFloat(savegame.xmlFile.handle, savegame.key .. string.format(".realDirtColor.wheel"..i.."#prevIndex")), 0.1);
							
							wCurDirtColor, wPrevDirtColor = self:floorRDCColorIndex(wCurDirtColor, wPrevDirtColor);

							if (wColorR ~= nil and wColorG ~= nil and wColorB ~= nil) then
								colorR, colorG, colorB = wColorR, wColorG, wColorB;
								wheel.currentDirtColor = wCurDirtColor;
								wheel.previousDirtColor = wPrevDirtColor;
								local _,_,targetRDColor = self:getNewColorIndex(0, 0, wCurDirtColor, wheel.targetRDColor);
								wheel.targetRDColor = targetRDColor;
								if wheel.bodypart ~= nil then
									for _, part in pairs(wheel.bodypart) do
										setShaderParameter(part, "dirtColor", colorR, colorG, colorB, 0, false);
									end;
								end;
							end;

						end;
					end;
				end;
			end;

		end;
    end;

	if self.spec_realDirtColor ~= nil and self.loadRealDirtColorTracks ~= nil then
		self:loadRealDirtColorTracks(savegame);
	end;

end;

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
------------------------------------------------------------------------------Save Current Dirt Color-------------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
function realDirtColor:saveToXMLFile(xmlFile, key)
	if self.spec_realDirtColor ~= nil then
		local spec = self.spec_realDirtColor;
		if spec.bodypart ~= nil and #spec.bodypart > 0 and getHasShaderParameter(spec.bodypart[1] , "dirtColor") then
			local colorRed, colorGreen, colorBlue, _ = getShaderParameter(spec.bodypart[1], "dirtColor");
			setXMLFloat(xmlFile.handle, key..".body#r", colorRed);
			setXMLFloat(xmlFile.handle, key..".body#g", colorGreen);
			setXMLFloat(xmlFile.handle, key..".body#b", colorBlue);
			setXMLFloat(xmlFile.handle,  key..".body#curIndex", spec.currentDirtColor);
			setXMLFloat(xmlFile.handle,  key..".body#prevIndex", spec.previousDirtColor);
		end;

		if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels ~= 0 then
			for i, wheel in pairs(self.spec_wheels.wheels) do
				if wheel.rDC ~= nil and wheel.rDC and wheel.bodypart ~= nil and getHasShaderParameter(wheel.bodypart[1] , "dirtColor") then
					local wheelColorRed, wheelColorGreen, wheelColorBlue, _  = getShaderParameter(wheel.bodypart[1], "dirtColor");
					setXMLFloat(xmlFile.handle, key..".wheel"..i.."#r", wheelColorRed);
					setXMLFloat(xmlFile.handle, key..".wheel"..i.."#g", wheelColorGreen);
					setXMLFloat(xmlFile.handle, key..".wheel"..i.."#b", wheelColorBlue);
					setXMLFloat(xmlFile.handle,  key..".wheel"..i.."#curIndex", wheel.currentDirtColor);
					setXMLFloat(xmlFile.handle,  key..".wheel"..i.."#prevIndex", wheel.previousDirtColor);
				end;
			end;
		end;

		if self.saveRealDirtColorTracks ~= nil then
			self:saveRealDirtColorTracks(xmlFile, key);
		end;
	end;
end;

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------Update Dirt Colors----------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
function realDirtColor:onUpdate(dt)
	local lastSpeedClamped = self:getLastSpeed();
	if lastSpeedClamped >= realDirtColor.minVehicleSpeed then
		local spec = self.spec_realDirtColor;
		if spec ~= nil and #spec.bodypart > 0 then
			lastSpeedClamped = MathUtil.clamp(lastSpeedClamped, 1.0, realDirtColor.maxSpeedChange);
			local zAdjust = 0;
			if spec.detectPointAdj ~= nil then
				zAdjust = spec.detectPointAdj;
			end;
			local x,y,z = getWorldTranslation(self.components[1].node);
			if zAdjust ~= 0 then
				local x,y,z = localToWorld(self.components[1].node, 0, 0, zAdjust);
			end;
			local yC = Utils.getNoNil(getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode,x,y,z),y);
			local x0,y0,z0 = localToWorld(self.components[1].node, spec.rdWidth, 0, zAdjust);
			local x1,y1,z1 = localToWorld(self.components[1].node, -spec.rdWidth, 0, zAdjust);
			y0 = Utils.getNoNil(getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode,x0,y0,z0),y0);
			y1 = Utils.getNoNil(getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode,x1,y1,z1),y1);
			local curDirtCol = {}
			curDirtCol.r, curDirtCol.g, curDirtCol.b, _  = getShaderParameter(spec.bodypart[1], "dirtColor");

			local function getIsInWater(y)
				local waterY = g_currentMission.waterY;
				if g_currentMission.waterBodyFX ~= nil then waterY = g_currentMission.waterBodyFX.waterY; end;
				if waterY ~= nil and y < waterY then return true; end;
				return false;
			end;
			local rainScale = g_currentMission.environment.weather:getRainFallScale(true);
			local rainWeather = g_currentMission.environment.weather:getIsRaining();
			local temperature = g_currentMission.environment.weather:getCurrentTemperature();
			rainWeather = rainWeather and rainScale > 0;
			local wetfield = g_currentMission.mudFieldSinkUpdater ~= nil and (g_currentMission.mudFieldSinkUpdater.fieldSinkAmount > 0);
			if not rainWeather and wetfield then rainWeather = true; end;
			local wetGround = getIsInWater(yC);
			local colorSpeed = realDirtColor.speeds.body;
		----------------------------------RAIN/WET GROUND DETECTION-----------------------------------
			if (wetGround or rainWeather) and (spec.newDirtColor == 0.1 or spec.currentDirtColor == 0.1) then -- Change Dirt color to "Mud color" depending on ground moisture level.
				spec.newDirtColor = 0.5;
				if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "mud color enabled", false); end;
			elseif (not wetGround and not rainWeather) and (spec.newDirtColor == 0.5 or spec.currentDirtColor == 0.5) then -- Change Dirt color back to "Dry Dirt" color.
				spec.newDirtColor = 0.1;
				--if realDirtColor.debugging then print("mud color disabled"); end;
			end;
		------------------------------------------------------------------------------------------------------------------------------------------------
		---------------------------------------------------------------BodyParts------------------------------------------------------------------------
		------------------------------------------------------------------------------------------------------------------------------------------------
			spec.previousDirtColor, spec.currentDirtColor, spec.targetRDColor = self:getNewColorIndex(spec.previousDirtColor, spec.currentDirtColor, spec.newDirtColor, spec.targetRDColor);
			colorSpeed = self:adjustColorSpeed(spec.currentDirtColor, spec.previousDirtColor, colorSpeed, temperature, lastSpeedClamped);

			if not self:thresholdCompareColor(curDirtCol, spec.targetRDColor, 0.0001) then --(curDirtCol.r ~= spec.targetRDColor.r) and (curDirtCol.g ~= spec.targetRDColor.g) and (curDirtCol.b ~= spec.targetRDColor.b) then
				if spec.newDirtColor > 0.0 then
					curDirtCol.r, curDirtCol.g, curDirtCol.b = self:colorChangeLerp(curDirtCol, spec.targetRDColor, colorSpeed);
					for partIndex, part in pairs(spec.bodypart) do
						if part ~= nil then
							setShaderParameter(part, "dirtColor", curDirtCol.r, curDirtCol.g, curDirtCol.b, 0, false);
						else
							table.remove(spec.bodypart, partIndex);
						end;
					end;
				end;
			end;
			-----------------------------------Effects-----------------------------------------------------------------------------------------------------------------------
			if self.updateTireEffectColors ~= nil then
				self:updateTireEffectColors(nil, curDirtCol, colorSpeed, dt);
			end;
			-----------------------------------------------------------------------------------------------------------------------------------------------------------------

			if not spec.useSimpleMode then
			------------------------------------------------------------------------------------------------------------------------------------------------
			-----------------------------------------------------------------WHEELS-------------------------------------------------------------------------
			------------------------------------------------------------------------------------------------------------------------------------------------	
				if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels > 0 then
					for i, wheel in pairs(self.spec_wheels.wheels) do
						if wheel.rDC ~= nil and wheel.rDC then
							local wheelCurDirtCol = {};
							if wheel.bodypart ~= nil and getHasShaderParameter(wheel.bodypart[1], "dirtColor") then
								wheelCurDirtCol.r, wheelCurDirtCol.g, wheelCurDirtCol.b, _  = getShaderParameter(wheel.bodypart[1], "dirtColor");

								if wheelCurDirtCol.r <= 0.0 then
									wheelCurDirtCol.r, wheelCurDirtCol.g, wheelCurDirtCol.b = self:findWheelDirtColor(wheel);
								end;

								if realDirtColor.debugging then
									if Input.isKeyPressed(Input.KEY_lctrl) and Input.isKeyPressed(Input.KEY_1) then
										debugColorInfo(self, tostring(" r:"..wheelCurDirtCol.r.." g:"..wheelCurDirtCol.g.." b:"..wheelCurDirtCol.b), true);
									end;
								end;

								local wheelColorSpeed = realDirtColor.speeds.wheels;

								if (wetGround or rainWeather) and (wheel.newDirtColor == 0.1 or wheel.currentDirtColor == 0.1) then -- Change Dirt color to "Mud color" depending on ground moisture level.
									wheel.newDirtColor = 0.5;
									if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "mud color enabled", true); end;
									wheelColorSpeed = wheelColorSpeed * realDirtColor.speeds.boost; --Dirt to Mud Speed Boost.
								elseif (not wetGround and not rainWeather) and (wheel.newDirtColor == 0.5 or wheel.currentDirtColor == 0.5) then -- Change Dirt color back to "Dry Dirt" color.
									wheel.newDirtColor = 0.1;
									--if realDirtColor.debugging then print("mud color disabled"); end;
								end;
								
								wheel.previousDirtColor, wheel.currentDirtColor, wheel.targetRDColor = self:getNewColorIndex(wheel.previousDirtColor, wheel.currentDirtColor, wheel.newDirtColor, wheel.targetRDColor);
								wheelColorSpeed = self:adjustColorSpeed(wheel.currentDirtColor, wheel.previousDirtColor, wheelColorSpeed, temperature, lastSpeedClamped);

								if not self:thresholdCompareColor(wheelCurDirtCol, wheel.targetRDColor, 0.0001) then
									if wheel.newDirtColor > 0.0 then
										wheelCurDirtCol.r, wheelCurDirtCol.g, wheelCurDirtCol.b = self:colorChangeLerp(wheelCurDirtCol, wheel.targetRDColor, wheelColorSpeed);
										for _, part in pairs(wheel.bodypart) do
											setShaderParameter(part, "dirtColor", wheelCurDirtCol.r, wheelCurDirtCol.g, wheelCurDirtCol.b, 0, false);
										end;
									end;
								end;
								-----------------------------------Effects-----------------------------------------------------------------------------------------------------------------------
								if self.updateTireEffectColors ~= nil then
									self:updateTireEffectColors(wheel, wheelCurDirtCol, wheelColorSpeed, dt);
								end;
								-----------------------------------------------------------------------------------------------------------------------------------------------------------------
							end;
						end;
					end;
				end;
			end;
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
------------------------------------------------------------------------------Terrain Detection for Color---------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

		---------------------------------------------------------------------------------------------
		-------------------------------------BODYPARTS-----------------------------------------------
		---------------------------------------------------------------------------------------------
			if realDirtColor.debugging then
				local r = math.random(0.00,1.00);
				local g = math.random(0.00,1.00);
				local b = math.random(0.00,1.00);
				if b == r or b == g then
					b = math.random(0.00,1.00);
				end;
				if g == b or g == r then
					g = math.random(0.00,1.00);
				end;
				drawDebugLine(x0,y0,z0, r,g,b, x1,y1,z1, r,g,b);--RDC Rainbow Line
			end;
	--------------------------------------FILL-TYPES------(FillPlane Detection)-------------------------------------------------------
			local foundFillType = self:getFillTypeColorIndex(spec, x0, y0, z0, x1, y1, z1);
			spec.isOnFillTypeRDC = foundFillType;
	--------------------------------------FRUIT-TYPES------(Surface Texture Detection)-------------------------------------------------
			if not foundFillType then
				local sprayedFruitColor = 0;
				local dirtDetected = false;
				sprayedFruitColor = self:getFruitTypeColorIndex(x0,z0,x1,z1, sprayedFruitColor);
	--------------------------------------SPRAY-TYPES------(Surface Texture Detection & General Field Detection)-----------------------
				dirtDetected, sprayedFruitColor = self:getSprayTypeColorIndex(x,yC,z, x0,z0,x1,z1, sprayedFruitColor);
	---------------------------------------------------Texture Detection---------------------------------------------------------------
				--dirtDetected = self:getTextureColorIndex(spec, x,yC,z, sprayedFruitColor, dirtDetected, false);
	-----------------------------------------------------------------------------------------------------------------------------------
				if sprayedFruitColor > 0 and spec.newDirtColor ~= sprayedFruitColor then
					spec.newDirtColor = sprayedFruitColor;  
				elseif sprayedFruitColor == 0 and dirtDetected and spec.newDirtColor ~= 0.1 then
					spec.newDirtColor = 0.1;
				elseif spec.newDirtColor ~= 0.0 then
					spec.newDirtColor = 0.0; -- Stay the same color
				end;
			end;
	-----------------------------------------------------------------------------------------------------------------------------------	
			if not spec.useSimpleMode then
			---------------------------------------------------------------------------------------------
			-----------------------------Terrain Detection for Color-------------------------------------
			--------------------------------------WHEELS-------------------------------------------------
			---------------------------------------------------------------------------------------------
				if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels > 0 then
					for i, wheel in pairs(self.spec_wheels.wheels) do
						if wheel.rDC ~= nil and wheel.rDC then
							local hasGroundContact = wheel.hasGroundContact;
							if self.isServer and hasGroundContact ~= nil and not hasGroundContact then
								if realDirtColor.debugging then realDirtColor:debugColorInfo(self, tostring("Wheel "..i.." has no ground contact"), true); end;
								wheel.newDirtColor = 0.0;
								return;
							end;
							local wheelX, wheelY, wheelZ = getWorldTranslation(wheel.driveNode);
							local wheelx0, wheely0, wheelz0 = 0, 0, 0;
							local wheelx1, wheely1, wheelz1 = 0, 0, 0;
							if wheel.additionalWheels ~= nil and #wheel.additionalWheels > 0 then -- increase wheel detecting area.
								local wheelXOffset = 0;
								local additionalOffset = wheel.additionalWheels[1].offset * #wheel.additionalWheels;
								local additionalWheelWidth = 0;
								for _, additionalWheel in pairs(wheel.additionalWheels) do
									additionalWheelWidth = additionalWheelWidth + additionalWheel.width;
								end;
								local newWheelWidth = wheel.width + additionalWheelWidth;
								local newWheelHalfWidth = newWheelWidth / 2;
								if wheel.isLeft then
									wheelXOffset = newWheelWidth;
									wheelXOffset = -(newWheelHalfWidth+wheel.width*0.5) + wheelXOffset;
									wheelx0, wheely0, wheelz0 = localToWorld(wheel.driveNode, wheelXOffset+(newWheelHalfWidth+(additionalOffset/2)), 0, 0);
									wheelx1, wheely1, wheelz1 = localToWorld(wheel.driveNode, wheelXOffset-newWheelHalfWidth, 0, 0);
								else
									wheelXOffset = -newWheelWidth;
									wheelXOffset = (newWheelHalfWidth+wheel.width*0.5) + wheelXOffset;
									wheelx0, wheely0, wheelz0 = localToWorld(wheel.driveNode, wheelXOffset+newWheelHalfWidth, 0, 0);
									wheelx1, wheely1, wheelz1 = localToWorld(wheel.driveNode, wheelXOffset-(newWheelHalfWidth+(additionalOffset/2)), 0, 0);
								end;
							else
								wheelx0, wheely0, wheelz0 = localToWorld(wheel.driveNode, wheel.width/2, 0, 0);
								wheelx1, wheely1, wheelz1 = localToWorld(wheel.driveNode, -(wheel.width/2), 0, 0);
							end;
							local wheelRadius = wheel.radius-0.001;
							local wheelyC = Utils.getNoNil(getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode,wheelX, wheelY-wheelRadius, wheelZ),wheelY);
							local wheely0 = Utils.getNoNil(getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode,wheelx0, wheely0-wheelRadius, wheelz0),wheely0);
							local wheely1 = Utils.getNoNil(getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode,wheelx1, wheely1-wheelRadius, wheelz1),wheely1);
							wheel.rDC_x,wheel.rDC_y,wheel.rDC_z = wheelX, wheelyC, wheelZ;
							if realDirtColor.debugging then
								if i == 1 then
									drawDebugLine(wheelx0,wheely0,wheelz0, 0,0,1, wheelx1,wheely1,wheelz1, 0,0,1);
								elseif i == 2 then
									drawDebugLine(wheelx0,wheely0,wheelz0, 0,1,0, wheelx1,wheely1,wheelz1, 0,1,0);
								elseif i == 3 then
									drawDebugLine(wheelx0,wheely0,wheelz0, 1,0,0, wheelx1,wheely1,wheelz1, 1,0,0);
								else
									drawDebugLine(wheelx0,wheely0,wheelz0, 1,0,1, wheelx1,wheely1,wheelz1, 1,0,1);
								end;
							end;
						--------------------------------------FILL-TYPES------(FillPlane Detection)-------------------------------------------------------
							local foundFillType = self:getFillTypeColorIndex(wheel, wheelx0, wheely0, wheelz0, wheelx1, wheely1, wheelz1, true);
							wheel.isOnFillTypeRDC = foundFillType;
					--------------------------------------FRUIT-TYPES------(Surface Texture Detection)-------------------------------------------------
							if not foundFillType then
								local wheelSprayedFruitColor = 0;
								local wheeldirtDetected = false;
								wheelSprayedFruitColor = self:getFruitTypeColorIndex(wheelx0, wheelz0, wheelx1, wheelz1, wheelSprayedFruitColor, true);
					--------------------------------------SPRAY-TYPES------(Surface Texture Detection & General Field Detection)-----------------------
								wheeldirtDetected, wheelSprayedFruitColor = self:getSprayTypeColorIndex(wheelX, wheelyC, wheelZ, wheelx0, wheelz0, wheelx1, wheelz1, wheelSprayedFruitColor, true);
					---------------------------------------------------Texture Detection---------------------------------------------------------------
								--wheeldirtDetected = self:getTextureColorIndex(wheel, wheelX, wheelyC, wheelZ, sprayedFruitColor, wheeldirtDetected, true);
					-----------------------------------------------------------------------------------------------------------------------------------
								if wheelSprayedFruitColor > 0 and wheel.newDirtColor ~= wheelSprayedFruitColor  then
									wheel.newDirtColor = wheelSprayedFruitColor; 
								elseif wheelSprayedFruitColor == 0 and wheeldirtDetected and wheel.newDirtColor ~= 0.1 then
									wheel.newDirtColor = 0.1;
								elseif wheel.newDirtColor ~= 0.0 then
									wheel.newDirtColor = 0.0; -- Stay the same color
									--if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "wheel.newDirtColor = 0", true); end;
								end;
					-----------------------------------------------------------------------------------------------------------------------------------	
							end;
						end;
					end;
				end;
			end;
		end;
	end;
end;

function realDirtColor:getFillTypeColorIndex(source, x1,y1,z1,x2,y2,z2, isWheel)
	local foundFillType = DensityMapHeightUtil.getFillTypeAtLine(x1,y1,z1,x2,y2,z2,1.0);
	if foundFillType == FillType.SNOW then
		if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found SNOW", isWheel); end;
		if source.newDirtColor ~= 2 then
			source.newDirtColor = 2;
		end;
		return true;
	elseif foundFillType == FillType.MANURE then
		if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found MANURE", isWheel); end;
		if source.newDirtColor ~= 4 then
			source.newDirtColor = 4;
		end;
		return true;
	elseif foundFillType == FillType.GRASS_WINDROW then
		if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found GRASS", isWheel); end;
		if source.newDirtColor ~= 1 then
			source.newDirtColor = 1;
		end;
		return true;
	elseif foundFillType == FillType.DRYGRASS_WINDROW then
		if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found DRYGRASS", isWheel); end;
		if source.newDirtColor ~= 1 then
			source.newDirtColor = 1;
		end;
		return true;
	elseif foundFillType == FillType.CHAFF then
		if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found CHAFF", isWheel); end;
		if source.newDirtColor ~= 1.5 then
			source.newDirtColor = 1.5;
		end;
		return true;
	elseif foundFillType == FillType.LIME then
		if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found LIME", isWheel); end;
		if source.newDirtColor ~= 3 then
			source.newDirtColor = 3;
		end;
		return true;
	elseif foundFillType == FillType.MUD then
		if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found MUD", isWheel); end;
		if source.newDirtColor ~= 0.55 then
			source.newDirtColor = 0.55;
		end;
		return true;
	end;

	return false;
end;

function realDirtColor:getFruitTypeColorIndex(x1,z1,x2,z2, sprayedFruitColor, isWheel)
	local groundTypeMapId, groundTypeFirstChannel, groundTypeNumChannels = g_currentMission.fieldGroundSystem:getDensityMapData(FieldDensityMap.GROUND_TYPE);
	local fruitCheck = g_currentMission.fieldCropsQuery; --FieldCropsQuery.new(groundTypeMapId)
	local grassDesc = g_fruitTypeManager:getFruitTypeByIndex(FruitType.GRASS);
	-----------------------------------------------------------------------------
	if grassDesc ~= nil then
		local xl, zl, widthX, widthZ, heightX, heightZ = MathUtil.getXZWidthAndHeight(x1, z1, x2, z2, x1, z1);
		fruitCheck:addRequiredCropType(grassDesc.terrainDataPlaneId, grassDesc.minHarvestingGrowthState+1, grassDesc.maxHarvestingGrowthState+1, grassDesc.startStateChannel, grassDesc.numStateChannels, groundTypeFirstChannel, groundTypeNumChannels);
		local grassArea, _ = fruitCheck:getParallelogram(xl,zl, widthX,widthZ, heightX,heightZ, true);
		if grassArea > 0 and sprayedFruitColor ~= 1 then
			if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found Grass Surface", isWheel); end;
			sprayedFruitColor = 1;
		elseif sprayedFruitColor ~= 0.0 then
			sprayedFruitColor = 0.0;
		end;
	end;

	return sprayedFruitColor;
end;

function realDirtColor:getSprayTypeColorIndex(x,y,z, x1,z1,x2,z2, sprayedFruitColor, isWheel)
	local dirtDetected = false;
	if g_currentMission.terrainDetailId ~= nil then
		local bits = getDensityAtWorldPos(g_currentMission.terrainDetailId, x, y, z);
		if bits ~= 0 then--is on a field
			dirtDetected = true;
			if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found Field", isWheel); end;
			for sprayType, filltype in pairs({FillType.LIME, FillType.MANURE, FillType.LIQUIDMANURE, FillType.DIGESTATE}) do
				local sprayTypeDesc = g_sprayTypeManager:getSprayTypeByFillTypeIndex(filltype);
				if sprayTypeDesc ~= nil then
					local sprayCheck = g_currentMission.fieldCropsQuery;
					local xl, zl, widthX, widthZ, heightX, heightZ = MathUtil.getXZWidthAndHeight(x1, z1, x2, z2, x1, z1);
					local _, sprayTypeFirstChannel, sprayTypeNumChannels = g_currentMission.fieldGroundSystem:getDensityMapData(FieldDensityMap.SPRAY_TYPE);
					sprayCheck:addRequiredGroundValue(sprayTypeDesc.sprayGroundType, sprayTypeDesc.sprayGroundType, sprayTypeFirstChannel, sprayTypeNumChannels);
					local totalArea, _ = sprayCheck:getParallelogram(xl,zl, widthX,widthZ, heightX,heightZ, true);

					if totalArea > 0 then
						if sprayType == 1 and sprayedFruitColor ~= 3 then
							if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found Lime Surface", isWheel); end;
							sprayedFruitColor = 3;
						elseif sprayType == 2 and sprayedFruitColor ~= 4 then
							if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found Manure Surface", isWheel); end;
							sprayedFruitColor = 4;
						elseif sprayType >= 3 and sprayedFruitColor ~= 5 then
							if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found Slurry Surface ", isWheel); end;
							sprayedFruitColor = 5;
						end;
					end;
				end;
			end;
		end;
	end;

	return dirtDetected, sprayedFruitColor;
end;

function realDirtColor:getTextureColorIndex(source, x,y,z, sprayedFruitColor, dirtDetected, isWheel)
	if (sprayedFruitColor == 0.0) and (dirtDetected == false) then
		local cR,cG,cB,t,_ = getTerrainAttributesAtWorldPos(g_currentMission.terrainRootNode, x, y, z, true, true, true, true, false);
		if self.spec_realDirtColor.dryDirtColor == 0 then
			if (cR > 0.10400 and cR < 0.10401) and (cG > 0.08399 and cG < 0.0840) and (cB > 0.06100 and cB < 0.06101) then -- Dirt Terrain
				dirtDetected = true;
				if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found Dirt Texture Surface", isWheel); end;
			end;
		else
			if (cR > 0.36 and cR < 0.36001) and (cG > 0.12499 and cG < 0.12501) and (cB > 0.056 and cB < 0.05601) then --redDirt Terrain
				dirtDetected = true;
				if realDirtColor.debugging then realDirtColor:debugColorInfo(self, "Found Dirt Texture Surface", isWheel); end;
			end;
		end;
		--[[if realDirtColor.debugging then realDirtColor:debugColorInfo(self, tostring("R: "..cR.." | G: "..cG.." | B: "..cB.." |"), isWheel); end;
		if ((cR > 0.4 ) and (cG > 0.4) and (cB > 0.4)) and source.onRoadSurface == nil then -- Road Surface
			source.onRoadSurface = true;
		elseif source.onRoadSurface ~= nil then
			source.onRoadSurface = nil;
		end;]]
	end;

	return dirtDetected;
end;

function realDirtColor:getNewColorIndex(previousDirtColor, currentDirtColor, newDirtColor, targetRDColor)

	local rDColors = realDirtColor.colors;

	if newDirtColor == 5  and currentDirtColor ~= 5 then --Slurry
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.slurry.r, rDColors.slurry.g, rDColors.slurry.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 5;
	elseif newDirtColor == 4 and currentDirtColor ~= 4 then --Manure
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.manure.r, rDColors.manure.g, rDColors.manure.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 4;
	elseif newDirtColor == 3 and currentDirtColor ~= 3 then --Lime
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.lime.r, rDColors.lime.g, rDColors.lime.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 3;
	elseif newDirtColor == 2 and currentDirtColor ~= 2 then --Snow
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.snow.r, rDColors.snow.g, rDColors.snow.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 2;
	elseif newDirtColor == 1.5 and currentDirtColor ~= 1.5 then -- Chaff
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.chaff.r, rDColors.chaff.g, rDColors.chaff.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 1.5;
	elseif newDirtColor == 1 and currentDirtColor ~= 1 then --Grass
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.grass.r, rDColors.grass.g, rDColors.grass.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 1;
	elseif newDirtColor == 0.5 and currentDirtColor ~= 0.5 then -- Mud (Field)
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.mud.r, rDColors.mud.g, rDColors.mud.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 0.5;
	elseif newDirtColor == 0.55 and currentDirtColor ~= 0.55 then -- Mud (FillType)
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.mud.r, rDColors.mud.g, rDColors.mud.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 0.55;
	elseif newDirtColor == 0.1 and currentDirtColor ~= 0.1 then -- Dirt
		targetRDColor.r, targetRDColor.g, targetRDColor.b = rDColors.dryDirt.r, rDColors.dryDirt.g, rDColors.dryDirt.b;
		previousDirtColor = currentDirtColor;
		currentDirtColor = 0.1;
	else
		newDirtColor = 0;
	end;
	
	--[[if realDirtColor.debugging then
		realDirtColor:debugTargetColor(currentDirtColor, targetRDColor);
	end;]]

	return previousDirtColor, currentDirtColor, targetRDColor;
end;

function realDirtColor:debugTargetColor(currentDirtColor, targetRDColor)
	local rDColors = realDirtColor.colors;
	if currentDirtColor == 5  and (targetRDColor.r ~= rDColors.slurry.r and targetRDColor.g ~= rDColors.slurry.g and targetRDColor.b ~= rDColors.slurry.b) then --Slurry
		Logging.error("Wrong TargetColor @Slurry");
		Logging.error(tostring("R: "..targetRDColor.r.." | G: "..targetRDColor.g.." | B: "..targetRDColor.b));
	elseif currentDirtColor == 4 and (targetRDColor.r ~= rDColors.manure.r and targetRDColor.g ~= rDColors.manure.g and targetRDColor.b ~= rDColors.manure.b) then --Manure
		Logging.error("Wrong TargetColor @Manure");
		Logging.error(tostring("R: "..targetRDColor.r.." | G: "..targetRDColor.g.." | B: "..targetRDColor.b));
	elseif currentDirtColor == 3 and (targetRDColor.r ~= rDColors.lime.r and targetRDColor.g ~= rDColors.lime.g and targetRDColor.b ~= rDColors.lime.b) then --Lime
		Logging.error("Wrong TargetColor @Lime");
		Logging.error(tostring("R: "..targetRDColor.r.." | G: "..targetRDColor.g.." | B: "..targetRDColor.b));
	elseif currentDirtColor == 2 and (targetRDColor.r ~= rDColors.snow.r and targetRDColor.g ~= rDColors.snow.g and targetRDColor.b ~= rDColors.snow.b) then --Snow
		Logging.error("Wrong TargetColor @Snow");
		Logging.error(tostring("R: "..targetRDColor.r.." | G: "..targetRDColor.g.." | B: "..targetRDColor.b));
	elseif currentDirtColor == 1.5 and (targetRDColor.r ~= rDColors.chaff.r and targetRDColor.g ~= rDColors.chaff.g and targetRDColor.b ~= rDColors.chaff.b) then -- Chaff
		Logging.error("Wrong TargetColor @Chaff");
		Logging.error(tostring("R: "..targetRDColor.r.." | G: "..targetRDColor.g.." | B: "..targetRDColor.b));
	elseif currentDirtColor == 1 and (targetRDColor.r ~= rDColors.grass.r and targetRDColor.g ~= rDColors.grass.g and targetRDColor.b ~= rDColors.grass.b) then --Grass
		Logging.error("Wrong TargetColor @Grass");
		Logging.error(tostring("R: "..targetRDColor.r.." | G: "..targetRDColor.g.." | B: "..targetRDColor.b));
	elseif (currentDirtColor == 0.5 or currentDirtColor == 0.55) and (targetRDColor.r ~= rDColors.mud.r and targetRDColor.g ~= rDColors.mud.g and targetRDColor.b ~= rDColors.mud.b) then -- Mud (Field)
		Logging.error("Wrong TargetColor @Mud1");
		Logging.error(tostring("R: "..targetRDColor.r.." | G: "..targetRDColor.g.." | B: "..targetRDColor.b));
	elseif currentDirtColor == 0.1 and (targetRDColor.r ~= rDColors.dryDirt.r and targetRDColor.g ~= rDColors.dryDirt.g and targetRDColor.b ~= rDColors.dryDirt.b) then -- Dirt
		Logging.error("Wrong TargetColor @Dirt");
		Logging.error(tostring("R: "..targetRDColor.r.." | G: "..targetRDColor.g.." | B: "..targetRDColor.b));
	end;
end;

function realDirtColor:colorChangeLerp(CurrentRDCol, TargetRDColor, colorSpeed)
	CurrentRDCol.r, CurrentRDCol.g, CurrentRDCol.b = MathUtil.vector3Lerp(CurrentRDCol.r, CurrentRDCol.g, CurrentRDCol.b, TargetRDColor.r, TargetRDColor.g, TargetRDColor.b, colorSpeed);

	return CurrentRDCol.r, CurrentRDCol.g, CurrentRDCol.b;
end;

function realDirtColor:adjustColorSpeed(currentDirtColor, previousDirtColor, colorSpeed, temp, lastSpeedClamped)
	--<<:COLORKEY:>>  0.1 = Dirt, 0.5/0.55 = Mud, 1 = Grass, 1.5 = Chaff, 2 = Snow, 3 = Lime, 4 = Manure, 5 = Slurry/Digestate
	colorSpeed = colorSpeed / realDirtColor.maxSpeedChange;
	colorSpeed = colorSpeed * lastSpeedClamped; --maxSpeedChange value required for full colorSpeed

	if currentDirtColor == 5 then
		colorSpeed = colorSpeed * realDirtColor.speeds.megaBoost;
	elseif currentDirtColor == 2 or currentDirtColor == 3 then--white
		colorSpeed = colorSpeed / realDirtColor.speeds.white;
	elseif previousDirtColor == 2 and (temp > 5 and g_currentMission.justSnow == nil) then--snow
		colorSpeed = colorSpeed * realDirtColor.speeds.megaBoost;
	elseif currentDirtColor == 1.5 or currentDirtColor == 1 then --grass/chaff
		colorSpeed = colorSpeed / realDirtColor.speeds.white;
	elseif currentDirtColor == 0.55 or currentDirtColor == 0.5 then
		colorSpeed = colorSpeed * realDirtColor.speeds.boost;
	elseif currentDirtColor == 0.1 and previousDirtColor == 0.55 or previousDirtColor == 0.5 then
		colorSpeed = colorSpeed / realDirtColor.speeds.revertFast;
	elseif currentDirtColor == 0.1 and previousDirtColor ~= 3 then
		colorSpeed = colorSpeed / realDirtColor.speeds.revert;
	end;

	if realDirtColor.coldSlowsColor then
		if temp <= 0 and currentDirtColor ~= 2 then
			colorSpeed = colorSpeed / 2;
		end;
	end;

	return colorSpeed;
end;

function realDirtColor:addPart(table, partToadd)
--"table" can be "self" or a wheel
	if table.bodypart ~= nil then
		table.bodypart[#table.bodypart+1] = partToAdd;
	end;
end;

function realDirtColor:removePart(self, partToRemove)

	if self ~= nil and partToRemove ~= nil then
		local spec = self.spec_realDirtColor;
		if spec.bodypart ~= nil then
			for partIndex, part in pairs(spec.bodypart) do
				if part == partToRemove then
					table.remove(spec.bodypart, partIndex);
					if realDirtColor.debugging then
						print(realDirtColor.title..": Part succesfully removed");
					end;
					return true;
				end;
			end;
		end;
		if self.spec_wheels.wheels ~= nil then
			for i, wheel in pairs(self.spec_wheels.wheels) do
				if wheel.bodypart ~= nil then
					for partIndex, wpart in pairs(wheel.bodypart) do
						if wpart == partToRemove then
							table.remove(wheel.bodypart, partIndex);
							if realDirtColor.debugging then
								print(realDirtColor.title..": Wheel part succesfully removed");
							end;
							return true;
						end;
					end;
				end;
			end;	
		end;
	else
		if realDirtColor.debugging then
			Logging.warning(realDirtColor.title..": Wheel part unsuccesfully removed; Unable to find part.");
		end;
		return false;
	end;

end;

function realDirtColor:findWheelDirtColor(wheel)

	local rValue, gValue, bValue = 0,0,0;

	if wheel["wheelOuterRim"] ~= nil and getHasShaderParameter(wheel["wheelOuterRim"], "dirtColor") then
		rValue, gValue, bValue, _  = getShaderParameter(wheel["wheelOuterRim"], "dirtColor");
		if rValue > 0.0 then
			--print("1");
			return rValue, gValue, bValue;
		end;
	end;

	if rValue <= 0.0 and wheel["wheelInnerRim"] ~= nil and getHasShaderParameter(wheel["wheelInnerRim"], "dirtColor") then
		rValue, gValue, bValue, _  = getShaderParameter(wheel["wheelInnerRim"], "dirtColor");
		if rValue > 0.0 then
			--print("2");
			return rValue, gValue, bValue;
		end;
	end;

	if rValue <= 0.0 and  wheel["wheelAdditional"] ~= nil and getHasShaderParameter(wheel["wheelAdditional"], "dirtColor") then
		rValue, gValue, bValue, _  = getShaderParameter(wheel["wheelAdditional"], "dirtColor");
		if rValue > 0.0 then
			--print("3");
			return rValue, gValue, bValue;
		end;
	end;

	if rValue <= 0.0 and wheel.bodypart ~= nil then
		for _, part in pairs(wheel.bodypart) do
			rValue, gValue, bValue, _  = getShaderParameter(part, "dirtColor");
			if rValue > 0.0 then
				--print("4");
				return rValue, gValue, bValue;
			end;
		end;
	end;
	Logging.warning(realDirtColor.title..": Cannot find dirt color from wheel");
end;

function realDirtColor:setNodeDirtColor(superFunc, ...)
	-- nil
end;

function realDirtColor:onReadStream(streamId, connection)
	if connection.isServer and self.spec_realDirtColor ~= nil then
		local spec = self.spec_realDirtColor;
		spec.currentDirtColor = streamReadInt8(streamId);
		spec.previousDirtColor = streamReadInt8(streamId);
		local _,_,targetRDColor = self:getNewColorIndex(0, 0, spec.currentDirtColor, spec.targetRDColor);
		spec.targetRDColor = targetRDColor;
		local r,g,b;
		r = streamReadFloat32(streamId);
		g = streamReadFloat32(streamId);
		b = streamReadFloat32(streamId);
		for _, part in pairs(spec.bodypart) do
			setShaderParameter(part, "dirtColor", r, g, b, 0, false);
		end;
		if not spec.useSimpleMode then
			if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels ~= 0 then
				for i, wheel in pairs(self.spec_wheels.wheels) do
					if wheel.rDC ~= nil and wheel.rDC then
						wheel.currentDirtColor = streamReadInt8(streamId);
						wheel.previousDirtColor = streamReadInt8(streamId);
						local _,_,targetRDColor = self:getNewColorIndex(0, 0, wheel.currentDirtColor, wheel.targetRDColor);
						wheel.targetRDColor = targetRDColor;
						if wheel.bodypart ~= nil then
							r = streamReadFloat32(streamId);
							g = streamReadFloat32(streamId);
							b = streamReadFloat32(streamId);
							for _, part in pairs(wheel.bodypart) do
								setShaderParameter(part, "dirtColor", r, g, b, 0, false);
							end;
						end;
					end;
				end;
			end;
		end;
	end;
end;

function realDirtColor:onWriteStream(streamId, connection)
	if not connection.isServer and self.spec_realDirtColor ~= nil then
		local spec = self.spec_realDirtColor;
		streamWriteInt8(streamId, spec.currentDirtColor);
		streamWriteInt8(streamId, spec.previousDirtColor);
		local r,g,b = getShaderParameter(spec.bodypart[1], "dirtColor");
		streamWriteFloat32(streamId, r);
		streamWriteFloat32(streamId, g);
		streamWriteFloat32(streamId, b);
		if not spec.useSimpleMode then
			if self.spec_wheels ~= nil and self.spec_wheels.wheels ~= nil and #self.spec_wheels.wheels ~= 0 then
				for i, wheel in pairs(self.spec_wheels.wheels) do
					if wheel.rDC ~= nil and wheel.rDC then
						streamWriteInt8(streamId, wheel.currentDirtColor);
						streamWriteInt8(streamId, wheel.previousDirtColor);
						if wheel.bodypart ~= nil then
							local r,g,b = getShaderParameter(wheel.bodypart[1], "dirtColor");
							streamWriteFloat32(streamId, r);
							streamWriteFloat32(streamId, g);
							streamWriteFloat32(streamId, b);
						end;
					end;
				end;
			end;
		end;
	end;
end;