--[[
	ManualDischarge.lua
	
	Autor: 		Ifko[nator]
	Datum: 		21.04.2022
	Version: 	2.2
	
	Changelog:	v1.0 @01.04.2020 - initial implementation in FS 19
				---------------------------------------------------------------------------------------------------------------------------------------------
				v1.1 @09.04.2020 - fix for straw harvest addon and script optimation
				---------------------------------------------------------------------------------------------------------------------------------------------
				v1.2 @02.06.2020 - fix for overload vehicles, witch have only one pipe state
				---------------------------------------------------------------------------------------------------------------------------------------------
				v1.3 @27.07.2020 - script optimation
				---------------------------------------------------------------------------------------------------------------------------------------------
				v1.4 @16.08.2020 - fix for error .../FS19_ManualDischarge/ManualDischarge.lua:123: attempt to index local 'specManualDischarge' (a nil value)
				---------------------------------------------------------------------------------------------------------------------------------------------
				v1.5 @10.10.2020 - if an augerwagon is detached, while overloading, the overloading will now be stopped
								 - fix for ccm mill
								 - augerwagons consume now pto power while overloading
				---------------------------------------------------------------------------------------------------------------------------------------------
				v2.0 @18.11.2021 - convert to FS 22
				---------------------------------------------------------------------------------------------------------------------------------------------
				v2.1 @13.12.2021 - a little code optimation
				---------------------------------------------------------------------------------------------------------------------------------------------
				v2.2 @21.04.2022 - fix for patch 1.4 and higher
]]

ManualDischarge = {};

function ManualDischarge.initSpecialization()
	if g_configurationManager.configurations["manualDischarge"] == nil then    
        g_configurationManager:addConfigurationType("manualDischarge", g_i18n:getText("configuration_manualDischarge"), "pipe", nil, nil, nil, ConfigurationUtil.SELECTOR_MULTIOPTION);
    end;
end;

function ManualDischarge.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Pipe, specializations);
end;

function ManualDischarge.registerEventListeners(vehicleType)
	local functionNames = {
		"onPreLoad",
		"onLoad",
		"onRegisterActionEvents",
		"onWriteStream",
		"onReadStream",
		"onUpdate"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, ManualDischarge);
	end;
end;

function ManualDischarge.registerFunctions(vehicleType)
	local newFunctions = {
		"setAllowDischarge",
		"foundObjectToDischarge",
		"objectToDischargeHasFreeCapacity"
	};
	
	for _, newFunction in ipairs(newFunctions) do
		SpecializationUtil.registerFunction(vehicleType, newFunction, ManualDischarge[newFunction]);
	end;
end;

function ManualDischarge.registerOverwrittenFunctions(vehicleType)
    local overwrittenFunctions = {
		"getCanDischargeToObject",
		"getDoConsumePtoPower",
		"getIsPowerTakeOffActive"
	};
	
	for _, overwrittenFunction in ipairs(overwrittenFunctions) do
		SpecializationUtil.registerOverwrittenFunction(vehicleType, overwrittenFunction, ManualDischarge[overwrittenFunction]);
	end;
end;

function ManualDischarge:onPreLoad(savegame)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");

	specManualDischarge.isBought = false;

	local configurationId = self.configurations["manualDischarge"];
	
    if configurationId ~= nil then
		local storeItem = g_storeManager:getItemByXMLFilename(self.configFileName);
		
		if storeItem ~= nil then
            local configuration = storeItem.configurations["manualDischarge"][configurationId];

            if configuration ~= nil then
                specManualDischarge.isBought = configuration.index == 2;
            end;
        end;
    end;
end;

function ManualDischarge:onLoad(savegame)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");
	local specFillUnit = ManualDischargeUtil.getSpecByName(self, "fillUnit");
	local specPowerConsumer = ManualDischargeUtil.getSpecByName(self, "powerConsumer");
	local specTrailer = ManualDischargeUtil.getSpecByName(self, "trailer");
	local specPipe = ManualDischargeUtil.getSpecByName(self, "pipe");

	if specTrailer ~= nil and specPipe ~= nil and specPowerConsumer ~= nil and specPowerConsumer.ptoRpm == 0 then
		--## add pto rpm to augerwagons for overloading
		specPowerConsumer.ptoRpm = 450;
	end;
	
	specManualDischarge.allowDischarge = false;
	specManualDischarge.showTurnOnMotorWarning = false;
	specManualDischarge.showVehicleIsEmptyWarning = false;

	if specManualDischarge.isBought then
		Pipe.onDischargeStateChanged = Utils.overwrittenFunction(Pipe.onDischargeStateChanged, ManualDischarge.onDischargeStateChanged);
	else
		specManualDischarge.allowDischarge = true;
	end;
	
	self:setAllowDischarge(specManualDischarge.allowDischarge);

	specManualDischarge.l10nTexts = {};

	local l10nTexts = {
		"START_DISCHARGE",
		"STOP_DISCHARGE",
		"EMPTY_WARNING",
		"input_DISCHARGE_BUTTON",
		"warning_motorNotStarted"
	};

	for _, l10nText in pairs(l10nTexts) do
		specManualDischarge.l10nTexts[l10nText] = g_i18n:getText(l10nText, ManualDischargeUtil.currentModName);
	end;
end;

function ManualDischarge:getDoConsumePtoPower(superFunc)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");

	if not specManualDischarge.isBought then
		specManualDischarge.allowDischarge = true;
	end;

    return specManualDischarge.allowDischarge and self:getDischargeState() ~= Dischargeable.DISCHARGE_STATE_OFF or self.getIsTurnedOn ~= nil and self:getIsTurnedOn();
end;

function ManualDischarge:getIsPowerTakeOffActive(superFunc)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");

	if not specManualDischarge.isBought then
		specManualDischarge.allowDischarge = true;
	end;

    return specManualDischarge.allowDischarge and self:getDischargeState() ~= Dischargeable.DISCHARGE_STATE_OFF or self.getIsTurnedOn ~= nil and self:getIsTurnedOn();
end;

function ManualDischarge:onDischargeStateChanged(superFunc, state)
    if self.isClient then
		local specPipe = ManualDischargeUtil.getSpecByName(self, "pipe");
		local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");

		if specManualDischarge ~= nil then
        	local dischargeNode = self:getCurrentDischargeNode();
			local dischargeNodeIndex = nil;

        	if dischargeNode ~= nil then
        	    dischargeNodeIndex = dischargeNode.index;
			end;

			if not specManualDischarge.isBought then
				specManualDischarge.allowDischarge = true;
			end;

        	if dischargeNodeIndex == specPipe.dischargeNodeIndex then
        	    if state == Dischargeable.DISCHARGE_STATE_OFF or not specManualDischarge.allowDischarge then
        	        g_animationManager:stopAnimations(specPipe.animationNodes);
        	    else
        	        g_animationManager:startAnimations(specPipe.animationNodes);
        	        g_animationManager:setFillType(specPipe.animationNodes, self:getFillUnitLastValidFillType(dischargeNode.fillUnitIndex));
        	    end;
			end;
		else
			superFunc(self, state);
		end;
    end;
end;

function ManualDischarge:onUpdate(dt, isActiveForInput, isActiveForInputIngnoreSelection, isSelected)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");
	
	if specManualDischarge.isBought then
		local specDischargeable = ManualDischargeUtil.getSpecByName(self, "dischargeable");
		local specCombine = ManualDischargeUtil.getSpecByName(self, "combine");
		local specTrailer = ManualDischargeUtil.getSpecByName(self, "trailer");
		local specPipe = ManualDischargeUtil.getSpecByName(self, "pipe");
		
		local currentDischargeNode = specDischargeable.currentDischargeNode;

		if currentDischargeNode ~= nil then
			local allowTipToGround = self:getCanToggleDischargeToGround() and self:getCanDischargeToGround(currentDischargeNode) and self:getCanDischargeAtPosition(currentDischargeNode);
			local getIsVehicleEmpty = ManualDischargeUtil.getIsVehicleEmpty(self);
			local allowDischarge = true;

			if currentDischargeNode.dischargeObject ~= nil then
				allowDischarge = self:objectToDischargeHasFreeCapacity(currentDischargeNode);
			end;

			local pipeIsUnFolded = specPipe.targetState > 1 or specPipe.numStates == 1;

			if specPipe.animation.name ~= nil then
				pipeIsUnFolded = pipeIsUnFolded and not self:getIsAnimationPlaying(specPipe.animation.name);
			end;

			local pipeCamButton = specManualDischarge.actionEvents[InputAction.PIPE_CAM_BUTTON];	

			if specCombine ~= nil then	
				if self:getIsAIActive() then
					self:setAllowDischarge(not getIsVehicleEmpty and not allowTipToGround and pipeIsUnFolded);
				end;
			
				if self.getIsCourseplayDriving ~= nil then
					if self:getIsCourseplayDriving() then
						self:setAllowDischarge(not getIsVehicleEmpty and not allowTipToGround and pipeIsUnFolded);
					end;
				end;
			end;

			if self.getRootVehicle ~= nil then
				local controlledTractor = self:getRootVehicle();

				if controlledTractor ~= nil then
					if controlledTractor.getIsCourseplayDriving ~= nil then
						if controlledTractor:getIsCourseplayDriving() then
							if specTrailer ~= nil and specPipe ~= nil then	
								self:setAllowDischarge(not getIsVehicleEmpty and not allowTipToGround and pipeIsUnFolded);
							end;
						end;
					end;
				end;
			end;

			if not currentDischargeNode.stopDischargeOnEmpty or self.xmlFile:hasProperty("vehicle.woodCrusher") and not specManualDischarge.allowDischarge then
				--## forage harevester or wood chrusher, no button needed to overload
				
				self:setAllowDischarge(true);
			end;

			if specManualDischarge.allowDischarge then
				if currentDischargeNode.stopDischargeOnEmpty then 
					allowDischarge = allowDischarge and not getIsVehicleEmpty and ManualDischargeUtil.getIsVehicleMotorStarted(self);
					allowTipToGround = allowTipToGround and not getIsVehicleEmpty and ManualDischargeUtil.getIsVehicleMotorStarted(self);
					
					if allowTipToGround and specDischargeable.currentDischargeState == 0 then
						self:setDischargeState(Dischargeable.DISCHARGE_STATE_GROUND);
					else
						self:setAllowDischarge(allowDischarge);
					end;
				end;
			end;

			if self:getIsActiveForInput() then
				--renderText(0.5, 0.5, 0.02, "allowDischarge = " .. tostring(allowDischarge));
				--renderText(0.5, 0.48, 0.02, "specDischargeable.currentDischargeState = " .. tostring(specDischargeable.currentDischargeState));
				--renderText(0.5, 0.46, 0.02, "self:getCanDischargeAtPosition(currentDischargeNode) = " .. tostring(self:getCanDischargeAtPosition(currentDischargeNode)));
				--renderText(0.5, 0.44, 0.02, "pipeIsUnFolded = " .. tostring(pipeIsUnFolded));
				--renderText(0.5, 0.42, 0.02, "self:objectToDischargeHasFreeCapacity(currentDischargeNode) = " .. tostring(self:objectToDischargeHasFreeCapacity(currentDischargeNode)));
			end;

			if currentDischargeNode.stopDischargeOnEmpty then
    			local allowShowButton = allowDischarge and currentDischargeNode.stopDischargeOnEmpty and self:getCanDischargeAtPosition(currentDischargeNode);
				local dischargeButton = specManualDischarge.actionEvents[InputAction.DISCHARGE_BUTTON];
				local actionEventTip = specDischargeable.actionEvents[InputAction.TOGGLE_TIPSTATE];
				local actionEventTipGround = specDischargeable.actionEvents[InputAction.TOGGLE_TIPSTATE_GROUND];
				
				if specPipe ~= nil then
					allowShowButton = allowShowButton and pipeIsUnFolded;
				end;

				if actionEventTip ~= nil then
					g_inputBinding:setActionEventTextVisibility(actionEventTip.actionEventId, false);
					g_inputBinding:setActionEventActive(actionEventTip.actionEventId, false);
				end;

    			if actionEventTipGround ~= nil then
					g_inputBinding:setActionEventTextVisibility(actionEventTipGround.actionEventId, false);
					g_inputBinding:setActionEventActive(actionEventTipGround.actionEventId, false);
    			end;

				if dischargeButton ~= nil then
					local currentText = specManualDischarge.l10nTexts.START_DISCHARGE;

					g_inputBinding:setActionEventTextVisibility(dischargeButton.actionEventId, allowShowButton);
					g_inputBinding:setActionEventActive(dischargeButton.actionEventId, allowShowButton);

					if specManualDischarge.allowDischarge then
						currentText = specManualDischarge.l10nTexts.STOP_DISCHARGE;
					else
						self:setDischargeEffectActive(currentDischargeNode, false, true);
						self:updateDischargeSound(currentDischargeNode, 99999);
					end;

					g_inputBinding:setActionEventText(dischargeButton.actionEventId, currentText);
				end;
			end;
		end;

		if specManualDischarge.showTurnOnMotorWarning then
			g_currentMission:showBlinkingWarning(specManualDischarge.l10nTexts.warning_motorNotStarted, 2000);
			
			specManualDischarge.showTurnOnMotorWarning = false;
		end;

		if specManualDischarge.showVehicleIsEmptyWarning then
			g_currentMission:showBlinkingWarning(string.format(specManualDischarge.l10nTexts.EMPTY_WARNING, self.typeDesc), 2000);

			specManualDischarge.showVehicleIsEmptyWarning = false;
		end;
	end;
end;

function ManualDischarge:objectToDischargeHasFreeCapacity(dischargeNode)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");
	local object = dischargeNode.dischargeObject;
	
    if object == nil then
        return false;
	end;
	
	local fillType = self:getDischargeFillType(dischargeNode);
 
    if not object:getFillUnitSupportsFillType(dischargeNode.dischargeFillUnitIndex, fillType) then
        return false;
	end;
	
    if not object:getFillUnitAllowsFillType(dischargeNode.dischargeFillUnitIndex, fillType) then
        return false;
	end;
	
    if object.getFillUnitFreeCapacity ~= nil and object:getFillUnitFreeCapacity(dischargeNode.dischargeFillUnitIndex, fillType, self:getActiveFarm()) <= 0 then
        return false;
	end;
	
    if object.getIsFillAllowedFromFarm ~= nil and not object:getIsFillAllowedFromFarm(self:getActiveFarm()) then
        return false;
	end;

	return true;
end;

function ManualDischarge:foundObjectToDischarge(dischargeNode)
	if dischargeNode == nil then
        return false;
	end;
	
	if not self:objectToDischargeHasFreeCapacity(dischargeNode) then
		return false;
	end;

    --## Adding should only be done if removing is allowed (generally adding is always allowed because it is beneficial to the receiver)
    --## A case where this is not allowed is when this is a pallet where the the controller does not own it (or can access it)
    if self.getMountObject ~= nil then
		local mounter = self:getDynamicMountObject() or self:getMountObject();
		
        if mounter ~= nil then
            --## if the active farm of the mounter has NO access to farmId fill unit: disallow
            if not g_currentMission.accessHandler:canFarmAccess(mounter:getActiveFarm(), self, true) then
                return false;
            end;
        end;
	end;
	
	return true;
end;

function ManualDischarge:getCanDischargeToObject(superFunc, dischargeNode)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");

	if not specManualDischarge.isBought then
		specManualDischarge.allowDischarge = true;
	end;

	if self.getIsEntered ~= nil then
		if not specManualDischarge.allowDischarge and not self:getIsEntered() then
			return false;
		end;
	end;

	return self:foundObjectToDischarge(dischargeNode) and specManualDischarge.allowDischarge;
end;

function ManualDischarge:setAllowDischarge(allowDischarge, noEventSend)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");
	local specPipe = ManualDischargeUtil.getSpecByName(self, "pipe");

    if allowDischarge ~= specManualDischarge.allowDischarge then
		if not noEventSend then
			if g_server ~= nil then
				g_server:broadcastEvent(ManualDischargeEvent.new(self, allowDischarge), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(ManualDischargeEvent.new(self, allowDischarge));
			end;
		end;

		if allowDischarge then
			self:setDischargeState(Dischargeable.DISCHARGE_STATE_OBJECT);

			if specPipe ~= nil then
				g_animationManager:startAnimations(specPipe.animationNodes);
			end;
		else
			self:setDischargeState(Dischargeable.DISCHARGE_STATE_OFF);
		end;

		specManualDischarge.allowDischarge = allowDischarge;
	end;
end;

function ManualDischarge:onRegisterActionEvents(isActiveForInput)
	if self.isClient then
        local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");
		
		if specManualDischarge.isBought then
			self:clearActionEventsTable(specManualDischarge.actionEvents);
			
			if self:getIsActiveForInput(true) then
				local _, dischargeButtonId = self:addActionEvent(specManualDischarge.actionEvents, InputAction.DISCHARGE_BUTTON, self, ManualDischarge.actionEventDischarge, false, true, false, true, nil);
				
				g_inputBinding:setActionEventTextPriority(dischargeButtonId, GS_PRIO_NORMAL);
				g_inputBinding:setActionEventTextVisibility(dischargeButtonId, false);
				g_inputBinding:setActionEventText(dischargeButtonId, specManualDischarge.l10nTexts.START_DISCHARGE);
				g_inputBinding:setActionEventActive(dischargeButtonId, false);
			end;
		end;
	end;
end;

function ManualDischarge.actionEventDischarge(self, actionName, inputValue, callbackState, isAnalog)
	local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");

	specManualDischarge.showVehicleIsEmptyWarning = ManualDischargeUtil.getIsVehicleEmpty(self);
	specManualDischarge.showTurnOnMotorWarning = not ManualDischargeUtil.getIsVehicleMotorStarted(self);

	if not ManualDischargeUtil.getIsVehicleEmpty(self) and ManualDischargeUtil.getIsVehicleMotorStarted(self) then
		self:setAllowDischarge(not specManualDischarge.allowDischarge);
	else
		self:setAllowDischarge(false);
	end;
end;

function ManualDischarge:onWriteStream(streamId, connection)
    if not connection:getIsServer() then 
		local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");

		if not specManualDischarge.isBought then
			specManualDischarge.allowDischarge = true;
		end;
		
		streamWriteBool(streamId, specManualDischarge.allowDischarge);
	end;
end;

function ManualDischarge:onReadStream(streamId, connection)
    if connection:getIsServer() then
		local specManualDischarge = ManualDischargeUtil.getSpecByName(self, "manualDischarge");

		if not specManualDischarge.isBought then
			specManualDischarge.allowDischarge = true;
		end;
		
		specManualDischarge.allowDischarge = streamReadBool(streamId);
	end;
end;

-----------------------------------------------------------------------------
--## Multiplayer Event
-----------------------------------------------------------------------------

ManualDischargeEvent = {};
ManualDischargeEvent_mt = Class(ManualDischargeEvent, Event);

InitEventClass(ManualDischargeEvent, "ManualDischargeEvent");

function ManualDischargeEvent.emptyNew()
	local self = Event.new(ManualDischargeEvent_mt);
    
	return self;
end;

function ManualDischargeEvent.new(vehicle, allowDischarge)
	local self = ManualDischargeEvent.emptyNew();
	
	self.vehicle = vehicle;
	self.allowDischarge = allowDischarge;

	return self;
end;

function ManualDischargeEvent:readStream(streamId, connection)
	self.vehicle = NetworkUtil.readNodeObject(streamId);
	self.allowDischarge = streamReadBool(streamId);
	
	self:run(connection);
end;

function ManualDischargeEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.vehicle);

	streamWriteBool(streamId, self.allowDischarge);
end;

function ManualDischargeEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(ManualDischargeEvent.new(self.vehicle, self.allowDischarge), nil, connection, self.vehicle);
	end;
	
    if self.vehicle ~= nil then
        self.vehicle:setAllowDischarge(self.allowDischarge, true);
	end;
end;