--
-- Enhanced filltype categories
--
-- Author: Wopster
-- Description: adds missing fill types to fill type categories.
-- Icon:
-- Name: Enhanced filltype categories
-- Hide: no
--
-- Copyright (c) Wopster, 2021

---@type string directory of the mod.
local modDirectory = g_currentModDirectory or ""
---@type string name of the mod.
local modName = g_currentModName or "unknown"

local function addFillTypeToCategory(fillTypeName, categoryName)
    local fillType = g_fillTypeManager:getFillTypeByName(fillTypeName:upper())
    local categoryIndex = g_fillTypeManager.nameToCategoryIndex[categoryName:upper()]
    if fillType ~= nil and categoryIndex ~= nil then
        if not g_fillTypeManager:addFillTypeToCategory(fillType.index, categoryIndex) then
            log(("Error: failed to add %s to %s"):format(fillTypeName, categoryName))
        end
    end
end

local function registerFillTypes()
    addFillTypeToCategory("lime", "manureSpreader")
    addFillTypeToCategory("lime", "spreader")
    addFillTypeToCategory("lime", "augerWagon")
    addFillTypeToCategory("water", "slurryTank")
end

local function registerEffects()
    local customEnvironment, _ = Utils.getModNameAndBaseDirectory(modDirectory)
    local filename = Utils.getFilename("data/effects/motionPathEffects.xml", modDirectory)
    local xmlFile = XMLFile.load("motionPathXML", filename)
    if xmlFile ~= nil then
        g_motionPathEffectManager:loadMotionPathEffects(xmlFile.handle, "motionPathEffects.motionPathEffect", modDirectory, customEnvironment)
        xmlFile:delete()
    end
end

local effectsToMerge = {
    ["MANURE_LIME_SPREADER"] = "MANURE_SPREADER"
}

local function mergeEffects()
    for from, into in pairs(effectsToMerge) do
        local intoEffects = g_motionPathEffectManager.effectsByType[into:upper()]

        if intoEffects ~= nil then
            local fromEffects = g_motionPathEffectManager.effectsByType[from:upper()]

            for _, intoEffect in pairs(intoEffects) do
                for i, effectMaterial in ipairs(intoEffect.effectMaterials) do
                    delete(effectMaterial.node)
                    g_motionPathEffectManager:deleteCustomShaderMaps(effectMaterial)
                    table.remove(intoEffect.effectMaterials, i)
                end

                intoEffect.effectMaterials = {}

                for _, fromEffect in pairs(fromEffects) do
                    for i, effectMaterial in pairs(fromEffect.effectMaterials) do
                        effectMaterial.parent = intoEffect
                        table.insert(intoEffect.effectMaterials, effectMaterial)
                        table.remove(fromEffect.effectMaterials, i)
                    end
                end
            end
        else
            Logging.error("Enhanced fillType categories is missing the effect type %s in your map, please check for mod conflicts!", into)
        end
    end
end

---Init the mod.
local function init()
    FillTypeManager.loadMapData = Utils.appendedFunction(FillTypeManager.loadMapData, registerFillTypes)
    MotionPathEffectManager.loadMapData = Utils.appendedFunction(MotionPathEffectManager.loadMapData, registerEffects)
    FSBaseMission.onFinishedLoading  = Utils.prependedFunction(FSBaseMission.onFinishedLoading, mergeEffects)
end

init()
