--
-- Author: E.T.A La Marchoise
-- Date: 01-11-22
-- Version: 1.0.0.0
-- Copyright (C) E.T.A La Marchoise, All Rights Reserved
-- CabinPanelsBaseVehicle.lua
--

CabinPanelsBaseVehicle = {};
CabinPanelsBaseVehicle.MOD_DIRECTORY = g_currentModDirectory;

local oldCabinPanelsOnLoadFunction = CabinPanels.onLoad;
CabinPanels.onLoad = function(self, ...)
    if not g_vehicleBaseCabinPanelsIntegration[self.xmlFile.filename] then
        oldCabinPanelsOnLoadFunction(self, ...);
    end
end;

function CabinPanelsBaseVehicle.prerequisitesPresent(specializations)
    return true;
end

function CabinPanelsBaseVehicle.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", CabinPanelsBaseVehicle);
end

function CabinPanelsBaseVehicle.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "doForHoseConnector", CabinPanelsBaseVehicle.doForHoseConnector);
    SpecializationUtil.registerFunction(vehicleType, "concatCabinIndexAndChild", CabinPanelsBaseVehicle.concatCabinIndexAndChild);
end

function CabinPanelsBaseVehicle:onLoad(savegame)
    if g_isCabinPanelsDevMode then
        Logging.info("Load spec CabinPanelsBaseVehicle for " .. tostring(self.xmlFile:getFilename()));
    end

    local cabinPanelData = g_vehicleBaseCabinPanelsIntegration[self.xmlFile:getFilename()];

    local cabinIndex = cabinPanelData.cabinNode;
    local cabinNode = I3DUtil.indexToObject(self.components, cabinIndex);
    if cabinNode == nil or cabinNode == 0 then
        Logging.xmlError(self.xmlFile, string.format("Invalid cabin node for %s - vehicle base integration", self.xmlFile.filename));
        return;
    end

    local cabinParent = getParent(cabinNode);
    local modIndex = I3DUtil.getNodePathIndices(cabinParent) .. "|";

    if g_isCabinPanelsDevMode then
        Logging.info("Cabin parent : " .. tostring(getName(cabinParent)) .. " => " .. tostring(cabinParent))
    end

    for iPanel, panel in pairs(cabinPanelData.cabinPanels) do
        local panelTransformGroup = createTransformGroup("panel" .. tostring(iPanel));
        setTranslation(panelTransformGroup, unpack(panel.translation));
        setRotation(panelTransformGroup, unpack(panel.rotation));

        link(cabinNode, panelTransformGroup);

        local numOfChildren = getNumOfChildren(cabinNode) - 1;
        self.xmlFile:setString(string.format("vehicle.cabinPanels.cabinPanel(%d)#node", iPanel - 1), self:concatCabinIndexAndChild(cabinIndex, numOfChildren));

        if panel.attacherJointIndices ~= nil then
            self.xmlFile:setString(string.format("vehicle.cabinPanels.cabinPanel(%d)#attacherJointIndices", iPanel - 1), panel.attacherJointIndices);
        end
    end

    for iHoseConnector, hoseConnector in pairs(cabinPanelData.hoseConnectors) do
        local key = string.format("vehicle.cabinPanels.hoseConnectors.hoseConnector(%d)", iHoseConnector - 1);

        self:doForHoseConnector(iHoseConnector, hoseConnector.indoor, modIndex, "indoor", key .. ".indoor", cabinNode, cabinIndex);
        self:doForHoseConnector(iHoseConnector, hoseConnector.outdoor, modIndex, "outdoor", key .. ".outdoor", cabinNode, cabinIndex);
    end

    for iSharedPart, sharedPart in pairs(cabinPanelData.sharedParts) do
        local filename = Utils.getFilename(sharedPart.filename, CabinPanelsBaseVehicle.MOD_DIRECTORY);
        local sharedPartsI3D = g_i3DManager:loadSharedI3DFile(filename);
        local sharedPartI3D = I3DUtil.indexToObject(sharedPartsI3D, sharedPart.childNode);

        link(cabinNode, sharedPartI3D);
        setTranslation(sharedPartI3D, unpack(sharedPart.translation));
        setRotation(sharedPartI3D, unpack(sharedPart.rotation));

        setShaderParameter(sharedPartI3D, "colorMat0", sharedPart.colorMat0[1], sharedPart.colorMat0[2], sharedPart.colorMat0[3], sharedPart.colorMat0[4], false);
    end

    if oldCabinPanelsOnLoadFunction ~= nil then
        oldCabinPanelsOnLoadFunction(self, savegame);
    end
end

function CabinPanelsBaseVehicle:doForHoseConnector(iHoseConnector, hoseConnector, modIndex, name, key, cabinNode, cabinIndex)
    local hoseConnectorTransformGroup = createTransformGroup(name .. tostring(iHoseConnector));

    setTranslation(hoseConnectorTransformGroup, unpack(hoseConnector.translation));
    setRotation(hoseConnectorTransformGroup, unpack(hoseConnector.rotation));

    link(cabinNode, hoseConnectorTransformGroup);

    local numOfChildren = getNumOfChildren(cabinNode) - 1;
    self.xmlFile:setString(key .. "#node", self:concatCabinIndexAndChild(cabinIndex, numOfChildren));
    self.xmlFile:setString(key .. "#type", hoseConnector.type);

    if hoseConnector.straighteningFactor ~= nil then
        self.xmlFile:setFloat(key .. "#straighteningFactor", hoseConnector.straighteningFactor);
    end
end

function CabinPanelsBaseVehicle:concatCabinIndexAndChild(cabinIndex, childIndex)
    local ending = ">";

    if cabinIndex == nil or childIndex == nil then
        return nil;
    end

    if cabinIndex:sub(-#ending) == ending then
        return cabinIndex .. childIndex;
    end

    return cabinIndex .. "|" .. childIndex;
end