--
-- Author: E.T.A La Marchoise
-- Date: 01-11-22
-- Version: 1.0.0.0
-- Copyright (C) E.T.A La Marchoise, All Rights Reserved
-- vehicleBaseCabinPanelsIntegration.lua
--

g_vehicleBaseCabinPanelsIntegration = {};

local function registerHoseConnectorSchema(schema, key, defaultType)
    schema:register(XMLValueType.STRING, key .. "#translation", "Translation of indoor hose connector node", "0 0 0");
    schema:register(XMLValueType.STRING, key .. "#rotation", "Rotation of indoor hose connector node", "0 0 0");
    schema:register(XMLValueType.STRING, key .. "#type", "Type of hose connector", defaultType);
    schema:register(XMLValueType.FLOAT, key .. "#straighteningFactor", "Straightening Factor", 1);
end

-- Create file schema
local schema = XMLSchema.new("vehicleBaseIntegration");
schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).storeItem", "Filename to storeItem", nil, true);
schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).newVehicleType", "New vehicle type name", nil, true);
schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise#cabinNode", "Cabin node - used as parent for panels", nil, true);

schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.cabinPanels.cabinPanel(?)#translation", "Translation of cabin panel node", "0 0 0");
schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.cabinPanels.cabinPanel(?)#rotation", "Rotation of cabin panel node", "0 0 0");
schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.cabinPanels.cabinPanel(?)#attacherJointIndices", "List of corresponding attacher joint indices", nil);

registerHoseConnectorSchema(schema, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.cabinPanels.hoseConnectors.hoseConnector(?).indoor", "cabinPanel_indoor");
registerHoseConnectorSchema(schema, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.cabinPanels.hoseConnectors.hoseConnector(?).outdoor", "cabinPanel_outdoor");

schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.sharedParts.sharedPart(?)#filename", nil, true);
schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.sharedParts.sharedPart(?)#childNode", "0");
schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.sharedParts.sharedPart(?)#translation", "0 0 0");
schema:register(XMLValueType.STRING, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.sharedParts.sharedPart(?)#rotation", "0 0 0");
schema:register(XMLValueType.COLOR, "vehicleBaseIntegration.vehicleBase(?).etaLaMarchoise.sharedParts.sharedPart(?)#colorMat0", {0, 0, 0, 1});

function getHoseConnector(hoseConnector, xmlFile, xmlKey, defaultType)
    hoseConnector.translation = string.getVectorN(xmlFile:getString(xmlKey .. "#translation", "0 0 0"))
    hoseConnector.rotation = string.getRadiansNoNil(xmlFile:getString(xmlKey .. "#rotation", "0 0 0"))
    hoseConnector.type = xmlFile:getString(xmlKey .. "#type", defaultType)
    hoseConnector.straighteningFactor = xmlFile:getFloat(xmlKey .. "#straighteningFactor", 1)
end

local xmlFile = XMLFile.load("vehicleBaseIntegration", Utils.getFilename(g_vehicleBaseCabinPanelsIntegrationXmlFilename, g_currentModDirectory), schema);

xmlFile:iterate("vehicleBaseIntegration.vehicleBase", function(index, key)
    local entry = {
        xml = xmlFile:getValue(key .. ".storeItem"),
        newVehicleType = g_currentModName .. "." .. xmlFile:getString(key .. ".newVehicleType"),
        cabinNode = xmlFile:getValue(key .. ".etaLaMarchoise#cabinNode"),
        cabinPanels = {},
        hoseConnectors = {},
        sharedParts = {}
    };

    xmlFile:iterate(key .. ".etaLaMarchoise.cabinPanels.cabinPanel", function(_, cabinPanelKey)
        table.insert(entry.cabinPanels, {
            translation = string.getVectorN(xmlFile:getString(cabinPanelKey .. "#translation", "0 0 0")),
            rotation = string.getRadiansNoNil(xmlFile:getString(cabinPanelKey .. "#rotation", "0 0 0")),
            attacherJointIndices = xmlFile:getValue(cabinPanelKey .. "#attacherJointIndices")
        })
    end);

    xmlFile:iterate(key .. ".etaLaMarchoise.cabinPanels.hoseConnectors.hoseConnector", function(_, hoseConnectorKey)
        local hoseConnector = {
            indoor = {},
            outdoor = {}
        };

        getHoseConnector(hoseConnector.indoor, xmlFile, hoseConnectorKey .. ".indoor", "cabinPanel_indoor");
        getHoseConnector(hoseConnector.outdoor, xmlFile, hoseConnectorKey .. ".outdoor", "cabinPanel_outdoor");

        table.insert(entry.hoseConnectors, hoseConnector);
    end);

    xmlFile:iterate(key .. ".etaLaMarchoise.sharedParts.sharedPart", function(_, sharedPartKey)
        local filename = xmlFile:getValue(sharedPartKey .. "#filename");

        if filename == nil then
            Logging.error(xmlFile, "Missing filename for (" .. sharedPartKey .. ")");
        else
            table.insert(entry.sharedParts, {
                filename = filename,
                translation = string.getVectorN(xmlFile:getString(sharedPartKey .. "#translation", "0 0 0")),
                rotation = string.getRadiansNoNil(xmlFile:getString(sharedPartKey .. "#rotation", "0 0 0")),
                childNode = xmlFile:getValue(sharedPartKey .. "#childNode", "0"),
                colorMat0 = xmlFile:getValue(sharedPartKey .. "#colorMat0", {0, 0, 0, 1})
            });
        end
    end);

    g_vehicleBaseCabinPanelsIntegration[entry.xml] = entry;
end);

xmlFile:delete();

local function loadVehicle(self, superFunc, vehicleData, ...)
    local xmlFilename = vehicleData.filename;
    local data = g_vehicleBaseCabinPanelsIntegration[xmlFilename];

    if data ~= nil then
        local newVehicleType = data.newVehicleType;
        local typeEntry = g_vehicleTypeManager:getTypeByName(newVehicleType);

        if typeEntry ~= nil then
            vehicleData.typeName = newVehicleType;
            self.typeName = newVehicleType;
        end
    end

    return superFunc(self, vehicleData, ...);
end

Vehicle.load = Utils.overwrittenFunction(Vehicle.load, loadVehicle);