-- CabinPanels
--
-- @author E.T.A La Marchoise
-- @date 18/01/2021
-- @version 1.0.0.0
--
-- Copyright (C) E.T.A La Marchoise, Confidential, All Rights Reserved.

local MOD_DIRECTORY = g_currentModDirectory;

source(MOD_DIRECTORY .. "scripts/model/interactivePanelButton.lua"); -- Load InteractivePanelButton class

Panel = {};
local Panel_mt = Class(Panel);

---@class Panel
function Panel:new(xmlFile, key, iPanel, components, i3dMappings, baseDirectory, customEnvironment)
	local self = setmetatable({}, Panel_mt);

	self.key = key;
	self.baseDirectory = baseDirectory;

	-- implement xml
	if g_isCabinPanelsDevMode then
		Logging.info(string.format("Try to load panel xml file from %s and %s", xmlFile:getValue(key .. "#filename"), self.baseDirectory))
	end

	local filename = Utils.getFilename(xmlFile:getValue(key .. "#filename"), self.baseDirectory);

	if g_isCabinPanelsDevMode then
		Logging.info(string.format("Try to load panel xml file from %s and %s => %s", xmlFile:getValue(key .. "#filename"), self.baseDirectory, tostring(filename)))
	end

	self.xmlPanelFile = XMLFile.load("xmlPanel", filename, AttachablePanels.xmlSchema);

	if self.xmlPanelFile == nil then
		Logging.error(string.format("Invalid panel xml filename (%s)", filename));
		return;
	end

	self.detachedNode = xmlFile:getValue(key .. "#detachedNode", nil, components, i3dMappings);
	self.rotation = xmlFile:getValue(key .. ".rotation", "0 0 0", true);

	--panel xml
	self.panelI3dFilename = self.xmlPanelFile:getValue("panel.filename");
	if self.panelI3dFilename == nil then
		Logging.error(string.format("Invalid panel i3d filename (%s)", self.panelI3dFilename));
		return;
	end

	self.connectionHoses = {};

	self.xmlPanelFile:iterate("panel.connectionHoses.hose", function(iHose, hoseKey)
		local connectionHose = {};

		if self:loadPanelHoseTargetNode(self.xmlPanelFile, hoseKey, connectionHose, components, i3dMappings) then
			table.insert(self.connectionHoses, connectionHose);
		end
	end);

	self.buttons = {};

	self.xmlPanelFile:iterate("panel.interactivePanel.buttons.button", function(iButton, buttonKey)
		table.insert(self.buttons, InteractivePanelButton:new(self.xmlPanelFile, buttonKey, baseDirectory, customEnvironment));
	end);

	self.objectChanges = {};

	return self;
end

function Panel:hasDetachedPosition()
	return self.detachedNode ~= nil;
end

function Panel:getDetachedNode()
	return self.detachedNode;
end

function Panel:getI3dFilename()
	return self.panelI3dFilename
end

function Panel:getI3dFilenameWithBaseDirectory()
	return Utils.getFilename(self.panelI3dFilename, self.baseDirectory);
end

function Panel:needToBeRotated()
	return self.rotation ~= nil;
end

function Panel:getRotation()
	return self.rotation;
end

function Panel:getConnectionHosesCount()
	return #self.connectionHoses;
end

function Panel:getConnectionHose(iHose)
	return self.connectionHoses[iHose];
end

function Panel:getConnectionHoses()
	return self.connectionHoses;
end

function Panel:loadPanelHoseTargetNode(xmlFile, targetKey, entry, components, i3dMappings)
	entry.attacherJointIndices = {}

	entry.inputNode = xmlFile:getValue(targetKey .. "#inputNode")
	entry.type = xmlFile:getValue(targetKey .. "#type")
	entry.specType = xmlFile:getValue(targetKey .. "#specType")
	entry.diameter = xmlFile:getValue(targetKey .. "#diameter")
	entry.straighteningFactor = xmlFile:getValue(targetKey .. "#straighteningFactor", 1)
	entry.straighteningDirection = xmlFile:getValue(targetKey .. "#straighteningDirection", nil, true)
	local socketName = xmlFile:getValue(targetKey .. "#socket")

	if socketName ~= nil then
		local socketColor = xmlFile:getValue(targetKey .. "#socketColor", nil, true)
		entry.socket = g_connectionHoseManager:linkSocketToNode(socketName, entry.node, self.customEnvironment, socketColor)
	end

	if entry.type == nil then
		Logging.xmlWarning(xmlFile, "Missing type for '%s'", targetKey)

		return false
	end

	entry.adapterName = xmlFile:getValue(targetKey .. "#adapterType", "DEFAULT")

	if entry.node ~= nil and entry.adapter == nil then
		entry.adapter = {
			node = entry.node,
			refNode = entry.node
		}
	end

	entry.objectChanges = {}

	ObjectChangeUtil.loadObjectChangeFromXML(xmlFile, targetKey, entry.objectChanges, self.components, self)
	ObjectChangeUtil.setObjectChanges(entry.objectChanges, false)

	return true
end