-- InteractivePanelButton
--
-- @author E.T.A La Marchoise
-- @date 25/01/2021
-- @version 1.0.0.0
--
-- Copyright (C) E.T.A La Marchoise, Confidential, All Rights Reserved.

InteractivePanelButton = {};
InteractivePanelButton.DEFAULT_RADIUS = 0.02;
local InteractivePanelButton_mt = Class(InteractivePanelButton);

---@class InteractivePanelButton
function InteractivePanelButton:new(xmlFile, key, baseDirectory, customEnvironment)
    local self = setmetatable({}, InteractivePanelButton_mt);

    self.key = key;

    self.node = xmlFile:getString(key .. "#node");
    self.markerNode = xmlFile:getString(key .. "#markerNode");

    local markerFilename = xmlFile:getValue(key .. "#markerFilename");
    if markerFilename == nil then
        self.markerFilename = g_markerFilename;
    else
        self.markerFilename = Utils.getFilename(markerFilename, baseDirectory);
    end

    self.radius = xmlFile:getValue(key .. "#radius", InteractivePanelButton.DEFAULT_RADIUS);

    local action = xmlFile:getValue(key .. ".action");
    if action ~= nil then
        local buttonAction = ButtonAction[string.upper(action)];

        if buttonAction == nil then
            Logging.xmlError(xmlFile, string.format("Invalid action name for interactive button"));
        else
            self.action = buttonAction;
        end
    end

    self.label = {
        activate = g_i18n:convertText(xmlFile:getValue(key .. "#activate", InteractivePanelButton.getDefaultLabel(self.action, true)), customEnvironment),
        deactivate = g_i18n:convertText(xmlFile:getValue(key .. "#deactivate", InteractivePanelButton.getDefaultLabel(self.action, false)), customEnvironment)
    };

    -- Animation played on attached implement when button is clicked
    self.actionAnimationName = xmlFile:getValue(key .. ".animation");

    -- Button animation played when button is clicked
    self.buttonAnimationKey = key .. ".buttonAnimation";
    self.buttonAnimationName = string.random(math.random(10, 40));
    self.animation = nil;

    self.active = false;

    return self;
end

function InteractivePanelButton:getActionAnimationName()
    return self.actionAnimationName;
end

function InteractivePanelButton:getAnimation()
    return self.animation;
end

function InteractivePanelButton:getButtonAnimationName()
    return self.buttonAnimationName;
end

function InteractivePanelButton:getActionLabel()
    return self.active and self.label.deactivate or self.label.activate;
end

function InteractivePanelButton:getAction()
    return self.action;
end

function InteractivePanelButton:getMarkerFilename()
    return self.markerFilename;
end

function InteractivePanelButton:getMarkerNode()
    return self.markerNode;
end

function InteractivePanelButton:getNode()
    return self.node;
end

function InteractivePanelButton:getRadius()
    return self.radius;
end

function InteractivePanelButton:isActive()
    return self.active;
end

function InteractivePanelButton:setActive(state)
    self.active = state;
end

function InteractivePanelButton.getDefaultLabel(action, isActivation)
    local defaultActionLabel = g_defaultActionLabels[action] or g_defaultActionLabels["default"];
    return isActivation and defaultActionLabel.activated or defaultActionLabel.deactivated;
end