-- CabinPanels
--
-- @author E.T.A La Marchoise
-- @date 18/01/2021
-- @version 1.0.0.0
--
-- Copyright (C) E.T.A La Marchoise, Confidential, All Rights Reserved.

CabinPanels = {};
CabinPanels.MOD_DIRECTORY = g_currentModDirectory;

CabinPanels.MOD_NAME = g_currentModName;

if g_isCabinPanelsDevMode then
	Logging.info("CabinPanels.MOD_NAME = " .. tostring(CabinPanels.MOD_NAME))
end

function CabinPanels.prerequisitesPresent(specializations)
	return true;
end

function CabinPanels.initSpecialization()
	local schema = Vehicle.xmlSchema;

	schema:setXMLSpecializationType("CabinPanels");
	schema:register(XMLValueType.NODE_INDEX, "vehicle.cabinPanels.cabinPanel(?)#node", "Cabin panel node");
	schema:register(XMLValueType.VECTOR_N, "vehicle.cabinPanels.cabinPanel(?)#attacherJointIndices", "List of corresponding attacher joint indices");

	schema:register(XMLValueType.VECTOR_N, "vehicle.cabinPanels.hoseConnectors.hoseConnector(?)#attacherJointIndices", "List of corresponding attacher joint indices");

	ConnectionHoses.registerHoseTargetNodesXMLPaths(schema, "vehicle.cabinPanels.outdoorHoseConnectors.hoseConnector(?)");
	CabinPanels.registerCabinPanelHoseConnectionNodesXMLPaths(schema, "vehicle.cabinPanels.hoseConnectors.hoseConnector(?).indoor");
	CabinPanels.registerCabinPanelHoseConnectionNodesXMLPaths(schema, "vehicle.cabinPanels.hoseConnectors.hoseConnector(?).outdoor");

	schema:setXMLSpecializationType();
end

function CabinPanels.registerCabinPanelHoseConnectionNodesXMLPaths(schema, basePath)
	schema:register(XMLValueType.STRING, basePath .. "#type", "Hose type")
	schema:register(XMLValueType.STRING, basePath .. "#specType", "Connection hose specialization type (if defined it needs to match the type of the other tool)")
	schema:register(XMLValueType.STRING, basePath .. "#hoseType", "Hose material type", "DEFAULT")
	schema:register(XMLValueType.NODE_INDEX, basePath .. "#node", "Hose output node")
	schema:register(XMLValueType.BOOL, basePath .. "#isTwoPointHose", "Is two point hose without sagging", false)
	schema:register(XMLValueType.BOOL, basePath .. "#isWorldSpaceHose", "Sagging is calculated in world space or local space of hose node", true)
	schema:register(XMLValueType.STRING, basePath .. "#dampingRange", "Damping range in meters", 0.05)
	schema:register(XMLValueType.FLOAT, basePath .. "#dampingFactor", "Damping factor", 50)
	schema:register(XMLValueType.FLOAT, basePath .. "#straighteningFactor", "Straightening Factor", 1)
	schema:register(XMLValueType.FLOAT, basePath .. "#centerPointDropFactor", "Can be used to manipulate how much the hose will drop while it's getting shorter then set", 1)
	schema:register(XMLValueType.FLOAT, basePath .. "#centerPointTension", "Defines the tension on the center control point (0: default behavior)", 0)
	schema:register(XMLValueType.ANGLE, basePath .. "#minCenterPointAngle", "Min. angle of sagged curve", "Defined on connectionHose xml, default 90 degree")
	schema:register(XMLValueType.VECTOR_TRANS, basePath .. "#minCenterPointOffset", "Min. center point offset from hose node", "unlimited")
	schema:register(XMLValueType.VECTOR_TRANS, basePath .. "#maxCenterPointOffset", "Max. center point offset from hose node", "unlimited")
	schema:register(XMLValueType.FLOAT, basePath .. "#minDeltaY", "Min. delta Y from center point")
	schema:register(XMLValueType.NODE_INDEX, basePath .. "#minDeltaYComponent", "Min. delta Y reference node")
	schema:register(XMLValueType.COLOR, basePath .. "#color", "Hose color")
	schema:register(XMLValueType.STRING, basePath .. "#adapterType", "Adapter type name")
	schema:register(XMLValueType.NODE_INDEX, basePath .. "#adapterNode", "Link node for detached adapter")
	schema:register(XMLValueType.STRING, basePath .. "#outgoingAdapter", "Adapter type that is used for outgoing connection hose")
	schema:register(XMLValueType.STRING, basePath .. "#socket", "Outgoing socket name to load")
	schema:register(XMLValueType.COLOR, basePath .. "#socketColor", "Socket custom color")
	ObjectChangeUtil.registerObjectChangeXMLPaths(schema, basePath)
end

function CabinPanels.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "loadCabinPanelHoseConnector", CabinPanels.loadCabinPanelHoseConnector);
	SpecializationUtil.registerFunction(vehicleType, "loadCabinPanelHoseConnectors", CabinPanels.loadCabinPanelHoseConnectors);
	SpecializationUtil.registerFunction(vehicleType, "getFreeCabinConnectionHoseSlot", CabinPanels.getFreeCabinConnectionHoseSlot);
	SpecializationUtil.registerFunction(vehicleType, "getClonedHose", CabinPanels.getClonedHose);
	SpecializationUtil.registerFunction(vehicleType, "loadCabinPanelNodes", CabinPanels.loadCabinPanelNodes);
	SpecializationUtil.registerFunction(vehicleType, "getDistance", CabinPanels.getDistance);
	SpecializationUtil.registerFunction(vehicleType, "deleteAttachedPanel", CabinPanels.deleteAttachedPanel);
	SpecializationUtil.registerFunction(vehicleType, "deleteChild", CabinPanels.deleteChild);
	SpecializationUtil.registerFunction(vehicleType, "loadCustomAnimation", CabinPanels.loadCustomAnimation);
	SpecializationUtil.registerFunction(vehicleType, "playCustomAnimation", CabinPanels.playCustomAnimation);
	SpecializationUtil.registerFunction(vehicleType, "setButtonState", CabinPanels.setButtonState);
	SpecializationUtil.registerFunction(vehicleType, "setPanelsVisualMarkersVisibility", CabinPanels.setPanelsVisualMarkersVisibility);
	SpecializationUtil.registerFunction(vehicleType, "hasAnyFreePanelSlot", CabinPanels.hasAnyFreePanelSlot);
	SpecializationUtil.registerFunction(vehicleType, "getFreeCabinPanelSlot", CabinPanels.getFreeCabinPanelSlot);
	SpecializationUtil.registerFunction(vehicleType, "addPanel", CabinPanels.addPanel);
	SpecializationUtil.registerFunction(vehicleType, "getAttachedCabinPanels", CabinPanels.getAttachedCabinPanels);
	SpecializationUtil.registerFunction(vehicleType, "hasCabinPanelAttacherJointIndex", CabinPanels.hasCabinPanelAttacherJointIndex);
	SpecializationUtil.registerFunction(vehicleType, "disconnectCabinPanelHose", CabinPanels.disconnectCabinPanelHose);
	SpecializationUtil.registerFunction(vehicleType, "isAnyCabinPanelsAttached", CabinPanels.isAnyCabinPanelsAttached);
	SpecializationUtil.registerFunction(vehicleType, "setInteractivePanelState", CabinPanels.setInteractivePanelState);
end

function CabinPanels.registerEventListeners(vehicleType)
  	SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", CabinPanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onLoad", CabinPanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onDraw", CabinPanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onUpdate", CabinPanels);
  	SpecializationUtil.registerEventListener(vehicleType, "mouseEvent", CabinPanels);
	SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", CabinPanels);
	SpecializationUtil.registerEventListener(vehicleType, "onCameraChanged", CabinPanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onPreAttachImplement", CabinPanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onPreDetachImplement", CabinPanels);
end

function CabinPanels:onPreLoad(savegame)
	local specName = ("spec_%s.cabinPanels"):format(CabinPanels.MOD_NAME);

	self.spec_cabinPanels = self[specName];
	self[specName] = nil;
end

function CabinPanels:onLoad()
	local spec = self.spec_cabinPanels;

	spec.cabinPanels = {};
	spec.hoses = {};
	spec.hoveredButton = {};

	spec.interactivePanelActive = false;
	spec.wasInteractivePanelActive = false;

	self:loadCabinPanelNodes(spec.cabinPanels);
	self:loadCabinPanelHoseConnectors(spec.hoses);
	self:addHoseTargetNodes(self.xmlFile, "vehicle.cabinPanels.outdoorHoseConnectors.hoseConnector");
end

function CabinPanels:onDraw()
	local spec = self.spec_cabinPanels;
	if spec.cabinPanels ~= nil and spec.interactivePanelActive then
		renderText(0.5, 0.5, 0.02, '+');
	end
end

function CabinPanels:onUpdate()
	local spec = self.spec_cabinPanels;

	-- detect activation state changed
	if spec.wasInteractivePanelActive ~= spec.interactivePanelActive then
		self:setPanelsVisualMarkersVisibility(spec.interactivePanelActive)

		spec.wasInteractivePanelActive = spec.interactivePanelActive;
	end

	spec.hoveredButton = nil;

	if spec.interactivePanelActive then
		if spec.cabinPanels ~= nil then
			for _, cabinPanel in pairs(spec.cabinPanels) do
				local panel = cabinPanel.attachedPanel;

				if panel ~= nil then
					for _, button in pairs(panel.buttons) do
						local node = I3DUtil.indexToObject(cabinPanel.node, button:getNode());
						local wX, wY, wZ = getWorldTranslation(node);
						local x, y, z = project(wX, wY, wZ);
						local radius = button:getRadius();

						if (0.5 <= x + radius and 0.5 >= x - radius) and (0.5 <= y + radius / 2 and 0.5 >= y - radius) then
							spec.hoveredButton = {button = button, cabinPanel = cabinPanel};
						end
					end
				end
			end
		end

		if spec.hoveredButton ~= nil then
			local label = spec.hoveredButton.button:getActionLabel();
			if label ~= nil then
				g_currentMission:addExtraPrintText(string.format(label, spec.hoveredButton.cabinPanel.vehicle:getName()));
			end
		end
	end
end

function CabinPanels:onCameraChanged(camera, cameraIndex)
	if not camera.isInside then
		self:setInteractivePanelState(false);
	end
end

function CabinPanels:onRegisterActionEvents()
	local spec = self.spec_cabinPanels;

	if self.isClient then
		self:clearActionEventsTable(spec.actionEvents);

		if g_isCabinPanelsDevMode then
			Logging.info("Register action events : self:isAnyCabinPanelsAttached() " .. tostring(self:isAnyCabinPanelsAttached()) .. " - " .. tostring(self.xmlFile:getFilename()))
		end

		if self:isAnyCabinPanelsAttached() then
			if g_isCabinPanelsDevMode then
				Logging.info("Register action events")
			end

			-- Activate / deactivate panel control
			local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.INTERACTIVE_PANEL_TOGGLE, self, CabinPanels.switchInteractivePanelState, false, true, false, true, nil);
			g_inputBinding:setActionEventTextVisibility(actionEventId, true);
			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL);
		end
	end
end

function CabinPanels:onPreAttachImplement(attachedVehicle, inputJointDescIndex, jointDescIndex)
	if g_isCabinPanelsDevMode then
		Logging.info(string.format("OnPostAttach inputJointDescIndex : %d | jointDescIndex : %d", inputJointDescIndex, jointDescIndex))
	end

	if attachedVehicle.spec_attachablePanels == nil then
		if g_isCabinPanelsDevMode then
			Logging.info("Panels not attached due to incompatible attached object");
		end

		return;
	end

	local attachablePanels = attachedVehicle:getAttachablePanels();

	local iPanel = 1;
	local freePanelSlot = self:getFreeCabinPanelSlot();
	if g_isCabinPanelsDevMode then
		Logging.info("iPanel : " .. tostring(iPanel) .. " - #attachablePanels : " .. tostring(#attachablePanels) .. " - freePanelSlot : " .. tostring(freePanelSlot))
	end

	while iPanel <= #attachablePanels and freePanelSlot ~= nil do
		local panel = attachablePanels[iPanel];

		if self:hasCabinPanelAttacherJointIndex(freePanelSlot, jointDescIndex) then
			attachedVehicle:setDetachedPanelVisibility(panel, false);
			self:addPanel(panel, freePanelSlot, attachedVehicle);

			local iPanelHoseConnection = 1;
			local panelHoseConnections = panel:getConnectionHoses();
			local freeCabinConnectionHoseSlot = self:getFreeCabinConnectionHoseSlot();
			local freeAttachedVehicleHoseConnection = attachedVehicle:getFreeTargetHoseConnection();

			if g_isCabinPanelsDevMode then
				Logging.info("iPanelHoseConnection : " .. tostring(iPanelHoseConnection) .. " - #panelHoseConnections : " .. tostring(#panelHoseConnections) .. " - freeCabinConnectionHoseSlot : " .. tostring(freeCabinConnectionHoseSlot).. " - freeAttachedVehicleHoseConnection : " .. tostring(freeAttachedVehicleHoseConnection))
			end

			while iPanelHoseConnection <= #panelHoseConnections and freeCabinConnectionHoseSlot ~= nil and freeAttachedVehicleHoseConnection ~= nil do
				local panelHoseConnection = panelHoseConnections[iPanelHoseConnection];

				local indoorHose = {
					base = freeCabinConnectionHoseSlot.indoor,
					target = panelHoseConnection,
				}

				indoorHose.target.node = I3DUtil.indexToObject(freePanelSlot.node, panelHoseConnection.inputNode);

				indoorHose.base.length = self:getDistance(freePanelSlot.node, indoorHose.target.node);
				indoorHose.base.diameter = indoorHose.target.diameter;

				local outdoorHose = {
					base = freeCabinConnectionHoseSlot.outdoor,
					target = freeAttachedVehicleHoseConnection.hose,
				}

				outdoorHose.base.length = outdoorHose.target.length;
				outdoorHose.base.diameter = indoorHose.target.diameter;

				if self:getClonedHose(indoorHose.base) and self:getClonedHose(outdoorHose.base) then
					if indoorHose.target.adapter == nil then
						indoorHose.target.adapter = {
							node = indoorHose.target.node,
							refNode = indoorHose.target.node
						}
					end

					if outdoorHose.target.adapter == nil then
						outdoorHose.target.adapter = {
							node = outdoorHose.target.node,
							refNode = outdoorHose.target.node
						}
					end

					self:connectHose(indoorHose.base, self, indoorHose.target, false);
					self:connectHose(outdoorHose.base, self, outdoorHose.target, false);

					table.insert(freePanelSlot.connectionHoses, {indoor = indoorHose, outdoor = outdoorHose});
				end

				freeCabinConnectionHoseSlot.panel = freePanelSlot;
				freeCabinConnectionHoseSlot = self:getFreeCabinConnectionHoseSlot()

				freeAttachedVehicleHoseConnection.isAttached = true;
				freeAttachedVehicleHoseConnection = attachedVehicle:getFreeTargetHoseConnection()

				iPanelHoseConnection = iPanelHoseConnection + 1;

				if g_isCabinPanelsDevMode then
					Logging.info("iPanelHoseConnection : " .. tostring(iPanelHoseConnection) .. " - #panelHoseConnections : " .. tostring(#panelHoseConnections) .. " - freeCabinConnectionHoseSlot : " .. tostring(freeCabinConnectionHoseSlot).. " - freeAttachedVehicleHoseConnection : " .. tostring(freeAttachedVehicleHoseConnection))
				end
			end
		end

		iPanel = iPanel + 1;
		freePanelSlot = self:getFreeCabinPanelSlot();

		if g_isCabinPanelsDevMode then
			Logging.info("iPanel : " .. tostring(iPanel) .. " - #attachablePanels : " .. tostring(#attachablePanels) .. " - freePanelSlot : " .. tostring(freePanelSlot))
		end
	end
end

function CabinPanels:onPreDetachImplement(implement)
	if g_isCabinPanelsDevMode then
		Logging.info("onPreDetachImplement")
	end

	local attachedImplement = implement.object;
	if attachedImplement.spec_attachablePanels == nil then
		if g_isCabinPanelsDevMode then
			Logging.info("Panels not detached due to incompatible attached object");
		end

		return;
	end

	local spec = self.spec_cabinPanels;

	for _, cabinPanel in pairs(spec.cabinPanels) do
		if cabinPanel.vehicle == attachedImplement then
			-- Set visibility of detached panel of attached implement
			cabinPanel.vehicle:setDetachedPanelVisibility(cabinPanel.attachedPanel, true);

			-- Delete link between hoses and panel
			for _, hose in pairs(spec.hoses) do
				if hose.panel == cabinPanel then
					hose.panel = nil;
				end
			end

			-- Disconnect hoses
			for iConnectionHose, connectionHose in pairs(cabinPanel.connectionHoses) do
				self:disconnectCabinPanelHose(connectionHose.indoor, true);
				self:disconnectCabinPanelHose(connectionHose.outdoor);
			end

			cabinPanel.connectionHoses = {};

			if g_isCabinPanelsDevMode then
				Logging.info("Remaining connection hose : " .. tostring(#cabinPanel.connectionHoses));
			end

			-- Delete panel from cabin
			self:deleteAttachedPanel(cabinPanel);
		end
	end

	if not self:isAnyCabinPanelsAttached() then
		self:setInteractivePanelState(false);
	end
end

function CabinPanels:disconnectCabinPanelHose(connectionHose, isIndoor)
	if isIndoor then
		connectionHose.target.node = nil;
		connectionHose.target.adapter = nil;
	end

	self:disconnectHose(connectionHose.base);
end

function CabinPanels:onClick()
	local hoveredButton = self.spec_cabinPanels.hoveredButton;

	if hoveredButton ~= nil and hoveredButton.button ~= nil then
		local button = hoveredButton.button;

		hoveredButton.cabinPanel.vehicle:onPanelButtonClicked(button);
		hoveredButton = nil;
	end
end

function CabinPanels:setButtonState(button, isEnabled)
	if button ~= nil then
		button:setActive(isEnabled);

		-- Play button animation
		self:playCustomAnimation(button:getAnimation(), button:isActive() and 1 or -1, nil, true);
	end
end

function CabinPanels:hasCabinPanelAttacherJointIndex(cabinPanel, attacherJointIndex)
	return #cabinPanel.attacherJointIndices == 0 or cabinPanel.attacherJointIndices[attacherJointIndex] ~= nil;
end

function CabinPanels:loadCabinPanelNodes(cabinPanels)
	self.xmlFile:iterate("vehicle.cabinPanels.cabinPanel", function(iPanel, key)
		local node = self.xmlFile:getValue(key .. "#node", nil, self.components, self.i3dMappings);
		local attacherJointIndicesValue = self.xmlFile:getValue(key .. "#attacherJointIndices", nil, true);
		local attacherJointIndices = {};

		for _, attacherJointIndex in ipairs(attacherJointIndicesValue) do
			attacherJointIndices[attacherJointIndex] = attacherJointIndex;
		end

		if node ~= nil then
			table.insert(cabinPanels, {
				node = node, -- vehicle cabin panel node
				attacherJointIndices = attacherJointIndices, -- Corresponding attacher joint indices
				attachedPanel = nil, -- Reference to attached panel (Object Panel)
				vehicle = nil, -- Reference to vehicle of attached panel
				connectionHoses = { }
			});
		end
	end);
end

-- Load base hose node
function CabinPanels:loadCabinPanelHoseConnector(xmlFile, hoseKey, entry, isBaseHose)
	entry.type = xmlFile:getValue(hoseKey .. "#type")
	entry.specType = xmlFile:getValue(hoseKey .. "#specType")

	if entry.type == nil then
		Logging.xmlWarning(xmlFile, "Missing type attribute in '%s'", hoseKey)

		return false
	end

	entry.hoseType = xmlFile:getValue(hoseKey .. "#hoseType", "DEFAULT")
	entry.node = xmlFile:getValue(hoseKey .. "#node", nil, self.components, self.i3dMappings)

	if entry.node == nil then
		Logging.xmlWarning(xmlFile, "Missing node for connection hose '%s'", hoseKey)

		return false
	end

	if isBaseHose then
		local spec = self.spec_connectionHoses
		local type = entry.type .. (entry.specType or "")

		if spec.numHosesByType[type] == nil then
			spec.numHosesByType[type] = 0
		end

		spec.numHosesByType[type] = spec.numHosesByType[type] + 1
		entry.typedIndex = spec.numHosesByType[type]
	end

	entry.isTwoPointHose = xmlFile:getValue(hoseKey .. "#isTwoPointHose", false)
	entry.isWorldSpaceHose = xmlFile:getValue(hoseKey .. "#isWorldSpaceHose", true)
	entry.component = self:getParentComponent(entry.node)
	entry.lastVelY = 0
	entry.lastVelZ = 0
	entry.dampingRange = xmlFile:getValue(hoseKey .. "#dampingRange", 0.05)
	entry.dampingFactor = xmlFile:getValue(hoseKey .. "#dampingFactor", 50)
	entry.straighteningFactor = xmlFile:getValue(hoseKey .. "#straighteningFactor", 1)
	entry.centerPointDropFactor = xmlFile:getValue(hoseKey .. "#centerPointDropFactor", 1)
	entry.centerPointTension = xmlFile:getValue(hoseKey .. "#centerPointTension", 0)
	entry.minCenterPointAngle = xmlFile:getValue(hoseKey .. "#minCenterPointAngle")
	entry.minCenterPointOffset = xmlFile:getValue(hoseKey .. "#minCenterPointOffset", nil, true)
	entry.maxCenterPointOffset = xmlFile:getValue(hoseKey .. "#maxCenterPointOffset", nil, true)

	if entry.minCenterPointOffset ~= nil and entry.maxCenterPointOffset ~= nil then
		for i = 1, 3 do
			if entry.minCenterPointOffset[i] == 0 then
				entry.minCenterPointOffset[i] = -math.huge
			end

			if entry.maxCenterPointOffset[i] == 0 then
				entry.maxCenterPointOffset[i] = math.huge
			end
		end
	end

	entry.minDeltaY = xmlFile:getValue(hoseKey .. "#minDeltaY", math.huge)
	entry.minDeltaYComponent = xmlFile:getValue(hoseKey .. "#minDeltaYComponent", entry.component, self.components, self.i3dMappings)
	entry.color = xmlFile:getValue(hoseKey .. "#color", nil, true)
	entry.adapterName = xmlFile:getValue(hoseKey .. "#adapterType")
	entry.outgoingAdapter = xmlFile:getValue(hoseKey .. "#outgoingAdapter")
	entry.adapterNode = xmlFile:getValue(hoseKey .. "#adapterNode", nil, self.components, self.i3dMappings)

	if entry.adapterNode ~= nil then
		local node = g_connectionHoseManager:getClonedAdapterNode(entry.type, entry.adapterName or "DEFAULT", self.customEnvironment, true)

		if node ~= nil then
			link(entry.adapterNode, node)
		else
			Logging.xmlWarning(xmlFile, "Unable to find detached adapter for type '%s' in '%s'", entry.adapterName or "DEFAULT", hoseKey)
		end
	end

	local socketName = xmlFile:getValue(hoseKey .. "#socket")

	if socketName ~= nil then
		local socketColor = xmlFile:getValue(hoseKey .. "#socketColor", nil, true)
		entry.socket = g_connectionHoseManager:linkSocketToNode(socketName, entry.node, self.customEnvironment, socketColor)

		if entry.socket ~= nil then
			setRotation(entry.socket.node, 0, math.pi, 0)
		end
	end

	entry.objectChanges = {}

	ObjectChangeUtil.loadObjectChangeFromXML(xmlFile, hoseKey, entry.objectChanges, self.components, self)
	ObjectChangeUtil.setObjectChanges(entry.objectChanges, false)

	return true
end

function CabinPanels:getClonedHose(entry)
	local hose, startStraightening, endStraightening, minCenterPointAngle = g_connectionHoseManager:getClonedHoseNode(entry.type, entry.hoseType, entry.length, entry.diameter, entry.color, CabinPanels.MOD_NAME)

	if hose == nil then
		Logging.xmlWarning(self.xmlFile, "Unable to find connection hose with length '%.2f' and diameter '%.2f'", entry.length, entry.diameter)

		return false
	end

	local outgoingNode = g_connectionHoseManager:getSocketTarget(entry.socket, entry.node)
	local visibilityNode = hose
	local rx = 0
	local ry = 0
	local rz = 0

	if entry.outgoingAdapter ~= nil then
		local node, referenceNode = g_connectionHoseManager:getClonedAdapterNode(entry.type, entry.outgoingAdapter, self.customEnvironment)

		if node ~= nil then
			link(outgoingNode, node)

			outgoingNode = referenceNode
			visibilityNode = node
			ry = math.pi

			if entry.socket == nil then
				setRotation(node, 0, ry, 0)
			end
		else
			Logging.xmlWarning(self.xmlFile, "Unable to find adapter type '%s'", entry.outgoingAdapter)
		end
	end

	link(outgoingNode, hose)
	setTranslation(hose, 0, 0, 0)
	setRotation(hose, rx, ry, rz)

	entry.hoseNode = hose
	entry.visibilityNode = visibilityNode
	entry.startStraightening = startStraightening * entry.straighteningFactor
	entry.endStraightening = endStraightening
	entry.endStraighteningBase = endStraightening
	entry.endStraighteningDirectionBase = {
		0,
		0,
		1
	}
	entry.endStraighteningDirection = entry.endStraighteningDirectionBase
	entry.minCenterPointAngle = entry.minCenterPointAngle or minCenterPointAngle

	setVisibility(entry.visibilityNode, false)

	return true
end

function CabinPanels:loadCabinPanelHoseConnectors(hoses)
	self.xmlFile:iterate("vehicle.cabinPanels.hoseConnectors.hoseConnector", function(iPanel, key)

		if g_isCabinPanelsDevMode then
			Logging.info("1- loadCabinPanelHoseConnectors " .. tostring(key .. ".indoor") .. " - " .. tostring(self:getName()));
		end

		if not self.xmlFile:hasProperty(key .. ".indoor") or not self.xmlFile:hasProperty(key .. ".outdoor") then
			Logging.xmlError(self.xmlFile, string.format("Missing 'indoor' or 'outdoor' child element from %s", key));
			return;
		end

		local hose = {
			panel = nil,
			indoor = {inputAttacherJointIndices = {}},
			outdoor = {attacherJointIndices = {}}
		};

		if self:loadCabinPanelHoseConnector(self.xmlFile, key .. ".indoor", hose.indoor) and self:loadCabinPanelHoseConnector(self.xmlFile, key .. ".outdoor", hose.outdoor) then
			if g_isCabinPanelsDevMode then
				Logging.info("loadCabinPanelHoseConnector " .. tostring(key .. ".indoor") .. " - " .. tostring(key .. ".outdoor"));
			end

			table.insert(hoses, hose);
		end
	end);
end

function CabinPanels:switchInteractivePanelState(actionName, inputValue)
	local spec = self.spec_cabinPanels;

	if #self:getAttachedCabinPanels() > 0 and self:getActiveCamera().isInside then
		self:setInteractivePanelState(not spec.interactivePanelActive);
	end
end

function CabinPanels:setInteractivePanelState(state)
	local spec = self.spec_cabinPanels;

	spec.interactivePanelActive = state;

	if spec.interactivePanelActive then
		-- Add mouse click event
		local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.INTERACTIVE_PANEL_INTERACT, self, CabinPanels.onClick, false, true, false, true, nil);
		g_inputBinding:setActionEventTextVisibility(actionEventId, false);
	else
		--Remove mouse click event
		self:removeActionEvent(spec.actionEvents, InputAction.INTERACTIVE_PANEL_INTERACT);
	end
end

function CabinPanels:getDistance(node, node2)
	local x, y, z = getWorldTranslation(node);
	local x2, y2, z2 = getWorldTranslation(node2);

	return MathUtil.vector3Length(x - x2, y - y2, z - z2);
end

function CabinPanels:addPanel(panel, cabinPanelSlot, attachedVehicle)
	--local filename = Utils.getFilename(panel:getI3dFilename(), attachedVehicle.baseDirectory);
	local parentPanelId = g_i3DManager:loadSharedI3DFile(panel:getI3dFilenameWithBaseDirectory());
	local panelId = getChildAt(parentPanelId, 0);

	if panel:needToBeRotated() then
		setRotation(panelId, unpack(panel:getRotation()));
	end

	link(cabinPanelSlot.node, panelId);
	delete(parentPanelId);

	ObjectChangeUtil.loadObjectChangeFromXML(panel.xmlPanelFile, "panel.objectChangesOnAttach", panel.objectChanges, cabinPanelSlot.node, self);
	ObjectChangeUtil.setObjectChanges(panel.objectChanges, false);

	for iButton = 1, #panel.buttons do
		local button = panel.buttons[iButton];

		-- Import visual markers
		local markerNode = button:getMarkerNode();
		if markerNode ~= nil then
			local marker = g_i3DManager:loadSharedI3DFile(button:getMarkerFilename());
			local panelVisualMarkerId = I3DUtil.indexToObject(cabinPanelSlot.node, markerNode);
			local clonedMarker = clone(marker, false, false, false);

			setVisibility(panelVisualMarkerId, false);
			link(panelVisualMarkerId, clonedMarker);

			delete(marker);
		end

		-- Load buttons animations
		local animation = {};
		if self:loadCustomAnimation(panel.xmlPanelFile, button.buttonAnimationKey, button:getButtonAnimationName(), animation, cabinPanelSlot.node) then
			button.animation = animation;
		end
	end

	-- Link panel to cabinPanels list
	cabinPanelSlot.attachedPanel = panel;
	cabinPanelSlot.vehicle = attachedVehicle;
end

function CabinPanels:deleteAttachedPanel(cabinPanel)
	self:deleteChild(cabinPanel.node);

	cabinPanel.attachedPanel = nil;
	cabinPanel.vehicle = nil;
end

function CabinPanels:deleteChild(id)
	local numOfChildren = getNumOfChildren(id);

	if numOfChildren ~= 0 and numOfChildren ~= nil then
		for i = numOfChildren - 1, 0, -1 do
			delete(getChildAt(id, i));
		end
	end
end

function CabinPanels:setPanelsVisualMarkersVisibility(visibility)
	local spec = self.spec_cabinPanels;

	for iCabinPanel, cabinPanel in pairs(spec.cabinPanels) do
		local panel = cabinPanel.attachedPanel;

		if panel ~= nil then
			for iButton = 1, #panel.buttons do
				local button = panel.buttons[iButton];

				local markerNode = button:getMarkerNode();
				if markerNode ~= nil then
					setVisibility(I3DUtil.indexToObject(cabinPanel.node, markerNode), visibility);
				end
			end
		end
	end
end

function CabinPanels:hasAnyFreePanelSlot()
	return self:getFreeCabinPanelSlot() ~= nil;
end

function CabinPanels:getFreeCabinPanelSlot()
	local spec = self.spec_cabinPanels;

	for _, cabinPanel in pairs(spec.cabinPanels) do
		if cabinPanel.attachedPanel == nil then
			return cabinPanel;
		end
	end

	return nil;
end

function CabinPanels:getFreeCabinConnectionHoseSlot()
	local spec = self.spec_cabinPanels;

	for _, hose in pairs(spec.hoses) do
		if hose.panel == nil then
			return hose;
		end
	end

	return nil;
end

function CabinPanels:isAnyCabinPanelsAttached()
	for _, hose in pairs(self.spec_cabinPanels.hoses) do
		if hose.panel ~= nil then
			return true;
		end
	end

	return false;
end

function CabinPanels:getAttachedCabinPanels()
	local attachedCabinPanels = {}

	for _, cabinPanel in pairs(self.spec_cabinPanels.cabinPanels) do
		if cabinPanel.attachedPanel ~= nil then
			table.insert(attachedCabinPanels, cabinPanel);
		end
	end

	return attachedCabinPanels;
end

function CabinPanels:loadCustomAnimation(xmlFile, key, actionName, animation, components)
	if actionName ~= nil then
		animation.name = actionName;
		animation.parts = {};
		animation.currentTime = 0;
		animation.previousTime = 0;
		animation.currentSpeed = 1;
		animation.looping = xmlFile:getBool(key .. "#looping", false);
		animation.resetOnStart = Utils.getNoNil(xmlFile:getBool(key .. "#resetOnStart"), true);
		animation.soundVolumeFactor = xmlFile:getInt(key .. "#soundVolumeFactor") or 1;
		animation.isKeyframe = xmlFile:getBool(key .. "#isKeyframe") or false;

		if animation.isKeyframe then
			animation.curvesByNode = {}
		end

		xmlFile:iterate(string.format("%s.part", key), function(_, partKey)
			local animationPart = {};

			if not animation.isKeyframe then
				if self:loadAnimationPart(xmlFile, partKey, animationPart, animation, components) then
					table.insert(animation.parts, animationPart);
				end
			else
				self:loadStaticAnimationPart(xmlFile, partKey, animationPart, animation, components);
			end
		end)

		animation.partsReverse = {};

		for _, part in ipairs(animation.parts) do
			table.insert(animation.partsReverse, part);
		end

		table.sort(animation.parts, AnimatedVehicle.animPartSorter);
		table.sort(animation.partsReverse, AnimatedVehicle.animPartSorterReverse);
		self:initializeAnimationParts(animation);

		animation.currentPartIndex = 1;
		animation.duration = 0;

		for _, part in ipairs(animation.parts) do
			animation.duration = math.max(animation.duration, part.startTime + part.duration)
		end

		if animation.isKeyframe then
			for node, curve in pairs(animation.curvesByNode) do
				animation.duration = math.max(animation.duration, curve.maxTime)
			end
		end

		animation.startTime = xmlFile:getInt(key .. "#startAnimTime") or 0;
		animation.currentTime = animation.startTime * animation.duration

		if self.isClient then
			animation.samples = {}

			local i = 0;
			local soundKey = string.format("sound(%d)", i);
			local baseKey = key .. "." .. soundKey;

			while xmlFile:hasProperty(baseKey) do
				local sample = g_soundManager:loadSampleFromXML(xmlFile, key, soundKey, self.baseDirectory, components or self.components, 0, AudioGroup.VEHICLE, self.i3dMappings, self)

				if sample ~= nil then
					sample.startTime = xmlFile:getValue(baseKey .. "#startTime", 0)
					sample.endTime = xmlFile:getValue(baseKey .. "#endTime")
					sample.direction = xmlFile:getValue(baseKey .. "#direction", 0)

					if sample.endTime == nil and sample.loops == 0 then
						sample.loops = 1
					end

					sample.volumeScale = sample.volumeScale * animation.soundVolumeFactor

					table.insert(animation.samples, sample)
				end

				i = i + 1;
				soundKey = string.format("sound(%d)", i);
				baseKey = key .. "." .. soundKey;
			end

			animation.eventSamples = {
				stopTimePos = {},
				stopTimeNeg = {}
			};

			xmlFile:iterate(key .. ".stopTimePosSound", function (index, _)
				local sample = g_soundManager:loadSampleFromXML(xmlFile, key, string.format("stopTimePosSound(%d)", index - 1), self.baseDirectory, components or self.components, 1, AudioGroup.VEHICLE, self.i3dMappings, self);

				if sample ~= nil then
					table.insert(animation.eventSamples.stopTimePos, sample);
				end
			end);

			xmlFile:iterate(key .. ".stopTimeNegSound", function (index, _)
				local sample = g_soundManager:loadSampleFromXML(xmlFile, key, string.format("stopTimeNegSound(%d)", index - 1), self.baseDirectory, components or self.components, 1, AudioGroup.VEHICLE, self.i3dMappings, self);

				if sample ~= nil then
					table.insert(animation.eventSamples.stopTimeNeg, sample);
				end
			end);
		end

		return true;
	end

	return false;
end

function CabinPanels:playCustomAnimation(animation, speed, animTime, noEventSend)
	local spec = self.spec_animatedVehicle;

	if animation ~= nil then
		SpecializationUtil.raiseEvent(self, "onPlayAnimation", animation.name);

		if speed == nil then
			speed = animation.currentSpeed;
		end

		if speed == nil or speed == 0 then
			return;
		end

		if animTime == nil then
			if self:getIsAnimationPlaying(name) then
				animTime = self:getAnimationTime(name);
			elseif speed > 0 then
				animTime = 0;
			else
				animTime = 1;
			end
		end

		if noEventSend == nil or noEventSend == false then
			local animatedVehicleStartEvent = AnimatedVehicleStartEvent:new(self, animation.name, speed, animTime);

			if g_server ~= nil then
				g_server:broadcastEvent(animatedVehicleStartEvent, nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(animatedVehicleStartEvent);
			end
		end

		if spec.activeAnimations[animation.name] == nil then
			spec.activeAnimations[animation.name] = animation;
			spec.numActiveAnimations = spec.numActiveAnimations + 1;

			SpecializationUtil.raiseEvent(self, "onStartAnimation", animation.name);
		end

		animation.currentSpeed = speed;
		animation.currentTime = animTime * animation.duration;

		self:resetAnimationValues(animation);

		if self.isClient then
			g_soundManager:playSample(animation.sample);
		end

		self:raiseActive();
	end
end