--
-- AttachablePanels
--
-- @author E.T.A La Marchoise
-- @date 18/01/2021
-- @version 1.0.0.0
--
-- Copyright (C) E.T.A La Marchoise, Confidential, All Rights Reserved.

local modDirectory = g_currentModDirectory;

source(modDirectory .. "scripts/model/panel.lua"); -- Load Panel class

AttachablePanels = {};
AttachablePanels.MOD_DIRECTORY = modDirectory;

AttachablePanels.MOD_NAME = g_currentModName;
AttachablePanels.xmlSchema = nil;

function AttachablePanels.prerequisitesPresent(specializations)
	return true;
end

function AttachablePanels.initSpecialization()
	local schema = Vehicle.xmlSchema
	AttachablePanels.registerTargetHoseConnectionsXMLPaths(schema, "vehicle.attachablePanels.panelConnectionHoses.hose(?)");
	schema:register(XMLValueType.STRING, "vehicle.attachablePanels.attachablePanel(?)#filename", "Filename to xml panel", nil, true);
	schema:register(XMLValueType.NODE_INDEX, "vehicle.attachablePanels.attachablePanel(?)#detachedNode", "Detached node", nil);
	schema:register(XMLValueType.VECTOR_ROT, "vehicle.attachablePanels.attachablePanel(?).rotation", "Detached node rotation", "0 0 0");
	schema:setXMLSpecializationType();

	local panelXMLSchema = XMLSchema.new("attachablePanel");

	panelXMLSchema:register(XMLValueType.STRING, "panel.filename", "Path to i3d file", nil, true);

	AttachablePanels.registerPanelHoseTargetNodesXMLPath(panelXMLSchema, "panel.connectionHoses.hose(?)");
	AttachablePanels.registerPanelButtonsXMLPath(panelXMLSchema, "panel.interactivePanel.buttons.button(?)");

	AttachablePanels.xmlSchema = panelXMLSchema;
end

function AttachablePanels.registerTargetHoseConnectionsXMLPaths(schema, basePath)
	schema:register(XMLValueType.NODE_INDEX, basePath .. "#node", "Hose output node");
	schema:register(XMLValueType.STRING, basePath .. "#type", "Hose type")
	schema:register(XMLValueType.STRING, basePath .. "#specType", "Connection hose specialization type (if defined it needs to match the type of the other tool)")
	schema:register(XMLValueType.FLOAT, basePath .. "#straighteningFactor", "Straightening Factor", 1)
	schema:register(XMLValueType.FLOAT, basePath .. "#length", "Hose length", 3)
	schema:register(XMLValueType.VECTOR_3, basePath .. "#straighteningDirection", "Straightening direction", "0 0 1")
	schema:register(XMLValueType.STRING, basePath .. "#socket", "Socket name to load")
	schema:register(XMLValueType.COLOR, basePath .. "#socketColor", "Socket custom color")
	schema:register(XMLValueType.STRING, basePath .. "#adapterType", "Adapter type to use", "DEFAULT")
	ObjectChangeUtil.registerObjectChangeXMLPaths(schema, basePath);
end

function AttachablePanels.registerPanelHoseTargetNodesXMLPath(schema, basePath)
	schema:register(XMLValueType.STRING, basePath .. "#inputNode", "Target node")
	schema:register(XMLValueType.FLOAT, basePath .. "#diameter", "Hose diameter")
	schema:register(XMLValueType.STRING, basePath .. "#type", "Hose type")
	schema:register(XMLValueType.STRING, basePath .. "#specType", "Connection hose specialization type (if defined it needs to match the type of the other tool)")
	schema:register(XMLValueType.FLOAT, basePath .. "#straighteningFactor", "Straightening Factor", 1)
	schema:register(XMLValueType.VECTOR_3, basePath .. "#straighteningDirection", "Straightening direction", "0 0 1")
	schema:register(XMLValueType.STRING, basePath .. "#socket", "Socket name to load")
	schema:register(XMLValueType.COLOR, basePath .. "#socketColor", "Socket custom color")
	schema:register(XMLValueType.STRING, basePath .. "#adapterType", "Adapter type to use", "DEFAULT")

	ObjectChangeUtil.registerObjectChangeXMLPaths(schema, basePath)
end

function AttachablePanels.registerPanelButtonsXMLPath(schema, basePath)
	schema:register(XMLValueType.NODE_INDEX, basePath .. "#node", "Button node", nil, true);
	schema:register(XMLValueType.NODE_INDEX, basePath .. "#markerNode", "Visual marker node", nil);
	schema:register(XMLValueType.FLOAT, basePath .. "#radius", "Button radius", InteractivePanelButton.DEFAULT_RADIUS, false);
	schema:register(XMLValueType.STRING, basePath .. ".action", "Action name", nil);
	schema:register(XMLValueType.STRING, basePath .. ".animation", "Animation name", nil);
	schema:register(XMLValueType.L10N_STRING, basePath .. "#activate", "Text to display when button will active action", nil, false);
	schema:register(XMLValueType.L10N_STRING, basePath .. "#deactivate", "Text to display when button will deactive action", nil, false);
	schema:register(XMLValueType.STRING, basePath .. "#markerFilename", "Filename of i3d file to use as a custom marker", nil);

	-- Load buttons animations
	AnimatedVehicle.registerAnimationXMLPaths(schema, basePath .. ".buttonAnimation");
	AttachablePanels.registerAnimationXMLPaths(schema, basePath .. ".buttonAnimation");
end

function AttachablePanels.registerAnimationXMLPaths(schema, basePath)
	schema:register(XMLValueType.ANGLE, basePath .. ".part(?)#startSteeringAngle", "Start steering angle")
	schema:register(XMLValueType.ANGLE, basePath .. ".part(?)#endSteeringAngle", "End steering angle")
	schema:register(XMLValueType.FLOAT, basePath .. ".part(?)#startBrakeFactor", "Start brake force factor")
	schema:register(XMLValueType.FLOAT, basePath .. ".part(?)#endBrakeFactor", "End brake force factor")
end

function AttachablePanels.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "getAttachablePanels", AttachablePanels.getAttachablePanels);
	SpecializationUtil.registerFunction(vehicleType, "getTargetHoseConnections", AttachablePanels.getTargetHoseConnections);
	SpecializationUtil.registerFunction(vehicleType, "getFreeTargetHoseConnection", AttachablePanels.getFreeTargetHoseConnection);
	SpecializationUtil.registerFunction(vehicleType, "getPanelAttacherVehicle", AttachablePanels.getPanelAttacherVehicle);
	SpecializationUtil.registerFunction(vehicleType, "setAttacherVehicle", AttachablePanels.setAttacherVehicle);

	SpecializationUtil.registerFunction(vehicleType, "hideDetachedPanels", AttachablePanels.hideDetachedPanels);
	SpecializationUtil.registerFunction(vehicleType, "setDetachedPanelVisibility", AttachablePanels.setDetachedPanelVisibility);

	SpecializationUtil.registerFunction(vehicleType, "onPanelButtonClicked", AttachablePanels.onPanelButtonClicked);
	SpecializationUtil.registerFunction(vehicleType, "loadHoseTargetConnectorNode", AttachablePanels.loadHoseTargetConnectorNode);

	SpecializationUtil.registerFunction(vehicleType, "onTurnEvent", AttachablePanels.onTurnEvent);
	SpecializationUtil.registerFunction(vehicleType, "spawnPanel", AttachablePanels.spawnPanel);
	SpecializationUtil.registerFunction(vehicleType, "setButtonStateForAction", AttachablePanels.setButtonStateForAction);
	SpecializationUtil.registerFunction(vehicleType, "forEachButton", AttachablePanels.forEachButton);
end

function AttachablePanels.registerEventListeners(vehicleType)
  	SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", AttachablePanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onLoad", AttachablePanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", AttachablePanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onPostAttach", AttachablePanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onPreDetach", AttachablePanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onPostDetach", AttachablePanels);

  	SpecializationUtil.registerEventListener(vehicleType, "onTurnedOn", AttachablePanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onTurnedOff", AttachablePanels);

  	SpecializationUtil.registerEventListener(vehicleType, "onFoldStateChanged", AttachablePanels);

  	SpecializationUtil.registerEventListener(vehicleType, "onSetLowered", AttachablePanels);

  	SpecializationUtil.registerEventListener(vehicleType, "onStartAnimation", AttachablePanels);
  	SpecializationUtil.registerEventListener(vehicleType, "onStopAnimation", AttachablePanels);

  	SpecializationUtil.registerEventListener(vehicleType, "onDischargeStateChanged", AttachablePanels);
end

function AttachablePanels:onPreLoad(savegame)
	self.spec_attachablePanels = self[("spec_%s.attachablePanels"):format(AttachablePanels.MOD_NAME)];
end

function AttachablePanels:onLoad(savegame, parentDirectory)
	local spec = self.spec_attachablePanels;

	if g_isCabinPanelsDevMode then
		Logging.info(string.format("AttachablePanels.OnLoad event from %s", self.xmlFile:getFilename()));
	end

	spec.attachablePanels = {};
	spec.targetHoseConnections = {}

	-- Load attachable panels
	self.xmlFile:iterate("vehicle.attachablePanels.attachablePanel", function(iPanel, key)
		---@type Panel
		local panel = Panel:new(self.xmlFile, key, iPanel, self.components, self.i3dMappings, parentDirectory or self.baseDirectory, self.customEnvironment);

		if panel ~= nil then
			table.insert(spec.attachablePanels, panel);
		end
	end);

	-- Load implement connection hose nodes
	self.xmlFile:iterate("vehicle.attachablePanels.panelConnectionHoses.hose", function (_, hoseKey)
		local entry = {};

		if self:loadHoseTargetConnectorNode(self.xmlFile, hoseKey, entry) then
			table.insert(spec.targetHoseConnections, {
				isAttached = false,
				hose = entry
			});
		end
	end);
end

function AttachablePanels:onPostLoad()
	for iPanel, panel in pairs(self:getAttachablePanels()) do
		if panel:hasDetachedPosition() then
			self:spawnPanel(panel);
		end
	end
end

function AttachablePanels:spawnPanel(panel, noEventSend)
	local panelId = g_i3DManager:loadSharedI3DFile(panel:getI3dFilenameWithBaseDirectory());

	link(panel:getDetachedNode(), panelId);
end

function AttachablePanels:onPostAttach(attacherVehicle, inputJointDescIndex, jointDescIndex)
	if attacherVehicle.spec_cabinPanels == nil then
		-- Hide panels due to incompatible attacher vehicle
		self:hideDetachedPanels();
	else
		self:setAttacherVehicle(attacherVehicle);
	end
end

function AttachablePanels:onPostDetach(attacherVehicle, inputJointDescIndex, jointDescIndex)
	for iTargetHoseConnection, targetHoseConnection in pairs(self:getTargetHoseConnections()) do
		targetHoseConnection.isAttached = false;
	end
end

function AttachablePanels:getAttachablePanels()
	return self.spec_attachablePanels.attachablePanels;
end

function AttachablePanels:getTargetHoseConnections()
	return self.spec_attachablePanels.targetHoseConnections;
end

function AttachablePanels:getFreeTargetHoseConnection()
	for iTargetHoseConnection, targetHoseConnection in pairs(self:getTargetHoseConnections()) do
		if not targetHoseConnection.isAttached then
			return targetHoseConnection;
		end
	end
end

function AttachablePanels:getPanelAttacherVehicle()
	return self.attacherVehicle;
end

function AttachablePanels:setAttacherVehicle(attacherVehicle)
	self.attacherVehicle = attacherVehicle;
end

function AttachablePanels:hideDetachedPanels()
	for iPanel, panel in pairs(self:getAttachablePanels()) do
		if panel:hasDetachedPosition() then
			setVisibility(panel:getDetachedNode(), false);
		end
	end
end

function AttachablePanels:onPreDetach()
	local attacherVehicle = self:getPanelAttacherVehicle();
	if attacherVehicle ~= nil and attacherVehicle.spec_cabinPanels ~= nil then
		self:setAttacherVehicle(nil);
	end
end

function AttachablePanels:onDischargeStateChanged(state)
	local attacherVehicle = self:getPanelAttacherVehicle();

	if attacherVehicle == nil or attacherVehicle.setButtonState == nil then
		return;
	end

	self:setButtonStateForAction(ButtonAction.TIP_TO_GROUND, function(button)
		attacherVehicle:setButtonState(button, state == Dischargeable.DISCHARGE_STATE_GROUND or (not state == Dischargeable.DISCHARGE_STATE_OFF or state == nil));
	end);

	self:setButtonStateForAction(ButtonAction.TIP_TO_OBJECT, function(button)
		attacherVehicle:setButtonState(button, state == Dischargeable.DISCHARGE_STATE_OBJECT or (not state == Dischargeable.DISCHARGE_STATE_OFF or state == nil));
	end);
end

function AttachablePanels:onFoldStateChanged(direction, moveToMiddle)
	if g_isCabinPanelsDevMode then
		Logging.info(string.format("onFoldStateChanged | direction : %s | moveToMiddle : %s", tostring(direction), tostring(moveToMiddle)))
	end

	local attacherVehicle = self:getPanelAttacherVehicle();
	if attacherVehicle == nil then
		return;
	end

	local isUnfolding = moveToMiddle and direction == -1;
	local isFolding = not moveToMiddle and direction == 1;

	if isUnfolding or isFolding then
		self:setButtonStateForAction(ButtonAction.FOLD_UNFOLD, function(button)
			attacherVehicle:setButtonState(button, direction == -1);
		end);
	end

	-- Lower implement when tool is folded
	if direction == -1 and not moveToMiddle then
		self:setButtonStateForAction(ButtonAction.FOLD_UNFOLD, function(button)
			attacherVehicle:setButtonState(button, true);
		end);
	end
end

function AttachablePanels:onSetLowered(lowered)
	local attacherVehicle = self:getPanelAttacherVehicle();
	if attacherVehicle == nil then
		return;
	end

	self:setButtonStateForAction(ButtonAction.RAISE_DOWN, function(button)
		attacherVehicle:setButtonState(button, lowered);
	end);
end

function AttachablePanels:onStartAnimation(animationName, speed)
	local attacherVehicle = self:getPanelAttacherVehicle();
	if attacherVehicle == nil then
		return;
	end

	self:forEachButton(function(button)
		if button:getActionAnimationName() == animationName then
			attacherVehicle:setButtonState(button, speed > 0, true);
		end
	end);
end

function AttachablePanels:onStopAnimation(animationName)
	local attacherVehicle = self:getPanelAttacherVehicle();
	if attacherVehicle == nil then
		return;
	end

	self:forEachButton(function(button)
		if button:getActionAnimationName() == animationName then
			attacherVehicle:setButtonState(button, -1, true);
		end
	end);
end

function AttachablePanels:onTurnedOn()
	self:onTurnEvent(true);
end

function AttachablePanels:onTurnedOff()
	self:onTurnEvent(false);
end

function AttachablePanels:onTurnEvent(state)
	local attacherVehicle = self:getPanelAttacherVehicle();
	if attacherVehicle == nil or attacherVehicle.spec_cabinPanels == nil then
		return;
	end

	self:setButtonStateForAction(ButtonAction.ACTIVATE_DEACTIVATE, function(button)
		attacherVehicle:setButtonState(button, state);
	end);
end

function AttachablePanels:setButtonStateForAction(action, callback)
	self:forEachButton(function(button, iButton, cabinPanel, iPanel)
		if button:getAction() == action and callback ~= nil then
			callback(button, iButton, cabinPanel, iPanel);
		end
	end);
end

function AttachablePanels:forEachButton(callback)
	local attacherVehicle = self:getPanelAttacherVehicle();
	if attacherVehicle == nil or attacherVehicle.spec_cabinPanels == nil then
		return;
	end

	local cabinPanels = attacherVehicle:getAttachedCabinPanels();
	for iCabinPanel, cabinPanel in pairs(cabinPanels) do
		local panel = cabinPanel.attachedPanel

		if panel ~= nil then
			for iButton, button in pairs(panel.buttons) do
				if callback ~= nil then
					callback(button, iButton, cabinPanel, iCabinPanel);
				end
			end
		end
	end
end

function AttachablePanels:onPanelButtonClicked(button)
	local actionAnimationName = button:getActionAnimationName();

	if actionAnimationName ~= nil then
		if self:getAnimationExists(actionAnimationName) then -- play animation
			local speed = button:isActive() and -1 or 1; -- -1 if button is active because button state is not switched yet

			self:playAnimation(actionAnimationName, speed, nil, true);
		end
	end

	if button.action ~= nil then
		if button.action == ButtonAction.FOLD_UNFOLD and self.spec_foldable ~= nil and Foldable.actionEventFold ~= nil then
			Foldable.actionEventFold(self);
		elseif button.action == ButtonAction.RAISE_DOWN and self.spec_foldable ~= nil and Foldable.actionEventFoldMiddle ~= nil then
			Foldable.actionEventFoldMiddle(self);
		elseif button.action == ButtonAction.ACTIVATE_DEACTIVATE and self.spec_turnOnVehicle ~= nil and TurnOnVehicle.actionEventTurnOn ~= nil then
			TurnOnVehicle.actionEventTurnOn(self);
		elseif button.action == ButtonAction.TIP_TO_GROUND and self.spec_dischargeable ~= nil and Dischargeable.actionEventToggleDischargeToGround ~= nil then
			Dischargeable.actionEventToggleDischargeToGround(self);
		elseif button.action == ButtonAction.TIP_TO_OBJECT and self.spec_dischargeable ~= nil and Dischargeable.actionEventToggleDischarging ~= nil then
			Dischargeable.actionEventToggleDischarging(self);
		end
	end
end

function AttachablePanels:setDetachedPanelVisibility(panel, visibility)
	if panel ~= nil and panel:hasDetachedPosition() then
		setVisibility(panel:getDetachedNode(), visibility);
	end
end

function AttachablePanels:loadHoseTargetConnectorNode(xmlFile, hoseKey, entry)
	entry.attacherJointIndices = {}

	entry.node = xmlFile:getValue(hoseKey .. "#node", nil, self.components, self.i3dMappings)
	entry.type = xmlFile:getValue(hoseKey .. "#type")
	entry.specType = xmlFile:getValue(hoseKey .. "#specType")
	entry.length = xmlFile:getValue(hoseKey .. "#length")
	entry.straighteningFactor = xmlFile:getValue(hoseKey .. "#straighteningFactor", 1)
	entry.straighteningDirection = xmlFile:getValue(hoseKey .. "#straighteningDirection", nil, true)
	local socketName = xmlFile:getValue(hoseKey .. "#socket")

	if socketName ~= nil then
		local socketColor = xmlFile:getValue(hoseKey .. "#socketColor", nil, true)
		entry.socket = g_connectionHoseManager:linkSocketToNode(socketName, entry.node, self.customEnvironment, socketColor)
	end

	if entry.type == nil then
		Logging.xmlWarning(xmlFile, "Missing type for '%s'", hoseKey)

		return false
	end

	entry.adapterName = xmlFile:getValue(hoseKey .. "#adapterType", "DEFAULT")

	if entry.node ~= nil and entry.adapter == nil then
		entry.adapter = {
			node = entry.node,
			refNode = entry.node
		}
	end

	entry.objectChanges = {}

	ObjectChangeUtil.loadObjectChangeFromXML(xmlFile, hoseKey, entry.objectChanges, self.components, self)
	ObjectChangeUtil.setObjectChanges(entry.objectChanges, false)

	return true
end