WorkCameraMain = {}

local WorkCameraMain_mt = Class(WorkCameraMain)

WorkCameraMain.TOP_LEFT = 1
WorkCameraMain.TOP_RIGHT = 2
WorkCameraMain.BOTTOM_RIGHT = 3
WorkCameraMain.BOTTOM_LEFT = 4

WorkCameraMain.DEFAUL_FOV = 80
WorkCameraMain.DEFAULT_NEAR_CLIP = 0.01
WorkCameraMain.DEFAULT_FAR_CLIP = 100

WorkCameraMain.ASPECT_RATIO = 1.6

WorkCameraMain.DEFAULT_SETTINGS = {
    relCorner = WorkCameraMain.TOP_RIGHT,
    position = {
        x = 0,
        y = 120
    },
    size = {
        width = 525,
        height = 328
    }
}

WorkCameraMain.actions = {{
    name = InputAction.WORK_CAMERA_TURN_ON_OFF,
    text = g_i18n:getText("action_turn_on"),
    priority = GS_PRIO_HIGH
}, {
    name = InputAction.WORK_CAMERA_CHANGE_CAMERA,
    text = g_i18n:getText("action_change_camera"),
    priority = GS_PRIO_HIGH
}, {
    name = InputAction.WORK_CAMERA_CHANGE_CAMERA_PREVIOUS,
    text = g_i18n:getText("action_change_camera_previous"),
    priority = GS_PRIO_HIGH
}, {
    name = InputAction.WORK_CAMERA_MENU,
    priority = GS_PRIO_HIGH
}}

function WorkCameraMain:new(modDirectory, modName)
    local self = {}

    setmetatable(self, WorkCameraMain_mt)

    self.modDirectory = modDirectory
    self.modName = modName

    self.actionEvents = {}
    self.itemsWorkCameras = {}
    self.currentCameraIndex = 1

    self:resetGeneralSettings()

    return self
end

function WorkCameraMain:onMissionLoaded(mission)
    -- register schemas
    self:registerItemCamerasSchema()
    self:registerGeneralSettingsSchema()

    -- modSettings folder
    local WORK_CAMERA_FOLDER = getUserProfileAppPath() .. "modSettings/FS22_WorkCamera/"
    createFolder(WORK_CAMERA_FOLDER)

    -- load in-game and custom cameras
    self:loadWorkCameras()

    -- general settings
    local GENERAL_SETTINGS_XML = Utils.getFilename("WorkCameraGeneralSettings.xml", WORK_CAMERA_FOLDER)
    local DEFAUL_GENERAL_SETTINGS_XML = Utils.getFilename("config/DefaultGeneralSettings.xml", self.modDirectory)

    if fileExists(GENERAL_SETTINGS_XML) then
        self:loadGeneralSettingsFromXml(GENERAL_SETTINGS_XML)
    else
        copyFile(DEFAUL_GENERAL_SETTINGS_XML, GENERAL_SETTINGS_XML, false)
    end

    -- add console command
    addConsoleCommand("wcCustomCamerasReload", "Reload custom cameras from mod settings", "consoleCustomCamerasReload",
        WorkCameraMain)

    -- hud
    self.cameraElement = CameraElement.new(self)

    -- menu
    local generalSettingFrame = WorkCameraGeneralSettingsFrame.new(self, self.modName)
    g_gui:loadGui(self.modDirectory .. "ui/WorkCameraGeneralSettingsFrame.xml", "WorkCameraGeneralSettingsFrame",
        generalSettingFrame, true)

    local cameraSettingFrame = WorkCameraCameraSettingsFrame.new(self, self.modName)
    g_gui:loadGui(self.modDirectory .. "ui/WorkCameraCameraSettingsFrame.xml", "WorkCameraCameraSettingsFrame",
        cameraSettingFrame, true)

    self.menu = WorkCameraTabbedMenu:new(g_messageCenter, g_i18n, g_inputBinding, self, self.modName)
    g_gui:loadGui(self.modDirectory .. "ui/WorkCameraTabbedMenu.xml", "WorkCameraTabbedMenu", self.menu)
end

function WorkCameraMain.installSpecializations(vehicleTypeManager, specializationManager, modDirectory, modName)
    specializationManager:addSpecialization("workCamera", "WorkCamera",
        Utils.getFilename("scripts/WorkCamera.lua", modDirectory), nil) -- Nil is important here

    if specializationManager:getSpecializationByName("workCamera") == nil then
        print("ERROR: workCamera | Failed to add specialization")
    else
        for typeName, typeEntry in pairs(vehicleTypeManager:getTypes()) do
            vehicleTypeManager:addSpecialization(typeName, modName .. ".workCamera")
        end
    end
end

function WorkCameraMain:onEnterVehicle()
    if g_currentMission:getIsClient() then
        self:updateAvailableCameras()
    end
end

function WorkCameraMain:onLeaveVehicle()
    if g_currentMission:getIsClient() then
        self.cameraElement:setCamera(nil)
        self:updateTurnOnOffInputBinding()
        self:updateChangeCameraInputBinding()
        self:updateMenuInputBinding()
    end
end

function WorkCameraMain:updateItemAvailableCameras(item, itemToIgnore)
    self:addItemCamerasToList(item)

    if item.getAttachedImplements ~= nil then
        for _, implement in pairs(item:getAttachedImplements()) do
            if implement.object ~= nil and implement.object ~= itemToIgnore then
                self:updateItemAvailableCameras(implement.object, itemToIgnore)
            end
        end
    end
end

function WorkCameraMain:updateAvailableCameras(itemToIgnore)
    self.itemsWorkCameras = {}
    self.currentCameraIndex = nil

    local vehicle = g_currentMission.controlledVehicle

    if vehicle ~= nil then
        self:updateItemAvailableCameras(vehicle, itemToIgnore)
    end

    if #self.itemsWorkCameras > 0 then
        if self.currentCameraIndex == nil then
            self.currentCameraIndex = 1
            local next = self.itemsWorkCameras[self.currentCameraIndex]
            next.item:setCurrentWorkCamera(next.camera)
        end

        self:updateCameraElement()
        self.cameraElement:setEnabled(true)
    else
        self.cameraElement:setCamera(nil)
        self.cameraElement:setEnabled(false)
    end

    self:updateTurnOnOffInputBinding()
    self:updateChangeCameraInputBinding()
    self:updateMenuInputBinding()
end

function WorkCameraMain:addItemCamerasToList(item)
    for _, camera in pairs(item:getWorkCameras()) do
        if item:getIsWorkCameraEnabled() then
            table.insert(self.itemsWorkCameras, {
                camera = camera,
                item = item
            })
            if self.currentCameraIndex == nil and camera == item:getCurrentWorkCamera() then
                self.currentCameraIndex = #self.itemsWorkCameras
            end
        end
    end
end

function WorkCameraMain:registerActionEvents(inputManager)
    self:unregisterActionEvents()

    for _, action in pairs(WorkCameraMain.actions) do
        local actionName = action.name
        local triggerUp = action.triggerUp ~= nil and action.triggerUp or false
        local triggerDown = action.triggerDown ~= nil and action.triggerDown or true
        local triggerAlways = action.triggerAlways ~= nil and action.triggerAlways or false
        local startActive = action.startActive ~= nil and action.startActive or true

        local _, eventId = g_inputBinding:registerActionEvent(actionName, self, WorkCameraMain.onActionCall, triggerUp,
            triggerDown, triggerAlways, startActive, nil)

        self.actionEvents[actionName] = eventId

        if g_inputBinding ~= nil and g_inputBinding.events ~= nil and g_inputBinding.events[eventId] ~= nil then
            if action.priority ~= nil then
                g_inputBinding:setActionEventTextPriority(eventId, action.priority)
            end

            if action.text ~= nil then
                g_inputBinding:setActionEventText(eventId, action.text)
            end
        end
    end

    self:updateTurnOnOffInputBinding()
    self:updateChangeCameraInputBinding()
    self:updateMenuInputBinding()
end

function WorkCameraMain:unregisterActionEvents()
    g_inputBinding:removeActionEventsByTarget(self)
end

function WorkCameraMain:onActionCall(actionName, keyStatus, callbackStatus, isAnalog, arg6)
    if actionName == InputAction.WORK_CAMERA_TURN_ON_OFF then
        self:turnOnOffItemCameras()
    elseif actionName == InputAction.WORK_CAMERA_CHANGE_CAMERA then
        self:switchCurrentCamera(1)
    elseif actionName == InputAction.WORK_CAMERA_CHANGE_CAMERA_PREVIOUS then
        self:switchCurrentCamera(-1)
    elseif actionName == InputAction.WORK_CAMERA_MENU then
        g_gui:showDialog("WorkCameraTabbedMenu")
    end
end

function WorkCameraMain:turnOnOffItemCameras()
    local vehicle = g_currentMission.controlledVehicle
    if vehicle ~= nil then
        local item = self:getSelectedItem()

        if #item:getWorkCameras() > 0 then
            item:toggleIsWorkCameraEnabled()
            self:updateAvailableCameras()
        end
    end
end

function WorkCameraMain:getSelectedItemRecursive(item)
    if item.getSelectedImplement == nil then
        return nil
    end

    local implement = item:getSelectedImplement()
    if implement ~= nil then
        return implement.object
    end

    if item.getAttachedImplements == nil then
        return nil
    end

    for k, implement in pairs(item:getAttachedImplements()) do
        local selected = self:getSelectedItemRecursive(implement.object)
        if selected ~= nil then
            return selected
        end
    end
end

function WorkCameraMain:getSelectedItem()
    local vehicle = g_currentMission.controlledVehicle
    if vehicle ~= nil then
        local implement = self:getSelectedItemRecursive(vehicle)
        if implement ~= nil then
            return implement
        end

        return vehicle
    end

    return nil
end

function WorkCameraMain:getSelectedCameraItem()
    if #self.itemsWorkCameras > 0 then
        return self.itemsWorkCameras[self.currentCameraIndex].item
    end

    return nil
end

function WorkCameraMain:switchCurrentCamera(direction)
    if direction == nil then
        direction = 1
    end

    local previous = self.itemsWorkCameras[self.currentCameraIndex]
    self.currentCameraIndex = self.currentCameraIndex + direction
    if self.currentCameraIndex > #self.itemsWorkCameras then
        self.currentCameraIndex = 1
    elseif self.currentCameraIndex < 1 then
        self.currentCameraIndex = #self.itemsWorkCameras
    end

    local next = self.itemsWorkCameras[self.currentCameraIndex]

    if previous.item == next.item then
        next.item:setCurrentWorkCamera(next.camera)
    else
        previous.item:setCurrentWorkCamera(nil)
        next.item:setCurrentWorkCamera(next.camera)
    end

    self:updateCameraElement()
end

function WorkCameraMain:updateCameraElement()
    local item = self.itemsWorkCameras[self.currentCameraIndex]
    if item ~= nil then
        self.cameraElement:setCamera(item.camera)
    end
end

function WorkCameraMain:updateTurnOnOffInputBinding()
    local actionEventId = self.actionEvents[InputAction.WORK_CAMERA_TURN_ON_OFF]

    local item = WorkCameraMain:getSelectedItem()

    local isActive = item ~= nil and #item:getWorkCameras() > 0
    g_inputBinding:setActionEventActive(actionEventId, isActive)

    if isActive then
        local eventText = item:getIsWorkCameraEnabled() and "action_turn_off" or "action_turn_on"
        g_inputBinding:setActionEventText(actionEventId, g_i18n:getText(eventText))
    end
end

function WorkCameraMain:updateChangeCameraInputBinding()
    local isActive = WorkCameraMain:getSelectedItem() and #self.itemsWorkCameras > 1

    local actionEventId = self.actionEvents[InputAction.WORK_CAMERA_CHANGE_CAMERA]
    g_inputBinding:setActionEventActive(actionEventId, isActive)
    actionEventId = self.actionEvents[InputAction.WORK_CAMERA_CHANGE_CAMERA_PREVIOUS]
    g_inputBinding:setActionEventActive(actionEventId, isActive)
end

function WorkCameraMain:updateMenuInputBinding()
    local actionEventId = self.actionEvents[InputAction.WORK_CAMERA_MENU]
    local isActive = WorkCameraMain:getSelectedItem() and #self.itemsWorkCameras > 0
    g_inputBinding:setActionEventActive(actionEventId, isActive)
end

function WorkCameraMain:update(dt)
    self.cameraElement:update(dt)
end

function WorkCameraMain:draw()
    self.cameraElement:draw()
end

function WorkCameraMain:delete()
    self.cameraElement:delete()

    removeConsoleCommand("wcCustomCamerasReload")
end

function WorkCameraMain:setRelCorner(relCorner)
    if relCorner >= 1 and relCorner <= 4 then
        self.generalSettings.relCorner = relCorner
    end
end

function WorkCameraMain:onUpdateGeneralSettings()
    self.cameraElement:updateHudSize()
    self.cameraElement:updateHudPosition()
    self:updateAvailableCameras()

    -- save to XML
    self:saveGeneralSettingsToXML()
end

function WorkCameraMain:saveGeneralSettingsToXML()
    local WORK_CAMERA_FOLDER = getUserProfileAppPath() .. "modSettings/FS22_WorkCamera/"
    local GENERAL_SETTINGS_XML = Utils.getFilename("WorkCameraGeneralSettings.xml", WORK_CAMERA_FOLDER)

    local schemaKey = "workCameraGeneralSettings"

    local xmlFile = XMLFile.create("WorkCameraGeneralSettingsXML", GENERAL_SETTINGS_XML, "workCameraGeneralSettings",
        self.generalSettingSchema)

    if xmlFile ~= 0 then
        xmlFile:setValue(schemaKey .. ".position#x", self.generalSettings.position.x)
        xmlFile:setValue(schemaKey .. ".position#y", self.generalSettings.position.y)
        xmlFile:setValue(schemaKey .. ".size#width", self.generalSettings.size.width)
        xmlFile:setValue(schemaKey .. ".size#height", self.generalSettings.size.height)
        xmlFile:setValue(schemaKey .. ".relCorner#value", self.generalSettings.relCorner)

        xmlFile:save()
        xmlFile:delete()
    end
end

function WorkCameraMain:resetGeneralSettings()
    self.generalSettings = {
        relCorner = WorkCameraMain.DEFAULT_SETTINGS.relCorner,
        position = {
            x = WorkCameraMain.DEFAULT_SETTINGS.position.x,
            y = WorkCameraMain.DEFAULT_SETTINGS.position.y
        },
        size = {
            width = WorkCameraMain.DEFAULT_SETTINGS.size.width,
            height = WorkCameraMain.DEFAULT_SETTINGS.size.height
        }
    }
end

function WorkCameraMain:isValidAspectRation(width, height)
    local aspectRatio = width / height

    return MathUtil.round(aspectRatio * 100) / 100 == WorkCameraMain.ASPECT_RATIO
end

function WorkCameraMain:getAspectRatio()
    return WorkCameraMain.ASPECT_RATIO
end

function WorkCameraMain:getHeightFromAspectRatio(width)
    return MathUtil.round(width / WorkCameraMain.ASPECT_RATIO)
end

function WorkCameraMain:getWidthFromAspectRatio(height)
    return MathUtil.round(height * WorkCameraMain.ASPECT_RATIO)
end

function WorkCameraMain:registerItemCamerasSchema()
    self.itemCamerasSchema = XMLSchema.new("workCameraItemCameras")
    local schemaKey = "workCameraItems.workCameras(?)"

    self.itemCamerasSchema:register(XMLValueType.STRING, schemaKey .. "#itemXml", "Item XML File")

    local itemSchemaKey = schemaKey .. ".workCamera(?)"

    self.itemCamerasSchema:register(XMLValueType.STRING, itemSchemaKey .. "#node", "Node where the camera will be added")
    self.itemCamerasSchema:register(XMLValueType.VECTOR_TRANS, itemSchemaKey .. "#translation", "Camera translation",
        "0 0 0")
    self.itemCamerasSchema:register(XMLValueType.VECTOR_ROT, itemSchemaKey .. "#rotation", "Camera rotation", "0 0 0")
    self.itemCamerasSchema:register(XMLValueType.ANGLE, itemSchemaKey .. "#fov", "Camera fov")
    self.itemCamerasSchema:register(XMLValueType.FLOAT, itemSchemaKey .. "#nearClip", "Camera near clip")
    self.itemCamerasSchema:register(XMLValueType.FLOAT, itemSchemaKey .. "#farClip", "Camera far clip")
    self.itemCamerasSchema:register(XMLValueType.STRING, itemSchemaKey .. "#selectedConfigs", "Selected configurations")
    self.itemCamerasSchema:register(XMLValueType.STRING, itemSchemaKey .. "#useConfigName", "Use configuration name")
end

function WorkCameraMain:loadItemsDataFromXml(xmlPath)
    local i = 0
    local xmlFile = XMLFile.load("configXml", xmlPath, self.itemCamerasSchema)
    if xmlFile ~= 0 then
        while true do
            local key = string.format("workCameraItems.workCameras(%d)", i)

            if not xmlFile:hasProperty(key) then
                break
            end

            local itemXml = xmlFile:getValue(key .. "#itemXml")

            itemXml = WorkCameraMain.getValidXmlName(itemXml)

            if itemXml ~= nil then
                if WorkCamera.DEFAULT_ITEMS[itemXml] == nil then
                    WorkCamera.DEFAULT_ITEMS[itemXml] = {}
                end

                local j = 0
                while true do
                    local itemKey = string.format("%s.workCamera(%d)", key, j)

                    if not xmlFile:hasProperty(itemKey) then
                        break
                    end

                    local cameraData = {}

                    cameraData.cameraParentNode = xmlFile:getValue(itemKey .. "#node")

                    local trans = {}
                    trans.x, trans.y, trans.z = xmlFile:getValue(itemKey .. "#translation", true)
                    cameraData.cameraTranslation = trans

                    local rot = {}
                    rot.x, rot.y, rot.z = xmlFile:getValue(itemKey .. "#rotation", true)
                    cameraData.cameraRotation = rot

                    cameraData.cameraFov = xmlFile:getValue(itemKey .. "#fov", WorkCameraMain.DEFAUL_FOV)
                    cameraData.cameraNearClip = xmlFile:getValue(itemKey .. "#nearClip",
                        WorkCameraMain.DEFAULT_NEAR_CLIP)
                    cameraData.cameraFarClip = xmlFile:getValue(itemKey .. "#farClip", WorkCameraMain.DEFAULT_FAR_CLIP)

                    cameraData.selectedConfigs = xmlFile:getValue(itemKey .. "#selectedConfigs", nil)
                    cameraData.useConfigName = xmlFile:getValue(itemKey .. "#useConfigName", nil)

                    table.insert(WorkCamera.DEFAULT_ITEMS[itemXml], cameraData)

                    j = j + 1
                end
            end
            i = i + 1
        end

        xmlFile:delete()
    end
end

function WorkCameraMain:registerGeneralSettingsSchema()
    self.generalSettingSchema = XMLSchema.new("workCameraGeneralSettings")
    local schemaKey = "workCameraGeneralSettings"

    self.generalSettingSchema:register(XMLValueType.INT, schemaKey .. ".relCorner#value", "Relative Corner")
    self.generalSettingSchema:register(XMLValueType.INT, schemaKey .. ".position#x", "Horizontal Offset")
    self.generalSettingSchema:register(XMLValueType.INT, schemaKey .. ".position#y", "Vertical Offset")
    self.generalSettingSchema:register(XMLValueType.INT, schemaKey .. ".size#width", "Camera Width")
    self.generalSettingSchema:register(XMLValueType.INT, schemaKey .. ".size#height", "Camera Height")
end

function WorkCameraMain:loadGeneralSettingsFromXml(xmlPath)
    local xmlFile = XMLFile.load("configXml", xmlPath, self.generalSettingSchema)

    if xmlFile ~= 0 then
        local key = "workCameraGeneralSettings"
        self.generalSettings.position.x = xmlFile:getValue(key .. ".position#x", self.generalSettings.position.x)
        self.generalSettings.position.y = xmlFile:getValue(key .. ".position#y", self.generalSettings.position.y)
        self.generalSettings.size.width = xmlFile:getValue(key .. ".size#width", self.generalSettings.size.width)
        self.generalSettings.size.height = xmlFile:getValue(key .. ".size#height", self.generalSettings.size.height)
        self.generalSettings.relCorner = xmlFile:getValue(key .. ".relCorner#value", self.generalSettings.relCorner)

        if not self:isValidAspectRation(self.generalSettings.size.width, self.generalSettings.size.height) then
            self.generalSettings.size.height = self:getHeightFromAspectRatio(self.generalSettings.size.width)
        end

        xmlFile:delete()
    end
end

function WorkCameraMain:consoleCustomCamerasReload()
    g_currentMission.workCamera:loadWorkCameras()
    print("Custom cameras loaded. Now you need to reset the vehicle with 'gsVehicleReload' command")
end

function WorkCameraMain:loadWorkCameras()
    WorkCamera.DEFAULT_ITEMS = {}

    -- load in-game items cameras
    local IN_GAME_ITEMS_XML = Utils.getFilename("config/InGameItems.xml", self.modDirectory)
    self:loadItemsDataFromXml(IN_GAME_ITEMS_XML)

    -- modSettings folder
    local WORK_CAMERA_FOLDER = getUserProfileAppPath() .. "modSettings/FS22_WorkCamera/"

    -- custom work cameras
    local USER_ITEMS_XML = Utils.getFilename("CustomWorkCameras.xml", WORK_CAMERA_FOLDER)
    local DEFAUL_USER_ITEMS_XML = Utils.getFilename("config/DefaultUserItems.xml", self.modDirectory)

    if fileExists(USER_ITEMS_XML) then
        self:loadItemsDataFromXml(USER_ITEMS_XML)
    else
        copyFile(DEFAUL_USER_ITEMS_XML, USER_ITEMS_XML, false)
    end
end

-- copied from UAL
function WorkCameraMain.getValidXmlName(configName)
    if configName == nil then
        return
    end

    local xmlFilename = configName
    if g_storeManager:getItemByXMLFilename(xmlFilename) then
        return xmlFilename
    end

    xmlFilename = g_modsDirectory .. configName
    if g_storeManager:getItemByXMLFilename(xmlFilename) then
        return xmlFilename
    end

    for i = 1, #g_dlcsDirectories do
        local dlcsDir = g_dlcsDirectories[i].path
        xmlFilename = dlcsDir .. configName
        if g_storeManager:getItemByXMLFilename(xmlFilename) then
            return xmlFilename
        end
    end
end
