WorkCamera = {};

function WorkCamera.prerequisitesPresent(specializations)
    return true
end

function WorkCamera.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", WorkCamera)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", WorkCamera)
    SpecializationUtil.registerEventListener(vehicleType, "onPostAttachImplement", WorkCamera)
    SpecializationUtil.registerEventListener(vehicleType, "onPostDetachImplement", WorkCamera)
    SpecializationUtil.registerEventListener(vehicleType, "onSelect", WorkCamera)
end

function WorkCamera.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "loadCamerasFromVehicleXml", WorkCamera.loadCamerasFromVehicleXml)
    SpecializationUtil.registerFunction(vehicleType, "createWorkCameras", WorkCamera.createWorkCameras)
    SpecializationUtil.registerFunction(vehicleType, "getWorkCameras", WorkCamera.getWorkCameras)
    SpecializationUtil.registerFunction(vehicleType, "setCurrentWorkCamera", WorkCamera.setCurrentWorkCamera)
    SpecializationUtil.registerFunction(vehicleType, "getCurrentWorkCamera", WorkCamera.getCurrentWorkCamera)
    SpecializationUtil.registerFunction(vehicleType, "getIsWorkCameraEnabled", WorkCamera.getIsWorkCameraEnabled)
    SpecializationUtil.registerFunction(vehicleType, "toggleIsWorkCameraEnabled", WorkCamera.toggleIsWorkCameraEnabled)
    SpecializationUtil.registerFunction(vehicleType, "setCurrentWorkCameraFov", WorkCamera.setCurrentWorkCameraFov)
    SpecializationUtil.registerFunction(vehicleType, "resetCurrentWorkCamera", WorkCamera.resetCurrentWorkCamera)
    SpecializationUtil.registerFunction(vehicleType, "setCurrentWorkCameraNearClip",
        WorkCamera.setCurrentWorkCameraNearClip)
    SpecializationUtil.registerFunction(vehicleType, "setCurrentWorkCameraFarClip",
        WorkCamera.setCurrentWorkCameraFarClip)
    SpecializationUtil.registerFunction(vehicleType, "setCurrentWorkCameraFlip", WorkCamera.setCurrentWorkCameraFlip)
end

function WorkCamera.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCanBeSelected", WorkCamera.getCanBeSelected)
end

function WorkCamera.initSpecialization()
    -- vehicle schema
    local schema = Vehicle.xmlSchema

    schema:setXMLSpecializationType("WorkCamera")

    local itemSchemaKey = "vehicle.workCameras.workCamera(?)"

    schema:register(XMLValueType.STRING, itemSchemaKey .. "#node", "Node where the camera will be added")
    schema:register(XMLValueType.VECTOR_TRANS, itemSchemaKey .. "#translation", "Camera translation", "0 0 0")
    schema:register(XMLValueType.VECTOR_ROT, itemSchemaKey .. "#rotation", "Camera rotation", "0 0 0")
    schema:register(XMLValueType.ANGLE, itemSchemaKey .. "#fov", "Camera fov", "80")
    schema:register(XMLValueType.FLOAT, itemSchemaKey .. "#nearClip", "Camera near clip")
    schema:register(XMLValueType.FLOAT, itemSchemaKey .. "#farClip", "Camera far clip")
    schema:register(XMLValueType.STRING, itemSchemaKey .. "#selectedConfigs", "Selected configurations")
    schema:register(XMLValueType.STRING, itemSchemaKey .. "#useConfigName", "Use configuration name")

    -- savegame schema
    local schemaSavegame = Vehicle.xmlSchemaSavegame
    local savegameKey = string.format("vehicles.vehicle(?).%s.workCamera", g_WorkCameraModName)

    schemaSavegame:register(XMLValueType.BOOL, savegameKey .. "#isEnabled", "Camera is enabled")
    schemaSavegame:register(XMLValueType.INT, savegameKey .. "#currentCameraIndex", "Current camera index")

    local cameraSavegameKey = savegameKey .. ".camera(?)"
    schemaSavegame:register(XMLValueType.ANGLE, cameraSavegameKey .. "#fov", "Camera fov")
    schemaSavegame:register(XMLValueType.FLOAT, cameraSavegameKey .. "#nearClip", "Camera near clip")
    schemaSavegame:register(XMLValueType.FLOAT, cameraSavegameKey .. "#farClip", "Camera far clip")
    schemaSavegame:register(XMLValueType.BOOL, cameraSavegameKey .. "#flip", "Camera horizontal flip")
end

function WorkCamera:onLoad(savegame)
    self.spec_workCamera = {
        workCameras = {},
        currentWorkCamera = nil,
        isEnabled = false
    }

    self:createWorkCameras(WorkCamera.DEFAULT_ITEMS[self.configFileName])
    self:createWorkCameras(self:loadCamerasFromVehicleXml())

    local originalWorkCameras = {}
    for k, camera in pairs(self.spec_workCamera.workCameras) do
        table.insert(originalWorkCameras, {
            fov = camera.fov,
            nearClip = camera.nearClip,
            farClip = camera.farClip,
            flip = camera.flip
        })
    end
    self.spec_workCamera.originalWorkCameras = originalWorkCameras
end

function WorkCamera:loadCamerasFromVehicleXml()
    local spec = self.spec_workCamera
    local workCameraMain = g_currentMission.workCamera

    local vehicleXmlCameras = {}

    local i = 0

    while true do
        local key = string.format("vehicle.workCameras.workCamera(%d)", i)

        if not self.xmlFile:hasProperty(key) then
            break
        end

        local cameraData = {}

        cameraData.cameraParentNode = self.xmlFile:getValue(key .. "#node")

        local trans = {}
        trans.x, trans.y, trans.z = self.xmlFile:getValue(key .. "#translation", true)
        cameraData.cameraTranslation = trans

        local rot = {}
        rot.x, rot.y, rot.z = self.xmlFile:getValue(key .. "#rotation", true)
        cameraData.cameraRotation = rot

        cameraData.cameraFov = self.xmlFile:getValue(key .. "#fov", workCameraMain.DEFAUL_FOV)
        cameraData.cameraNearClip = self.xmlFile:getValue(key .. "#nearClip", workCameraMain.DEFAULT_NEAR_CLIP)
        cameraData.cameraFarClip = self.xmlFile:getValue(key .. "#farClip", workCameraMain.DEFAULT_FAR_CLIP)

        cameraData.selectedConfigs = self.xmlFile:getValue(key .. "#selectedConfigs", nil)
        cameraData.useConfigName = self.xmlFile:getValue(key .. "#useConfigName", nil)

        table.insert(vehicleXmlCameras, cameraData)

        i = i + 1
    end

    return vehicleXmlCameras
end

function WorkCamera:createWorkCameras(camerasToCreate)
    local spec = self.spec_workCamera

    if camerasToCreate ~= nil then
        for _, data in pairs(camerasToCreate) do
            local isValidConfig = WorkCamera.getIsValidConfiguration(self, data.selectedConfigs, data.useConfigName,
                self.xmlFile)

            if isValidConfig then
                local parentNode = I3DUtil.indexToObject(self.components, data.cameraParentNode, self.i3dMappings)

                if parentNode ~= nil then
                    local cameraNode = createCamera("workCamera", data.cameraFov, data.cameraNearClip,
                        data.cameraFarClip)
                    link(parentNode, cameraNode)
                    setTranslation(cameraNode, data.cameraTranslation.x, data.cameraTranslation.y,
                        data.cameraTranslation.z)
                    setRotation(cameraNode, data.cameraRotation.x, data.cameraRotation.y, data.cameraRotation.z)
                    table.insert(spec.workCameras, {
                        cameraNode = cameraNode,
                        fov = getFovY(cameraNode),
                        nearClip = getNearClip(cameraNode),
                        farClip = getFarClip(cameraNode),
                        flip = false
                    })
                end
            end
        end
    end
end

function WorkCamera:onPostLoad(savegame)
    if savegame == nil or savegame.resetVehicles then
        return
    end

    local spec = self.spec_workCamera

    local key = string.format("%s.%s.%s", savegame.key, g_WorkCameraModName, "workCamera")
    spec.isEnabled = savegame.xmlFile:getValue(key .. "#isEnabled", false)

    local i = 0

    while i < #spec.workCameras do
        local cameraKey = string.format("%s.camera(%d)", key, i)

        if not savegame.xmlFile:hasProperty(cameraKey) then
            break
        end

        local tableIdx = i + 1

        spec.workCameras[tableIdx].fov = savegame.xmlFile:getValue(cameraKey .. "#fov", spec.workCameras[tableIdx].fov)
        spec.workCameras[tableIdx].nearClip = savegame.xmlFile:getValue(cameraKey .. "#nearClip",
            spec.workCameras[tableIdx].nearClip)
        spec.workCameras[tableIdx].farClip = savegame.xmlFile:getValue(cameraKey .. "#farClip",
            spec.workCameras[tableIdx].farClip)
        spec.workCameras[tableIdx].flip = savegame.xmlFile:getValue(cameraKey .. "#flip",
            spec.workCameras[tableIdx].flip)

        -- apply settings
        setFovY(spec.workCameras[tableIdx].cameraNode, spec.workCameras[tableIdx].fov)
        setNearClip(spec.workCameras[tableIdx].cameraNode, spec.workCameras[tableIdx].nearClip)
        setFarClip(spec.workCameras[tableIdx].cameraNode, spec.workCameras[tableIdx].farClip)

        i = i + 1
    end

    local currentCameraIndex = savegame.xmlFile:getValue(key .. "#currentCameraIndex")
    if currentCameraIndex ~= nil and spec.workCameras[currentCameraIndex] ~= nil then
        self:setCurrentWorkCamera(spec.workCameras[currentCameraIndex])
    end
end

function WorkCamera:saveToXMLFile(xmlFile, key, usedModNames)
    local spec = self.spec_workCamera

    xmlFile:setValue(key .. "#isEnabled", self:getIsWorkCameraEnabled())

    if self:getCurrentWorkCamera() ~= nil then
        for index, camera in pairs(spec.workCameras) do
            if camera == self:getCurrentWorkCamera() then
                xmlFile:setValue(key .. "#currentCameraIndex", index)
                break
            end
        end
    end

    local i = 0

    while i < #spec.workCameras do
        local cameraKey = string.format("%s.camera(%d)", key, i)
        local tableIdx = i + 1

        xmlFile:setValue(cameraKey .. "#fov", spec.workCameras[tableIdx].fov)
        xmlFile:setValue(cameraKey .. "#farClip", spec.workCameras[tableIdx].farClip)
        xmlFile:setValue(cameraKey .. "#nearClip", spec.workCameras[tableIdx].nearClip)
        xmlFile:setValue(cameraKey .. "#flip", spec.workCameras[tableIdx].flip)

        i = i + 1
    end
end

function WorkCamera:onPostAttachImplement()
    local workCameraMain = g_currentMission.workCamera
    workCameraMain:updateAvailableCameras()
end

function WorkCamera:onPostDetachImplement(implementIndex)
    local workCameraMain = g_currentMission.workCamera
    local attachment = self:getObjectFromImplementIndex(implementIndex)
    workCameraMain:updateAvailableCameras(attachment)
end

function WorkCamera:onSelect()
    local workCameraMain = g_currentMission.workCamera
    workCameraMain:updateTurnOnOffInputBinding()
end

function WorkCamera:getWorkCameras()
    return self.spec_workCamera.workCameras
end

function WorkCamera:setCurrentWorkCamera(camera)
    self.spec_workCamera.currentWorkCamera = camera
end

function WorkCamera:getCurrentWorkCamera()
    return self.spec_workCamera.currentWorkCamera
end

function WorkCamera:resetCurrentWorkCamera()
    for k, camera in pairs(self:getWorkCameras()) do
        if camera == self:getCurrentWorkCamera() then
            local original = self.spec_workCamera.originalWorkCameras[k]
            self:setCurrentWorkCameraFov(original.fov)
            self:setCurrentWorkCameraNearClip(original.nearClip)
            self:setCurrentWorkCameraFarClip(original.farClip)
            self:setCurrentWorkCameraFlip(original.flip)
        end
    end
end

function WorkCamera:setCurrentWorkCameraFov(fov)
    local currentWorkCamera = self:getCurrentWorkCamera()
    if currentWorkCamera ~= nil then
        currentWorkCamera.fov = fov
        setFovY(currentWorkCamera.cameraNode, fov)
    end
end

function WorkCamera:setCurrentWorkCameraNearClip(nearClip)
    local currentWorkCamera = self:getCurrentWorkCamera()
    if currentWorkCamera ~= nil then
        currentWorkCamera.nearClip = nearClip
        setNearClip(currentWorkCamera.cameraNode, nearClip)
    end
end

function WorkCamera:setCurrentWorkCameraFarClip(farClip)
    local currentWorkCamera = self:getCurrentWorkCamera()
    if currentWorkCamera ~= nil then
        currentWorkCamera.farClip = farClip
        setFarClip(currentWorkCamera.cameraNode, farClip)
    end
end

function WorkCamera:setCurrentWorkCameraFlip(flip)
    local currentWorkCamera = self:getCurrentWorkCamera()
    if currentWorkCamera ~= nil then
        currentWorkCamera.flip = flip
    end
end

function WorkCamera:getIsWorkCameraEnabled()
    return self.spec_workCamera.isEnabled
end

function WorkCamera:toggleIsWorkCameraEnabled()
    self.spec_workCamera.isEnabled = not self.spec_workCamera.isEnabled
end

function WorkCamera:getIsValidConfiguration(selectedConfigs, useConfigName)
    if selectedConfigs == nil or selectedConfigs == "ALL" then
        return true
    end

    local selectedConfigsList = selectedConfigs:split(",")

    local configName = useConfigName or "design"
    if self.configurations ~= nil and self.configurations[configName] ~= nil then
        local selectedDesign = self.configurations[configName]
        for _, n in ipairs(selectedConfigsList) do
            if tonumber(n) == tonumber(selectedDesign) then
                return true
            end
        end
    end

    return false
end

function WorkCamera:getCanBeSelected(superFunc)
    return #self:getWorkCameras() > 0 or superFunc(self)
end
