CameraElement = {}

local CameraElement_mt = Class(CameraElement)

function CameraElement.new(workCameraMain)
    local self = setmetatable({}, CameraElement_mt)
    self.workCameraMain = workCameraMain
    self.isEnabled = false
    self.overlayScale = 0.91
    self.renderOverlay = nil
    self.frameOverlay = nil

    self:updateHudSize()
    self:updateHudPosition()

    self:createFrameOverlay()

    return self
end

function CameraElement:delete()
    if self.renderOverlay ~= nil then
        delete(self.renderOverlay)

        self.renderOverlay = nil
    end

    if self.frameOverlay ~= nil then
        delete(self.frameOverlay)

        self.frameOverlay = nil
    end
end

function CameraElement:setEnabled(isEnabled)
    self.isEnabled = isEnabled
end

function CameraElement:setCamera(camera)
    local resolutionX = math.ceil(g_screenWidth * self.width) * 2
    local resolutionY = math.ceil(g_screenHeight * self.height) * 2
    local aspectRatio = resolutionX / resolutionY

    if camera == nil then
        self.renderOverlay = nil
    else
        local shapesMask = 98304
        local lightMask = 98304

        self.renderOverlay = createRenderOverlay(camera.cameraNode, aspectRatio, resolutionX, resolutionY, false,
            shapesMask, lightMask)

        if camera.flip then
            setOverlayUVs(self.renderOverlay, 1, 0, 1, 1, 0, 0, 0, 1)
        end
    end
end

function CameraElement:update(dt)
    if not self.isEnabled or self.renderOverlay == nil then
        return
    end

    updateRenderOverlay(self.renderOverlay)
end

function CameraElement:draw()
    if not self.isEnabled or self.renderOverlay == nil then
        return
    end

    local posX = self.midPosX - self.width / 2
    local posY = self.midPosY - self.height / 2

    local cameraPosX = self.midPosX - self.width / 2 * self.overlayScale
    local cameraPosY = self.midPosY - self.height / 2 * self.overlayScale
    local cameraWidth = self.width * self.overlayScale
    local cameraHeight = self.height * self.overlayScale

    renderOverlay(self.renderOverlay, cameraPosX, cameraPosY, cameraWidth, cameraHeight)

    if self.frameOverlay ~= nil then
        renderOverlay(self.frameOverlay, posX, posY, self.width, self.height)
    end
end

function CameraElement:updateHudPosition()
    local settings = self.workCameraMain.generalSettings
    local posX, posY = getNormalizedScreenValues(settings.position.x, settings.position.y)

    if settings.relCorner == WorkCameraMain.TOP_LEFT then
        self.midPosX = g_safeFrameOffsetX + posX + self.width / 2
        self.midPosY = 1 - g_safeFrameOffsetY - posY - self.height / 2
    elseif settings.relCorner == WorkCameraMain.TOP_RIGHT then
        self.midPosX = 1 - g_safeFrameOffsetX - posX - self.width / 2
        self.midPosY = 1 - g_safeFrameOffsetY - posY - self.height / 2
    elseif settings.relCorner == WorkCameraMain.BOTTOM_RIGHT then
        self.midPosX = 1 - g_safeFrameOffsetX - posX - self.width / 2
        self.midPosY = g_safeFrameOffsetY + posY + self.height / 2
    elseif settings.relCorner == WorkCameraMain.BOTTOM_LEFT then
        self.midPosX = g_safeFrameOffsetX + posX + self.width / 2
        self.midPosY = g_safeFrameOffsetY + posY + self.height / 2
    end
end

function CameraElement:updateHudSize()
    local pixelWidth = self.workCameraMain.generalSettings.size.width
    local pixelHeight = self.workCameraMain.generalSettings.size.height

    self.width, self.height = getNormalizedScreenValues(pixelWidth, pixelHeight)
end

function CameraElement:createFrameOverlay()
    self.frameOverlay = createImageOverlay(Utils.getFilename("assets/frame.png", self.workCameraMain.modDirectory))

    setOverlayUVs(self.frameOverlay, 0, 0.1875, 0, 1 - 0.1875, 1, 0.1875, 1, 1 - 0.1875)
end
