--
-- Real Mower for FS22
-- @author:    	kenny456 (kenny456@seznam.cz)
-- @history:	v1.0 - 2022-10-07 - conversion to FS22
--
RealMower = {};
RealMower.confDir = getUserProfileAppPath().. "modsSettings/RealMower/";
RealMower.modDirectory = g_currentModDirectory
local modName = g_currentModName

function RealMower.prerequisitesPresent(specializations)
	return true
end
function RealMower.registerOverwrittenFunctions(vehicleType)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "processMowerArea", 		RealMower.processMowerArea)
end
function RealMower.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "toggleActive", 						RealMower.toggleActive)
end
function RealMower:onRegisterActionEvents(isSelected, isOnActiveVehicle)
	local spec = self.spec_realMower
	if not spec.modInitialized then
		return
	end
	
	if g_dedicatedServerInfo ~= nil then
		return
	end
	if spec.event_IDs == nil then
		spec.event_IDs = {}
	end
	if self:getIsActiveForInput() or (self.getIscontrolled ~= nil and self:getIsControlled()) then
		local actions = { InputAction.REALMOWER_ACTIVATE }

		for _,actionName in pairs(actions) do
			local always = (actionName == InputAction.REALSHOVEL_MENU_PLUS or actionName == InputAction.REALSHOVEL_MENU_MINUS) and true or false
			local _, eventID = g_inputBinding:registerActionEvent(actionName, self, RealMower.actionCallback, true, true, always, true)
			spec.event_IDs[actionName] = eventID
			if g_inputBinding ~= nil and g_inputBinding.events ~= nil and g_inputBinding.events[eventID] ~= nil then
				if actionName == 'something with lower priority' then
					g_inputBinding:setActionEventTextPriority(eventID, GS_PRIO_NORMAL)
				else
					g_inputBinding:setActionEventTextPriority(eventID, GS_PRIO_VERY_HIGH)
				end
				if actionName == InputAction.REALMOWER_TOGGLE_HELP then
					g_inputBinding:setActionEventTextVisibility(eventID, true)
				elseif actionName == InputAction.REALMOWER_TOGGLE then
					g_inputBinding:setActionEventTextVisibility(eventID, false)
				else
					g_inputBinding:setActionEventTextVisibility(eventID, RealMower.showHelp)
				end
			end
			local colliding = false
			_, colliding, _ = g_inputBinding:checkEventCollision(actionName)
			if colliding then
				if g_inputBinding.nameActions[actionName].bindings[1] ~= nil then
					if g_inputBinding.nameActions[actionName].bindings[1].inputString ~= nil then
						print(string.format('Warning: RealMower got a colliding input action: %s', actionName)..' ('..g_inputBinding.nameActions[actionName].bindings[1].inputString..'). You can remap it in controls settings')
					end
				else
					print(string.format('Warning: RealMower got a colliding input action: %s', actionName))
				end
			end
		end
	end
end
function RealMower.registerEventListeners(vehicleType)
	for _,n in pairs( { "onLoad", "onPostLoad", "saveToXMLFile", "onUpdate", "onRegisterActionEvents", "toggleActive", "onReadStream", "onWriteStream"} ) do
		SpecializationUtil.registerEventListener(vehicleType, n, RealMower)
	end
end
function RealMower:onLoad(savegame)
	self.spec_realMower = {}
	local spec = self.spec_realMower
	
	spec.modInitialized = false
	if self.spec_workArea.workAreas ~= nil then
		for _, workArea in pairs(self.spec_workArea.workAreas) do
			if workArea.type == WorkAreaType.MOWER then
				spec.modInitialized = true
				workArea.requiresOwnedFarmland = false
			end
		end
	end
	if not spec.modInitialized then
		print("Error: RealMower initialization failed for "..tostring(self:getFullName()).." !")
		return
	end
	spec.event_IDs = {}
	spec.modActive = true
	RealMower.showHelp = true
end
function RealMower:onPostLoad(savegame)
	local spec = self.spec_realMower
	if not spec.modInitialized then
		return
	end
	if savegame ~= nil then
		local xmlFile = savegame.xmlFile
		local key = savegame.key.."."..modName..".RealMower"
		spec.modActive = Utils.getNoNil(getXMLBool(xmlFile.handle, key.."#modActive"), spec.modActive)
		self:toggleActive(spec.modActive, true)
	end
end
function RealMower:saveToXMLFile(xmlFile, key)
	local spec = self.spec_realMower
	
	setXMLBool(xmlFile.handle, key.."#modActive", spec.modActive)
end
function RealMower:onDelete()
	local spec = self.spec_realMower
	if not spec.modInitialized then
		return
	end
end
function RealMower:processMowerArea(superFunc, workArea, dt)
	local spec = self.spec_mower
    local xs,_,zs = getWorldTranslation(workArea.start)
    local xw,_,zw = getWorldTranslation(workArea.width)
    local xh,_,zh = getWorldTranslation(workArea.height)
    if self:getLastSpeed() > 1 then
        spec.isWorking = true
        spec.stoneLastState = FSDensityMapUtil.getStoneArea(xs, zs, xw, zw, xh, zh)
    else
        spec.stoneLastState = 0
    end
	if self.spec_realMower.modActive then
		FSDensityMapUtil.clearDecoArea(xs, zs, xw, zw, xh, zh)
	end
    local limitToField = self:getIsAIActive()
    for inputFruitType, converterData in pairs(spec.fruitTypeConverters) do
        local changedArea, totalArea, sprayFactor, plowFactor, limeFactor, weedFactor, stubbleFactor, rollerFactor, _, growthState, _ = FSDensityMapUtil.updateMowerArea(inputFruitType, xs, zs, xw, zw, xh, zh, limitToField)
        if changedArea > 0 then
            local multiplier = g_currentMission:getHarvestScaleMultiplier(inputFruitType, sprayFactor, plowFactor, limeFactor, weedFactor, stubbleFactor, rollerFactor)
            changedArea = changedArea * multiplier
            local pixelToSqm = g_currentMission:getFruitPixelsToSqm()
            local sqm = changedArea * pixelToSqm
            local litersToDrop = sqm * g_fruitTypeManager:getFillTypeLiterPerSqm(converterData.fillTypeIndex, 1)
            workArea.lastPickupLiters = litersToDrop
            workArea.pickedUpLiters = litersToDrop
            local dropArea = self:getDropArea(workArea)
            if dropArea ~= nil then
                dropArea.litersToDrop = dropArea.litersToDrop + litersToDrop
                dropArea.fillType = converterData.fillTypeIndex
                dropArea.workAreaIndex = workArea.index
                -- if there is already dryGrass on the field and we cannot tip grass on the same spot
                -- we pickup the dryGrass and drop it as grass again
                if dropArea.fillType == FillType.GRASS_WINDROW then
                    local lsx, lsy, lsz, lex, ley, lez, radius = DensityMapHeightUtil.getLineByArea(workArea.start, workArea.width, workArea.height, true)
                    local pickup
                    pickup, workArea.lineOffset = DensityMapHeightUtil.tipToGroundAroundLine(self, -math.huge, FillType.DRYGRASS_WINDROW, lsx, lsy, lsz, lex, ley, lez, radius, nil, workArea.lineOffset or 0, false, nil, false)
                    dropArea.litersToDrop = dropArea.litersToDrop - pickup
                end
                -- limit liters to drop so we don't buffer unlimited amount of grass
                dropArea.litersToDrop = math.min(dropArea.litersToDrop, 1000)
            elseif spec.fillUnitIndex ~= nil then
                if self.isServer then
                    self:addFillUnitFillLevel(self:getOwnerFarmId(), spec.fillUnitIndex, litersToDrop, converterData.fillTypeIndex, ToolType.UNDEFINED)
                end
            end
            spec.workAreaParameters.lastInputFruitType = inputFruitType
            spec.workAreaParameters.lastInputGrowthState = growthState
            spec.workAreaParameters.lastCutTime = g_time
            spec.workAreaParameters.lastChangedArea = spec.workAreaParameters.lastChangedArea + changedArea
            spec.workAreaParameters.lastStatsArea = spec.workAreaParameters.lastStatsArea + changedArea
            spec.workAreaParameters.lastTotalArea   = spec.workAreaParameters.lastTotalArea + totalArea
            spec.workAreaParameters.lastUsedAreas = spec.workAreaParameters.lastUsedAreas + 1
        end
    end
    spec.workAreaParameters.lastUsedAreasSum = spec.workAreaParameters.lastUsedAreasSum + 1
    return spec.workAreaParameters.lastChangedArea, spec.workAreaParameters.lastTotalArea
end
function RealMower:onUpdate(dt, vehicle)
	local spec = self.spec_realMower
	
	if not spec.modInitialized then
		return
	end
	if self.isClient then
		if self:getIsActiveForInput() or (self.getIscontrolled ~= nil and self:getIsControlled()) then
			if spec.event_IDs ~= nil and g_dedicatedServerInfo == nil then
				for actionName,eventID in pairs(spec.event_IDs) do
					if actionName == InputAction.REALMOWER_ACTIVATE then
						g_inputBinding:setActionEventActive(eventID, true)
						g_inputBinding:setActionEventText(eventID, spec.modActive and g_i18n:getText('REALMOWER_DEACTIVATE') or g_i18n:getText('REALMOWER_ACTIVATE'))
					end
				end
			end
		end
	end
end
function RealMower:actionCallback(actionName, keyStatus, arg4, arg5, arg6)
	local spec = self.spec_realMower
	
	if not spec.modInitialized then
		return
	end
	if keyStatus > 0 then
		if actionName == 'REALMOWER_ACTIVATE' then
			spec.modActive = not spec.modActive
			self:toggleActive(spec.modActive, false)
		end
	end
end
function RealMower:toggleActive(modActive, noEventSend)
	local spec = self.spec_realMower
	
	if not spec.modInitialized then
		return
	end
	spec.modActive = modActive
	if self.spec_workArea.workAreas ~= nil then
		for _, workArea in pairs(self.spec_workArea.workAreas) do
			if workArea.type == WorkAreaType.MOWER then
				workArea.requiresOwnedFarmland = not spec.modActive
			end
		end
	end
	RealMowerToggleActiveEvent.sendEvent(self, modActive, noEventSend)
end
function RealMower:onReadStream(streamId, connection)
	local spec = self.spec_realMower
	if not spec.modInitialized then
		return
	end
	
    local modActive = streamReadBool(streamId)
	if modActive ~= nil then
		self:toggleActive(modActive, true)
	end
end
function RealMower:onWriteStream(streamId, connection)
	local spec = self.spec_realMower
	if not spec.modInitialized then
		return
	end
	
    streamWriteBool(streamId, spec.modActive)
end

RealMowerToggleActiveEvent = {};
RealMowerToggleActiveEvent_mt = Class(RealMowerToggleActiveEvent, Event)

InitEventClass(RealMowerToggleActiveEvent, "RealMowerToggleActiveEvent")

function RealMowerToggleActiveEvent.emptyNew()
    local self = Event.new(RealMowerToggleActiveEvent_mt)
    self.className="RealMowerToggleActiveEvent"
    return self
end

function RealMowerToggleActiveEvent.new(object, modActive)
	local self = RealMowerToggleActiveEvent.emptyNew()
	self.object = object
	self.modActive = modActive
	return self
end

function RealMowerToggleActiveEvent:readStream(streamId, connection)
	self.object = NetworkUtil.readNodeObject(streamId)
    self.modActive = streamReadBool(streamId)
    self:run(connection)
end

function RealMowerToggleActiveEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.object)
	streamWriteBool(streamId, self.modActive)
end

function RealMowerToggleActiveEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(self, false, connection, self.object)
	end
	if self.object ~= nil then
		self.object:toggleActive(self.modActive, true)
	end
end

function RealMowerToggleActiveEvent.sendEvent(vehicle, modActive, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(RealMowerToggleActiveEvent.new(vehicle, modActive), nil, nil, vehicle)
		else
			g_client:getServerConnection():sendEvent(RealMowerToggleActiveEvent.new(vehicle, modActive))
		end
	end
end