--[[
Copyright (C) GtX (Andy), 2022

Author: GtX | Andy
Date: 14.07.2022
Revision: FS22-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

local modName = g_currentModName or ""
local modDirectory = g_currentModDirectory or ""
local modSettingsDirectory = g_currentModSettingsDirectory or ""

local versionString = "0.0.0.0"
local buildId = 2

local oldSpecFunctions = {}

local enterablePassengerExtension
local validationFail

local defaultIkChainValues = {
    rightFoot = {
        position = {-0.048, -0.465, 0.442},
        rotation = {0, -8.5, 0}
    },
    leftFoot = {
        position = {0.13, -0.465, 0.384},
        rotation = {0, 5.5, 0}
    },
    rightArm = {
        position = {-0.048, 0.118, 0.34},
        rotation = {72.06, 81.629, 128.648}
    },
    leftArm = {
        position = {0.073, 0.107, 0.395},
        rotation = {155.689, -65.777, 166.266}
    }
}

local function isActive()
    return g_modIsLoaded[modName] and enterablePassengerExtension ~= nil
end

local function isDlcActive()
    if g_modIsLoaded["pdlc_kubotaPack"] then
        local directory = g_modNameToDirectory["pdlc_kubotaPack"]

        if directory ~= nil then
            directory = directory:lower()

            for i = 1, #g_dlcsDirectories do
                local path = g_dlcsDirectories[i].path or ""

                if directory:sub(1, #path) == path:lower() then
                    return pdlc_kubotaPack ~= nil and pdlc_kubotaPack.EnterablePassenger ~= nil
                end
            end
        end
    end

    return false
end

local function storeSpecFunction(object, funcName)
    local originalFunc = object[funcName]

    if originalFunc ~= nil then
        table.insert(oldSpecFunctions, {
            object = object,
            funcName = funcName,
            originalFunc = originalFunc
        })
    end

    return originalFunc
end

local function createNode(name, linkNode, x, y, z, rx, ry, rz)
    local node = createTransformGroup(name)

    link(linkNode, node)

    setTranslation(node, x or 0, y or 0, z or 0)
    setRotation(node, rx or 0, ry or 0, rz or 0)

    return node
end

local function getMappingNodeIfExists(vehicle, xmlFile, key, default, showWarnings)
    local nodeStr = xmlFile:getString(key)

    if nodeStr ~= nil then
        if vehicle.i3dMappings ~= nil and not nodeStr:find(">", 1, true) then
            local mapping = vehicle.i3dMappings[nodeStr]

            if mapping ~= nil and type(mapping) == "table" and mapping.nodeId ~= nil then
                return mapping.nodeId, nodeStr
            elseif showWarnings then
                Logging.warning("Mapping node with name '%s' no longer exists or is invalid, using vehicle root node. This may lead to undesired passenger positioning.", nodeStr)
            end
        elseif showWarnings then
            Logging.warning("Vehicle does not contain I3D Mappings or '%s' at '%s' is not a mapping value, using vehicle root node. This may lead to undesired passenger positioning.", nodeStr, key)
        end
    end

    return default
end

local function loadVehicleCharacter(vehicle, xmlFile, key, linkNode, name)
    local passengerSeat = createNode(name or "passengerSeat", linkNode or vehicle.rootNode)

    local x, y, z = xmlFile:getValue(key .. ".characterNode#position", "0 0 0", false)
    local rx, ry, rz = xmlFile:getValue(key .. ".characterNode#rotation", "0 0 0", false)

    local playerRoot = createNode("playerRoot", passengerSeat, x, y, z, rx, ry, rz)
    local characterNode = createNode("playerSkin", playerRoot)

    local ikChainTargets = {}

    for ikName, defaultValues in pairs (defaultIkChainValues) do
        local x, y, z = xmlFile:getValue(string.format("%s.characterNode.%s#position", key, ikName), defaultValues.position, false)
        local rx, ry, rz = xmlFile:getValue(string.format("%s.characterNode.%s#rotation", key, ikName), defaultValues.rotation, false)

        local poseId = xmlFile:getValue(string.format("%s.characterNode.%s#poseId", key, ikName)) -- narrowFingers | wideFingers | flatFingers
        local targetNode = createNode(ikName, playerRoot, x, y, z, rx, ry, rz)

        ikChainTargets[ikName] = {
            ikName = ikName,
            targetNode = targetNode,
            setDirty = true,
            poseId = poseId,
            rotationNodes = {}
        }
    end

    local vehicleCharacter = VehicleCharacter.new(vehicle)

    vehicleCharacter.characterNode = characterNode
    vehicleCharacter.parentComponent = vehicle:getParentComponent(characterNode)
    vehicleCharacter.characterCameraMinDistance = xmlFile:getValue(key .. ".characterNode#cameraMinDistance", 0.3)
    vehicleCharacter.characterDistanceRefNode = characterNode

    setVisibility(vehicleCharacter.characterNode, false)

    vehicleCharacter.useAnimation = false
    vehicleCharacter.useIdleAnimation = true
    vehicleCharacter.ikChainTargets = ikChainTargets

    vehicleCharacter.characterSpineRotation = xmlFile:getValue(key .. ".characterNode#spineRotation", "-90 0 90", true)

    vehicleCharacter.characterSpineSpeedDepended = false
    vehicleCharacter.characterSpineNodeMinRot = 10
    vehicleCharacter.characterSpineNodeMaxRot = -10
    vehicleCharacter.characterSpineNodeMinAcc =  -1 / (1000 * 1000)
    vehicleCharacter.characterSpineNodeMaxAcc = 1 / (1000 * 1000)
    vehicleCharacter.characterSpineNodeAccDeadZone = 0.2 / (1000 * 1000)
    vehicleCharacter.characterSpineLastRotation = 0
    vehicleCharacter:setCharacterVisibility(false)
    vehicleCharacter.maxUpdateDistance = xmlFile:getValue(key .. ".characterNode#maxUpdateDistance", VehicleCharacter.DEFAULT_MAX_UPDATE_DISTANCE)

    setClipDistance(vehicleCharacter.characterNode, xmlFile:getValue(key .. ".characterNode#clipDistance", VehicleCharacter.DEFAULT_CLIP_DISTANCE))

    return vehicleCharacter, playerRoot, passengerSeat
end

local function loadVehicleCamera(vehicle, xmlFile, key, linkNode, name)
    local cameraNode = createCamera("indoorCamera", math.rad(60), 0.1, 5000)

    local x, y, z = xmlFile:getValue(key .. "#cameraPosition", "0 0.75 0", false)
    local rx, ry, rz = xmlFile:getValue(key .. "#cameraRotation", "-10 180 0", false)

    link(linkNode or vehicle.rootNode, cameraNode)

    setTranslation(cameraNode, x, y, z)
    setRotation(cameraNode, rx, ry, rz)

    local camera = VehicleCamera.new(vehicle)

    camera.cameraNode = cameraNode
    camera.fovY = calculateFovY(cameraNode)

    setFovY(cameraNode, camera.fovY)

    camera.isRotatable = xmlFile:getValue(key .. "#rotatable", true)
    camera.limit = xmlFile:getValue(key .. "#limit", true)

    if camera.limit then
        camera.rotMinX = xmlFile:getValue(key .. "#rotMinX")
        camera.rotMaxX = xmlFile:getValue(key .. "#rotMaxX")
        camera.transMin = xmlFile:getValue(key .. "#transMin")
        camera.transMax = xmlFile:getValue(key .. "#transMax")

        if camera.transMax ~= nil then
            camera.transMax = math.max(camera.transMin, camera.transMax * Platform.gameplay.maxCameraZoomFactor)
        end

        if camera.rotMinX == nil or camera.rotMaxX == nil or camera.transMin == nil or camera.transMax == nil then
            Logging.xmlWarning(xmlFile, "Missing 'rotMinX', 'rotMaxX', 'transMin' or 'transMax' for camera '%s'", key)

            return false
        end
    end

    camera.isInside = xmlFile:getValue(key .. "#isInside", false)
    camera.allowHeadTracking = xmlFile:getValue(key .. "#allowHeadTracking", camera.isInside)

    local shadowFocusBoxCameraIndex = xmlFile:getValue(key .. "#shadowFocusBoxCameraIndex")

    if shadowFocusBoxCameraIndex ~= nil then
        local enterableSpec = vehicle.spec_enterable
        local shadowFocusBoxCamera = enterableSpec.cameras ~= nil and enterableSpec.cameras[shadowFocusBoxCameraIndex]

        if shadowFocusBoxCamera ~= nil and shadowFocusBoxCamera.shadowFocusBoxNode ~= nil then
            camera.shadowFocusBoxNode = shadowFocusBoxCamera.shadowFocusBoxNode
        end
    end

    if camera.shadowFocusBoxNode == nil then
        camera.shadowFocusBoxNode = getMappingNodeIfExists(vehicle, xmlFile, key .. "#shadowFocusBox", nil, false)

        if camera.shadowFocusBoxNode ~= nil and not getHasClassId(camera.shadowFocusBoxNode, ClassIds.SHAPE) then
            Logging.xmlWarning(xmlFile, "Invalid camera shadow focus box '%s'. Must be a shape and cpu mesh", getName(camera.shadowFocusBoxNode))

            camera.shadowFocusBoxNode = nil
        end
    end

    camera.useOutdoorSounds = xmlFile:getValue(key .. "#useOutdoorSounds", not camera.isInside)
    camera.hasExtraRotationNode = false
    camera.allowTranslation = false
    camera.changeObjects = {}

    camera.useMirror = xmlFile:getValue(key .. "#useMirror", false)
    camera.useWorldXZRotation = xmlFile:getValue(key .. "#useWorldXZRotation")
    camera.resetCameraOnVehicleSwitch = xmlFile:getValue(key .. "#resetCameraOnVehicleSwitch")
    camera.suspensionNodeIndex = xmlFile:getValue(key .. "#suspensionNodeIndex")

    camera.positionSmoothingParameter = 0
    camera.lookAtSmoothingParameter = 0

    local useDefaultPositionSmoothing = xmlFile:getValue(key .. "#useDefaultPositionSmoothing", false)

    if useDefaultPositionSmoothing then
        if camera.isInside then
            camera.positionSmoothingParameter = 0.128
            camera.lookAtSmoothingParameter = 0.176
        else
            camera.positionSmoothingParameter = 0.016
            camera.lookAtSmoothingParameter = 0.022
        end
    end

    camera.positionSmoothingParameter = xmlFile:getValue(key .. "#positionSmoothingParameter", camera.positionSmoothingParameter)
    camera.lookAtSmoothingParameter = xmlFile:getValue(key .. "#lookAtSmoothingParameter", camera.lookAtSmoothingParameter)

    local useHeadTracking = g_gameSettings:getValue("isHeadTrackingEnabled") and isHeadTrackingAvailable() and camera.allowHeadTracking

    if useHeadTracking then
        camera.positionSmoothingParameter = 0
        camera.lookAtSmoothingParameter = 0
    end

    camera.cameraPositionNode = cameraNode
    camera.rotateNode = camera.cameraPositionNode

    if camera.positionSmoothingParameter > 0 then
        camera.cameraPositionNode = createTransformGroup("cameraPositionNode")
        local camIndex = getChildIndex(cameraNode)

        link(getParent(cameraNode), camera.cameraPositionNode, camIndex)

        x, y, z = getTranslation(cameraNode)
        rx, ry, rz = getRotation(cameraNode)

        setTranslation(camera.cameraPositionNode, x, y, z)
        setRotation(camera.cameraPositionNode, rx, ry, rz)

        unlink(cameraNode)
    end

    camera.rotYSteeringRotSpeed = 0

    if useHeadTracking then
        local dx, _, dz = localDirectionToLocal(camera.cameraPositionNode, getParent(camera.cameraPositionNode), 0, 0, 1)
        local tx, ty, tz = localToLocal(camera.cameraPositionNode, getParent(camera.cameraPositionNode), 0, 0, 0)
        camera.headTrackingNode = createTransformGroup("headTrackingNode")

        link(getParent(camera.cameraPositionNode), camera.headTrackingNode)
        setTranslation(camera.headTrackingNode, tx, ty, tz)

        if math.abs(dx) + math.abs(dz) > 0.0001 then
            setDirection(camera.headTrackingNode, dx, 0, dz, 0, 1, 0)
        else
            setRotation(camera.headTrackingNode, 0, 0, 0)
        end
    end

    camera.origRotX, camera.origRotY, camera.origRotZ = getRotation(camera.rotateNode)
    camera.rotX = camera.origRotX
    camera.rotY = camera.origRotY
    camera.rotZ = camera.origRotZ
    camera.origTransX, camera.origTransY, camera.origTransZ = getTranslation(camera.cameraPositionNode)
    camera.transX = camera.origTransX
    camera.transY = camera.origTransY
    camera.transZ = camera.origTransZ

    local transLength = MathUtil.vector3Length(camera.origTransX, camera.origTransY, camera.origTransZ) + 0.00001 -- prevent division by zero
    camera.zoom = transLength
    camera.zoomTarget = transLength
    camera.zoomDefault = transLength
    camera.zoomLimitedTarget = -1

    local trans1OverLength = 1 / transLength
    camera.transDirX = trans1OverLength * camera.origTransX
    camera.transDirY = trans1OverLength * camera.origTransY
    camera.transDirZ = trans1OverLength * camera.origTransZ

    table.insert(camera.raycastNodes, camera.rotateNode)

    camera.headTrackingPositionOffset = {
        0,
        0,
        0
    }

    camera.headTrackingRotationOffset = {
        0,
        0,
        0
    }

    return camera
end

local function overwriteSpecializationFunctions()
    local oldOnLoad = storeSpecFunction(pdlc_kubotaPack.EnterablePassenger, "onLoad")

    if oldOnLoad ~= nil then
        function pdlc_kubotaPack.EnterablePassenger:onLoad(savegame)
            oldOnLoad(self, savegame)

            local spec = self.spec_enterablePassenger

            if enterablePassengerExtension ~= nil then
                local numPassengerSeats = #spec.passengerSeats

                if numPassengerSeats == 0 then
                    local xmlFile, key = enterablePassengerExtension:getGlobalXMLData(self.configFileName)

                    if xmlFile ~= nil then
                        local enterableSpec = self.spec_enterable

                        xmlFile:iterate(key .. ".passengerSeat", function (_, passengerSeatKey)
                            local name = string.format("passengerSeat%02.f", #spec.passengerSeats + 1)

                            local linkNode, linkNodeStr = getMappingNodeIfExists(self, xmlFile, passengerSeatKey .. "#linkNode", self.rootNode, true)
                            local vehicleCharacter, playerRoot, passengerNode = loadVehicleCharacter(self, xmlFile, passengerSeatKey, linkNode, name)

                            if passengerNode ~= nil then
                                local seat = {
                                    node = playerRoot,
                                    cameras = {},
                                    camIndex = 1,
                                    isUsed = false,
                                    vehicleCharacter = vehicleCharacter,
                                    nicknameOffset = xmlFile:getValue(passengerSeatKey .. "#nicknameOffset", 1.5)
                                }

                                if enterableSpec.cameras ~= nil then
                                    local cameraIndex = xmlFile:getValue(passengerSeatKey .. "#outdoorCameraIndex", 1)

                                    if cameraIndex ~= nil and enterableSpec.cameras[cameraIndex] ~= nil then
                                        table.insert(seat.cameras, enterableSpec.cameras[cameraIndex])
                                    end
                                end

                                local camera = loadVehicleCamera(self, xmlFile, passengerSeatKey .. ".camera", passengerNode)

                                if camera ~= nil then
                                    table.insert(seat.cameras, camera)
                                end

                                local exitNodePosition = xmlFile:getValue(passengerSeatKey .. "#exitNodePosition", nil, true)

                                if exitNodePosition ~= nil then
                                    if linkNode == self.rootNode then
                                        seat.exitPoint = createNode("exitPoint", passengerNode, exitNodePosition[1], exitNodePosition[2], exitNodePosition[3])
                                    else
                                        seat.exitPoint = createNode(name .. "ExitPoint", self.rootNode, exitNodePosition[1], exitNodePosition[2], exitNodePosition[3])
                                    end
                                end

                                seat.disabledReverseDriving = xmlFile:getValue(passengerSeatKey .. "#disabledReverseDriving", false)

                                table.insert(spec.passengerSeats, seat)
                            end
                        end)

                        numPassengerSeats = #spec.passengerSeats

                        if numPassengerSeats > 0 then
                            local vehicleCharacter = self:getVehicleCharacter()

                            if vehicleCharacter ~= nil and vehicleCharacter.characterCameraMinDistance > 0.4 then
                                vehicleCharacter.characterCameraMinDistance = 0.4
                            end

                            -- Development only
                            if self.isServer and enterablePassengerExtension.creator ~= nil then
                                if enterablePassengerExtension.passengerVehicles ~= nil then
                                    for i, seat in ipairs (spec.passengerSeats) do
                                        seat.creatorInfo = {
                                            i3dCharacterNode = getParent(seat.vehicleCharacter.characterNode),
                                            linkNodeStr = xmlFile:getString(string.format("%s.passengerSeat(%d)#linkNode", key, i - 1)),
                                            exitNodePositionStr = xmlFile:getString(string.format("%s.passengerSeat(%d)#exitNodePosition", key, i - 1))
                                        }
                                    end

                                    table.insert(enterablePassengerExtension.passengerVehicles, self)
                                end
                            end
                        end

                        xmlFile:delete()
                    end
                end

                if numPassengerSeats > 0 then
                    -- Temporary fix until @Giants fixes the issue where changing deactivates all attachments when AI is active
                    self.deactivate = Utils.overwrittenFunction(self.deactivate, function(vehicle, superFunc)
                        if vehicle:getIsAIActive() then
                            return
                        end

                        superFunc(vehicle)
                    end)

                    -- Allow use in Single Player because why not? :-)
                    if not spec.available then
                        spec.available = true

                        -- Appear to be the only message subscriptions
                        g_messageCenter:subscribe(MessageType.USER_REMOVED, self.onPassengerUserRemoved, self)
                        g_messageCenter:subscribe(MessageType.PASSENGER_CHARACTER_CHANGED, self.onPassengerPlayerStyleChanged, self)
                    end

                    -- Show passengers and camera in store to allow for quick adjustments in GE or XML
                    if self.isServer and enterablePassengerExtension.showPassengersInStore and self.propertyState == Vehicle.PROPERTY_STATE_SHOP_CONFIG then
                        local style = PlayerStyle.new()

                        style.xmlFilename = "dataS/character/humans/player/player01.xml"
                        style.bottomConfig.selection = 1
                        style.topConfig.selection = 1
                        style.footwearConfig.selection = 1
                        style.hairStyleConfig.selection = 1
                        style.hairStyleConfig.color = 1
                        style.faceConfig.selection = 1

                        for i = 1, numPassengerSeats do
                            self:setPassengerSeatCharacter(i, style)
                        end

                        spec.renderPassengerCameraLocations = true
                    end
                end
            end
        end

        local oldOnPostUpdate = storeSpecFunction(pdlc_kubotaPack.EnterablePassenger, "onPostUpdate")

        if oldOnPostUpdate ~= nil then
            function pdlc_kubotaPack.EnterablePassenger:onPostUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
                oldOnPostUpdate(self, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)

                local spec = self.spec_enterablePassenger

                if self.isClient and spec.available and spec.passengerEntered then
                    if not self:getIsControlled() and self:getIsAIActive() then
                        local enterableSpec = self.spec_enterable
                        local drivableSpec = self.spec_drivable

                        if drivableSpec and drivableSpec.steeringWheel then
                            local steeringWheel = drivableSpec.steeringWheel

                            if drivableSpec.idleTurningAllowed and drivableSpec.idleTurningActive and not drivableSpec.idleTurningUpdateSteeringWheel then
                                steeringWheel = nil
                            end

                            if steeringWheel ~= nil then
                                local rotation = (self.rotatedTime or 0) * steeringWheel.outdoorRotation

                                if steeringWheel.lastRotation ~= rotation then
                                    steeringWheel.lastRotation = rotation

                                    setRotation(steeringWheel.node, 0, rotation, 0)

                                    if enterableSpec.vehicleCharacter ~= nil and enterableSpec.vehicleCharacter:getAllowCharacterUpdate() then
                                        enterableSpec.vehicleCharacter:setDirty(true)
                                    end
                                end
                            end
                        end

                        for _, modifier in ipairs(enterableSpec.characterTargetNodeModifiers) do
                            self:updateCharacterTargetNodeModifier(dt, modifier)
                        end

                        if enterableSpec.vehicleCharacter ~= nil then
                            enterableSpec.vehicleCharacter:update(dt)
                        end
                    end

                    -- Update all passenger character idle animations
                    local numPassengerSeats = #spec.passengerSeats

                    if numPassengerSeats > 1 then
                        for i = 1, numPassengerSeats do
                            if spec.currentSeatIndex ~= i then
                                local seat = spec.passengerSeats[i]

                                if seat.isUsed and seat.vehicleCharacter ~= nil then
                                    seat.vehicleCharacter:update(dt)
                                end
                            end
                        end
                    end

                    if self.spec_reverseDriving ~= nil and self.spec_reverseDriving.isChangingDirection then
                        local seat = spec.passengerSeats[spec.currentSeatIndex]

                        if seat ~= nil and seat.disabledReverseDriving then
                            self:leaveLocalPassengerSeat(false)
                        end
                    end
                end

                if self.isServer and spec.renderPassengerCameraLocations then
                    for _, seat in ipairs (spec.passengerSeats) do
                        local indoorCamera = seat.cameras[2]

                        if indoorCamera ~= nil then
                            if seat.creatorInfo then
                                local x, y, z = getWorldTranslation(seat.node)
                                local name = seat.creatorInfo.linkNodeStr or getName(getParent(getParent(indoorCamera.cameraNode)))

                                Utils.renderTextAtWorldPosition(x, y + seat.nicknameOffset, z, name, getCorrectTextSize(0.02), 0)
                            end

                            DebugUtil.drawDebugCube(indoorCamera.cameraNode, 0.25, 0.25, 0.35)
                            DebugUtil.drawDebugCube(indoorCamera.cameraNode, 0.15, 0.15, 0.15, nil, nil, nil, 0, 0, -0.25)
                        end
                    end
                end
            end
        end

        -- Fix for changing seats (Driver > Passenger) that causes duplicate hotspots when deleting a vehicle entered by passenger and also 'Inside Steering Rotation' to be used
        local oldEnterVehiclePassengerSeat = storeSpecFunction(pdlc_kubotaPack.EnterablePassenger, "enterVehiclePassengerSeat")

        if oldEnterVehiclePassengerSeat ~= nil then
            function pdlc_kubotaPack.EnterablePassenger:enterVehiclePassengerSeat(seatOwner, ...)
                oldEnterVehiclePassengerSeat(self, seatOwner, ...)

                if seatOwner then
                    g_currentMission.controlledVehicle = nil
                end
            end
        end

        -- Not sure the purpose here? Seems to be a way to stop sale or reset when passenger is entered but also stops AI being hired when there is a passenger, more important I think :-)
        local oldGetIsInUse = storeSpecFunction(pdlc_kubotaPack.EnterablePassenger, "getIsInUse")

        if oldGetIsInUse ~= nil then
            function pdlc_kubotaPack.EnterablePassenger:getIsInUse(superFunc, connection)
                return superFunc(self, connection)
            end
        end

        -- Reverse driving option
        local oldGetIsPassengerSeatAvailable = storeSpecFunction(pdlc_kubotaPack.EnterablePassenger, "getIsPassengerSeatAvailable")

        if oldGetIsPassengerSeatAvailable ~= nil then
            function pdlc_kubotaPack.EnterablePassenger:getIsPassengerSeatAvailable(seat, ...)
                if seat.disabledReverseDriving and self.spec_reverseDriving ~= nil and (self.spec_reverseDriving.isReverseDriving or self.spec_reverseDriving.isChangingDirection) then
                    return false
                end

                return oldGetIsPassengerSeatAvailable(self, seat, ...)
            end
        end

        -- Temporary fix until @Giants fixes this, sound is not updated no matter the camera settings
        local oldEnablePassengerActiveCamera = storeSpecFunction(pdlc_kubotaPack.EnterablePassenger, "enablePassengerActiveCamera")

        if oldEnablePassengerActiveCamera ~= nil then
            function pdlc_kubotaPack.EnterablePassenger:enablePassengerActiveCamera(...)
                oldEnablePassengerActiveCamera(self, ...)

                local activeCamera = self.spec_enterable.activeCamera

                if activeCamera ~= nil then
                    g_soundManager:setIsIndoor(not activeCamera.useOutdoorSounds)
                    g_currentMission.ambientSoundSystem:setIsIndoor(not activeCamera.useOutdoorSounds)
                end
            end
        end

        -- This is SP only. Requested by 'MichelMichel' to allow an AI worker to travel as a passenger when in a vehicle
        if g_dedicatedServerInfo == nil and not g_careerScreen.isMultiplayer then
            local oldOnRegisterActionEvents = storeSpecFunction(pdlc_kubotaPack.EnterablePassenger, "onRegisterActionEvents")
            local oldRegisterEventListeners = storeSpecFunction(pdlc_kubotaPack.EnterablePassenger, "registerEventListeners")

            if oldOnRegisterActionEvents ~= nil and oldRegisterEventListeners ~= nil then
                source(modDirectory .. "scripts/EnterablePassengerAIExtension.lua")

                g_enterablePassengerAIExtension = EnterablePassengerAIExtension.new(versionString, buildId, modName, modDirectory, modSettingsDirectory)
                g_enterablePassengerAIExtension.developmentMode = enterablePassengerExtension.developmentMode

                if g_enterablePassengerAIExtension:load() then
                    function pdlc_kubotaPack.EnterablePassenger:onEnterVehicle()
                        if g_enterablePassengerAIExtension ~= nil and g_enterablePassengerAIExtension.active then
                            g_enterablePassengerAIExtension:setAIPassengerState(self, true)
                        end
                    end

                    function pdlc_kubotaPack.EnterablePassenger:onLeaveVehicle()
                        if g_enterablePassengerAIExtension ~= nil then
                            g_enterablePassengerAIExtension:setAIPassengerState(self, false)
                        end
                    end

                    function pdlc_kubotaPack.EnterablePassenger:onReverseDirectionChanged()
                        if g_enterablePassengerAIExtension ~= nil and g_enterablePassengerAIExtension.active then
                            if self.spec_reverseDriving ~= nil and not self.spec_reverseDriving.isReverseDriving then
                                g_enterablePassengerAIExtension:setAIPassengerState(self, true)
                            end
                        end
                    end

                    function pdlc_kubotaPack.EnterablePassenger:onStartReverseDirectionChange()
                        if g_enterablePassengerAIExtension ~= nil and g_enterablePassengerAIExtension.active then
                            if self.spec_reverseDriving ~= nil and self.spec_reverseDriving.isReverseDriving then
                                local spec = self.spec_enterablePassenger

                                if spec.aiPassengerSeatIndex ~= nil then
                                    local seat = spec.passengerSeats[spec.aiPassengerSeatIndex]

                                    if seat ~= nil and seat.disabledReverseDriving then
                                        g_enterablePassengerAIExtension:setAIPassengerState(self, false)
                                    end
                                end
                            end
                        end
                    end

                    function pdlc_kubotaPack.EnterablePassenger.registerEventListeners(vehicleType)
                        oldRegisterEventListeners(vehicleType)

                        SpecializationUtil.registerEventListener(vehicleType, "onEnterVehicle", pdlc_kubotaPack.EnterablePassenger)
                        SpecializationUtil.registerEventListener(vehicleType, "onLeaveVehicle", pdlc_kubotaPack.EnterablePassenger)

                        if vehicleType.eventListeners ~= nil then
                            if vehicleType.eventListeners["onReverseDirectionChanged"] ~= nil then
                                SpecializationUtil.registerEventListener(vehicleType, "onReverseDirectionChanged", pdlc_kubotaPack.EnterablePassenger)
                            end

                            if vehicleType.eventListeners["onStartReverseDirectionChange"] ~= nil then
                                SpecializationUtil.registerEventListener(vehicleType, "onStartReverseDirectionChange", pdlc_kubotaPack.EnterablePassenger)
                            end
                        end
                    end

                    function pdlc_kubotaPack.EnterablePassenger:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
                        local spec = self.spec_enterablePassenger

                        if spec.available and self:getIsEntered() and self:getIsActiveForInput(true, true) and g_enterablePassengerAIExtension ~= nil then
                            local showInputBindings = g_enterablePassengerAIExtension.showInputBindings
                            local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.TOGGLE_AI_PASSENGER, self, EnterablePassengerAIExtension.actionEventToggleAIPassenger, false, true, false, true, nil)

                            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)
                            g_inputBinding:setActionEventTextVisibility(actionEventId, showInputBindings)

                            _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.NEXT_AI_PASSENGER, self, EnterablePassengerAIExtension.actionEventNextAIPassenger, false, true, false, true, nil)

                            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)
                            g_inputBinding:setActionEventTextVisibility(actionEventId, showInputBindings and g_enterablePassengerAIExtension.active)
                        end

                        oldOnRegisterActionEvents(self, isActiveForInput, isActiveForInputIgnoreSelection)
                    end
                else
                    g_enterablePassengerAIExtension:delete()
                    g_enterablePassengerAIExtension = nil

                    Logging.warning("[Passenger Extension] Failed to load AI passenger characters, to avoid errors feature has been disabled!")
                end
            end
        end

        return true
    end

    return false
end

local function validateMod()
    if g_globalMods ~= nil then
        local mod = g_modManager:getModByName(modName)

        if g_globalMods.enterablePassengerExtension ~= nil then
            Logging.warning("[Passenger Extension] Validation of '%s' failed, already loaded by '%s'.", mod.modName, g_globalMods.enterablePassengerExtension.modName)

            return false
        end

        versionString = mod.version or versionString

        if mod.modName == "FS22_PassengerExtension" or mod.modName == "FS22_PassengerExtension_update" then
            if mod.author ~= nil and #mod.author == 3 then
                return true
            end
        end

        validationFail = {
            startUpdateTime = 2000,

            update = function(self, dt)
                self.startUpdateTime = self.startUpdateTime - dt

                if self.startUpdateTime < 0 then
                    local text = string.format(g_i18n:getText("passengerExtension_loadError", mod.modName), mod.modName, mod.author or "Unknown")

                    if g_dedicatedServerInfo == nil then
                        if not g_gui:getIsGuiVisible() then
                            g_gui:showYesNoDialog({
                                title = string.format("%s - Version %s", mod.title, versionString),
                                text = text,
                                dialogType = DialogElement.TYPE_LOADING,
                                callback = self.openModHubLink,
                                yesText = g_i18n:getText("button_ok"),
                                noText = g_i18n:getText("button_modHubDownload")
                            })
                        end
                    else
                        print("\n" .. text .. "\n    - https://farming-simulator.com/mods.php?lang=en&country=be&title=fs2022&filter=org&org_id=129652&page=0" .. "\n")
                        self.openModHubLink(true)
                    end
                end
            end,

            openModHubLink = function(ignore)
                if ignore == false then
                    local language = g_languageShort
                    local link = "mods.php?lang=en&country=be&title=fs2022&filter=org&org_id=129652&page=0"
                    if language == "de" or language == "fr" then
                        link = "mods.php?lang=" .. language .. "&country=be&title=fs2022&filter=org&org_id=129652&page=0"
                    end

                    openWebFile(link, "")
                end

                removeModEventListener(validationFail)
                validationFail = nil
            end
        }

        addModEventListener(validationFail)
    end

    return false
end

local function createDLCMessage(message, title)
    local mod = g_modManager:getModByName(modName)

    validationFail = {
        startUpdateTime = 2000,
        title = mod.title
    }

    validationFail.update = function(self, dt)
        self.startUpdateTime = self.startUpdateTime - dt

        if self.startUpdateTime < 0 then
            local text = string.format("%s\n\n- Kubota Pack -", g_i18n:getText("ui_notAllModsAvailable"))

            if g_dedicatedServerInfo == nil then
                if not g_gui:getIsGuiVisible() then
                    g_gui:showYesNoDialog({
                        title = string.format("%s - Version %s", self.title, versionString),
                        text = text,
                        dialogType = DialogElement.TYPE_LOADING,
                        callback = self.openModHubLink,
                        yesText = g_i18n:getText("button_ok"),
                        noText = g_i18n:getText("button_modHubDownload")
                    })
                end
            else
                print("\n" .. text .. "\n    - https://farming-simulator.com/dlc-detail.php?&dlc_id=fs22kubota" .. " -\n")
                self.openModHubLink(true)
            end
        end
    end

    validationFail.openModHubLink = function(ignore)
        if ignore == false then
            local language = g_languageShort
            local link = "dlc-detail.php?lang=en&country=be&dlc_id=fs22kubota"

            if language == "de" or language == "fr" then
                link = "dlc-detail.php?lang=" .. language .. "&country=be&dlc_id=fs22kubota"
            end

            openWebFile(link, "")
        end

        removeModEventListener(validationFail)
        validationFail = nil
    end

    addModEventListener(validationFail)
end

local function wildlifeSpawnerUpdate(self, superFunc, dt)
    -- remove outdated areas of interest
    self:updateAreaOfInterest(dt)

    if self.isEnabled then
        -- add animals
        self.nextCheckTime = self.nextCheckTime - dt

        if (self.nextCheckTime < 0.0) then
            self.nextCheckTime = self.checkTimeInterval
            self:updateSpawner()
        end

        -- remove animals too far away
        self:removeFarAwayAnimals()

        self.playerNode = nil

        if g_currentMission.controlPlayer and g_currentMission.player ~= nil then
            self.playerNode = g_currentMission.player.rootNode
        elseif g_currentMission.controlledVehicle ~= nil then
            self.playerNode = g_currentMission.controlledVehicle.rootNode
        elseif g_currentMission.enteredPassengerVehicle ~= nil then
            -- @Giants This is required also because when a second passenger enters they were never a vehicle controller and are no longer a player
            -- Also retaining 'g_currentMission.controlledVehicle' when swapping from driver seat breaks AI and also the steering  (IKChains) as mentioned in notes above
            self.playerNode = g_currentMission.enteredPassengerVehicle.rootNode
        end

        -- update animal context
        for _, area in pairs(self.areas) do
            for _, species in pairs(area.species) do
                if species.classType == "companionAnimal" then
                    for _, spawn in pairs(species.spawned) do
                        if spawn.spawnId ~= nil and self.playerNode ~= nil then
                            setCompanionDaytime(spawn.spawnId, g_currentMission.environment.dayTime)

                            if spawn.avoidNode ~= self.playerNode then
                                setCompanionAvoidPlayer(spawn.spawnId, self.playerNode, self.avoidDistance)
                                spawn.avoidNode = self.playerNode
                            end
                        end
                    end
                elseif species.classType == "lightWildlife" then
                    species.lightWildlife:update(dt)
                end
            end
        end

        if self.debugShow then
            self:debugDraw() -- debug
        end
    end
end

local function guiTopDownCameraSetControlledVehicle(self, superFunc, vehicle)
    -- Set the vehicle to the passenger vehicle so the construction screen works correctly
    superFunc(self, vehicle or g_currentMission.enteredPassengerVehicle)
end

local function ingameMapUpdatePlayerPosition(self, superFunc)
    -- No passenger update when in the construction screen
    if self.topDownCamera ~= nil then
        local playerPosX, _, playerPosZ, playerRotation, playerVelocity = self.topDownCamera:determineMapPosition()

        self.playerRotation = playerRotation
        self.playerVelocity = playerVelocity

        self.normalizedPlayerPosX = MathUtil.clamp((playerPosX + self.worldCenterOffsetX) / self.worldSizeX, 0, 1)
        self.normalizedPlayerPosZ = MathUtil.clamp((playerPosZ + self.worldCenterOffsetZ) / self.worldSizeZ, 0, 1)

        return
    end

    superFunc(self)
end

local function playerMoveTo(self, x, y, z, isAbsolute, isRootNode)
    -- When a player is a passenger they cannot teleport to hotspots etc, this fixes that
    if self.isOwner and g_currentMission.enteredPassengerVehicle ~= nil and g_client ~= nil then
        g_currentMission.enteredPassengerVehicle:leaveLocalPassengerSeat(false)
    end
end

local function unload(mission)
    g_enterablePassengerExtension = nil

    if g_globalMods ~= nil then
        g_globalMods.enterablePassengerExtension = nil
    end

    if enterablePassengerExtension ~= nil then
        enterablePassengerExtension:delete()
        enterablePassengerExtension = nil
    end

    if g_enterablePassengerAIExtension ~= nil then
        g_enterablePassengerAIExtension:delete()
        g_enterablePassengerAIExtension = nil
    end

    -- Restore original functions
    for i = #oldSpecFunctions, 1, -1 do
        local data = oldSpecFunctions[i]

        if data.object ~= nil and data.originalFunc ~= nil then
            data.object[data.funcName] = data.originalFunc
        end

        oldSpecFunctions[i] = nil
    end
end

local function init()
    if g_iconGenerator == nil and validateMod() then
        source(modDirectory .. "scripts/EnterablePassengerExtension.lua")

        local oldFinalizeTypes = TypeManager.finalizeTypes

        function TypeManager:finalizeTypes(...)
            if self.typeName == "vehicle" and isActive() then
                -- Only load with Genuine DLC. I do not support theft ;-)
                if isDlcActive() then
                    if overwriteSpecializationFunctions() then
                        WildlifeSpawner.update = Utils.overwrittenFunction(WildlifeSpawner.update, wildlifeSpawnerUpdate)
                        GuiTopDownCamera.setControlledVehicle = Utils.overwrittenFunction(GuiTopDownCamera.setControlledVehicle, guiTopDownCameraSetControlledVehicle)
                        IngameMap.updatePlayerPosition = Utils.overwrittenFunction(IngameMap.updatePlayerPosition, ingameMapUpdatePlayerPosition)
                        Player.moveTo = Utils.prependedFunction(Player.moveTo, playerMoveTo)

                        g_asyncTaskManager:addTask(function ()
                            if enterablePassengerExtension ~= nil then
                                enterablePassengerExtension:loadGlobalVehicles(true)
                            end
                        end)
                    else
                        unload()
                    end
                else
                    createDLCMessage()
                    unload()
                end
            end

            oldFinalizeTypes(self, ...)
        end

        enterablePassengerExtension = EnterablePassengerExtension.new(modName, modDirectory)

        enterablePassengerExtension.developmentMode = buildId < 1
        enterablePassengerExtension.versionString = versionString
        enterablePassengerExtension.buildId = buildId

        enterablePassengerExtension:load()

        g_enterablePassengerExtension = enterablePassengerExtension
        g_globalMods.enterablePassengerExtension = g_enterablePassengerExtension

        if g_globalMods.easyDevControls ~= nil and g_globalMods.easyDevControls.godMode ~= nil then
            if g_globalMods.easyDevControls.godMode:enableModDevelopment("FS22_PassengerExtension", enterablePassengerExtension) then
                enterablePassengerExtension.developmentMode = true
            end
        end

        FSBaseMission.delete = Utils.appendedFunction(FSBaseMission.delete, unload)
    end
end

init()
