--[[
Copyright (C) GtX (Andy), 2022

Author: GtX | Andy
Date: 12.09.2022
Revision: FS22-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Idea / Suggestion: MichelMichel

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]


EnterablePassengerAIExtension = {}

local EnterablePassengerAIExtension_mt = Class(EnterablePassengerAIExtension)
local initiated = false

function EnterablePassengerAIExtension.new(versionString, buildId, customEnvironment, baseDirectory, modSettingsDirectory)
    local self = setmetatable({}, EnterablePassengerAIExtension_mt)

    self.styleKeys = {}
    self.lastStyleId = -1

    self.currentStyle = nil
    self.playerModel = nil

    self.active = false
    self.currentStyleId = 1
    self.showInputBindings = false

    self.versionString = versionString
    self.buildId = buildId

    self.customEnvironment = customEnvironment
    self.baseDirectory = baseDirectory

    if customEnvironment:endsWith("_update") then
        modSettingsDirectory = string.format("%smodSettings/FS22_PassengerExtension/", getUserProfileAppPath())
    elseif modSettingsDirectory == nil or modSettingsDirectory == "" then
        modSettingsDirectory = string.format("%smodSettings/%s/", getUserProfileAppPath(), customEnvironment)
    end

    self.modSettingsDirectory = modSettingsDirectory

    addConsoleCommand("gtxReloadAIPassengerStyles", "Reloads the AI passenger styles XML [SP Only]", "consoleCommandReloadAIPassengerStyles", self)

    return self
end

function EnterablePassengerAIExtension:load()
    local xmlFile = self:loadStylesXML()
    local styleKeys = {}

    if xmlFile ~= nil then
        xmlFile:iterate("aiStyles.aiStyle", function(index, styleKey)
            table.insert(styleKeys, styleKey)
        end)

        xmlFile:delete()
    end

    self.styleKeys = styleKeys

    if #self.styleKeys > 0 then
        EnterablePassengerAIExtension.overwriteFunctions()

        return true
    end

    return false
end

function EnterablePassengerAIExtension:delete()
    if self.playerModel ~= nil then
        self.playerModel:delete()
        self.playerModel = nil
    end

    if self.currentStyle ~= nil then
        self.currentStyle:delete()
        self.currentStyle = nil
    end

    removeConsoleCommand("gtxReloadAIPassengerStyles")
end

function EnterablePassengerAIExtension:loadStylesXML()
    if self.developmentMode then
        return XMLFile.load("aiStyles", self.baseDirectory .. "shared/aiStyles.xml")
    end

    local filepath = self.modSettingsDirectory .. "aiStyles.xml"

    if not fileExists(filepath) then
        createFolder(self.modSettingsDirectory)
        copyFile(self.baseDirectory .. "shared/aiStyles.xml", filepath, false)
    end

    return XMLFile.load("aiStyles", filepath)
end

function EnterablePassengerAIExtension:getPassengerStyle()
    if self.currentStyle == nil or self.currentStyleId ~= self.lastStyleId then
        self.lastStyleId = self.currentStyleId

        local key = self.styleKeys[self.currentStyleId]

        if key == nil then
            self.currentStyleId = 1
            key = self.styleKeys[self.currentStyleId]
        end

        if self.currentStyle ~= nil then
            self.currentStyle:delete()
        end

        local xmlFile = self:loadStylesXML()

        if xmlFile ~= nil then
            if xmlFile:getString(key .. "#filename") ~= nil then
                self.currentStyle = PlayerStyle.new()
                self.currentStyle:loadFromXMLFile(xmlFile, key)
            else
                self.currentStyle = PlayerStyle.defaultStyle()
                Logging.warning("[Passenger Extension] Failed to load AI passenger style at %s, using default style!", key)
            end

            xmlFile:delete()
        else
            self.currentStyle = PlayerStyle.defaultStyle()
            Logging.warning("[Passenger Extension] Failed to load AI passenger style as XML file is missing, Using default style!")
        end

        if self.playerModel ~= nil then
            self.playerModel:delete()
        end

        self.playerModel = PlayerModel.new()

        self.playerModel:load(self.currentStyle.xmlFilename, false, false, false, function(_, success, args)
            if success then
                if self.playerModel.rootNode ~= nil then
                    self.playerModel:setStyle(self.currentStyle, false, nil)
                    self.playerModel:setVisibility(false)
                else
                    self.playerModel:delete()
                    self.playerModel = nil
                end
            else
                self.playerModel:delete()
                self.playerModel = nil
            end
        end)
    end

    return self.currentStyle
end

function EnterablePassengerAIExtension:setAIPassengerState(vehicle, active)
    local spec = vehicle.spec_enterablePassenger

    if spec ~= nil and spec.available then
        if active then
            local seatIndex = vehicle:getFirstAvailablePassengerSeat()
            local seat = seatIndex ~= nil and spec.passengerSeats[seatIndex] or nil

            if seat ~= nil then
                seat.playerStyle = self:getPassengerStyle()
                seat.isUsed = true
                seat.userId = 0

                vehicle:setPassengerSeatCharacter(seatIndex, seat.playerStyle)
                spec.aiPassengerSeatIndex = seatIndex
            end
        else
            if spec.aiPassengerSeatIndex ~= nil then
                local seat = spec.passengerSeats[spec.aiPassengerSeatIndex]

                if seat ~= nil then
                    vehicle:setPassengerSeatCharacter(spec.aiPassengerSeatIndex, nil)

                    seat.playerStyle = nil
                    seat.isUsed = false
                    seat.userId = nil
                end

                spec.aiPassengerSeatIndex = nil
            end
        end
    end
end

function EnterablePassengerAIExtension:consoleCommandReloadAIPassengerStyles()
    if g_currentMission == nil then
        return "Not possible until game is loaded!"
    end

    if self:load() then
        if self.gameSettingsFrame ~= nil and self.gameSettingsFrame.multiAIPassengerStyle ~= nil then
            local aiPassengerStyleTexts = {}
            local characterText = g_i18n:getText("character_body_name")

            for i = 1, #self.styleKeys do
                aiPassengerStyleTexts[i] = string.format(characterText, tostring(i))
            end

            self.gameSettingsFrame.multiAIPassengerStyle:setTexts(aiPassengerStyleTexts)
        end

        self.lastStyleId = -1

        if self.currentStyleId > #self.styleKeys then
            self.currentStyleId = 1
        end

        local vehicle = g_currentMission.controlledVehicle

        if vehicle ~= nil then
            local spec = vehicle.spec_enterablePassenger

            if spec ~= nil and spec.aiPassengerSeatIndex ~= nil then
                local currentSeat = spec.passengerSeats[spec.aiPassengerSeatIndex]

                if currentSeat ~= nil then
                    currentSeat.playerStyle = self:getPassengerStyle()
                    vehicle:setPassengerSeatCharacter(spec.aiPassengerSeatIndex, currentSeat.playerStyle)
                end
            end
        end

        if self.developmentMode then
            return "DevInfo: Reloaded AI Styles XML file at '" .. tostring(self.baseDirectory) .. "shared/aiStyles.xml'.  Number of available styles: " .. tostring(#self.styleKeys)
        end

        return "Info: Reloaded AI Styles XML file at '" .. tostring(self.modSettingsDirectory) .. "aiStyles.xml'.  Number of available styles: " .. tostring(#self.styleKeys)
    end

    return "Failed to reload AI Passenger styles!"
end

function EnterablePassengerAIExtension.actionEventNextAIPassenger(vehicle, actionName, inputValue, callbackState, isAnalog, isMouse)
    local spec = vehicle.spec_enterablePassenger

    if spec ~= nil and spec.aiPassengerSeatIndex ~= nil then
        local currentSeat = spec.passengerSeats[spec.aiPassengerSeatIndex]
        local aiPassengerExtension = g_enterablePassengerAIExtension

        if currentSeat ~= nil and aiPassengerExtension ~= nil then
            aiPassengerExtension.currentStyleId = aiPassengerExtension.currentStyleId + 1

            if aiPassengerExtension.currentStyleId > #aiPassengerExtension.styleKeys then
                aiPassengerExtension.currentStyleId = 1
            end

            currentSeat.playerStyle = aiPassengerExtension:getPassengerStyle()

            vehicle:setPassengerSeatCharacter(spec.aiPassengerSeatIndex, currentSeat.playerStyle)
        end
    end
end

function EnterablePassengerAIExtension.actionEventToggleAIPassenger(vehicle, actionName, inputValue, callbackState, isAnalog, isMouse)
    local spec = vehicle.spec_enterablePassenger
    local aiPassengerExtension = g_enterablePassengerAIExtension

    if spec ~= nil and aiPassengerExtension ~= nil then
        aiPassengerExtension.active = not aiPassengerExtension.active
        aiPassengerExtension:setAIPassengerState(vehicle, aiPassengerExtension.active)

        local actionEvent = spec.actionEvents[InputAction.NEXT_AI_PASSENGER]

        if actionEvent ~= nil then
            g_inputBinding:setActionEventTextVisibility(actionEvent.actionEventId, aiPassengerExtension.showInputBindings and aiPassengerExtension.active)
        end
    end
end

function EnterablePassengerAIExtension.overwriteFunctions()
    if initiated then
        return
    end

    initiated = true

    InGameMenuGameSettingsFrame.onFrameOpen = Utils.overwrittenFunction(InGameMenuGameSettingsFrame.onFrameOpen, function (frame, superFunc, element)
        superFunc(frame, element)

        if g_enterablePassengerAIExtension ~= nil then
            local extension = g_enterablePassengerAIExtension

            if frame.checkAIPassenger == nil then
                extension.gameSettingsFrame = frame

                if frame.checkDirt ~= nil and frame.multiFuelUsage ~= nil then
                    local boxLayout = frame.boxLayout

                    for i = 1, #boxLayout.elements do
                        local element = boxLayout.elements[i]

                        if element:isa(TextElement) then
                            local header = element:clone(boxLayout)

                            header:setText(g_i18n:getText("ui_passengerExtension", extension.customEnvironment))
                            header:reloadFocusHandling(true)

                            break
                        end
                    end

                    local checkAIPassenger = frame.checkDirt:clone(boxLayout)

                    function checkAIPassenger.onClickCallback(_, state)
                        local aiPassengerExtension = g_enterablePassengerAIExtension

                        if aiPassengerExtension ~= nil then
                            aiPassengerExtension.active = state == CheckedOptionElement.STATE_CHECKED

                            local vehicle = g_currentMission.controlledVehicle

                            if vehicle ~= nil then
                                aiPassengerExtension:setAIPassengerState(vehicle, aiPassengerExtension.active)
                                vehicle:requestActionEventUpdate()
                            end
                        end
                    end

                    checkAIPassenger.id = "checkAIPassenger"
                    frame.checkAIPassenger = checkAIPassenger

                    checkAIPassenger.elements[4]:setText(g_i18n:getText("setting_aiPassengerState", extension.customEnvironment))
                    checkAIPassenger.elements[6]:setText(g_i18n:getText("toolTip_aiPassengerState", extension.customEnvironment))

                    checkAIPassenger:reloadFocusHandling(true)
                    checkAIPassenger:setIsChecked(extension.active)

                    local aiPassengerStyleTexts = {}
                    local characterText = g_i18n:getText("character_body_name")

                    for i = 1, #extension.styleKeys do
                        aiPassengerStyleTexts[i] = string.format(characterText, tostring(i))
                    end

                    local multiAIPassengerStyle = frame.multiFuelUsage:clone(boxLayout)

                    multiAIPassengerStyle:setTexts(aiPassengerStyleTexts)

                    function multiAIPassengerStyle.onClickCallback(_, state)
                        local aiPassengerExtension = g_enterablePassengerAIExtension

                        if aiPassengerExtension ~= nil then
                            if aiPassengerExtension.styleKeys[state] == nil then
                                state = 1
                            end

                            aiPassengerExtension.currentStyleId = state

                            local vehicle = g_currentMission.controlledVehicle

                            if vehicle ~= nil then
                                local spec = vehicle.spec_enterablePassenger

                                if spec ~= nil and spec.aiPassengerSeatIndex ~= nil then
                                    local currentSeat = spec.passengerSeats[spec.aiPassengerSeatIndex]

                                    if currentSeat ~= nil then
                                        currentSeat.playerStyle = aiPassengerExtension:getPassengerStyle()
                                        vehicle:setPassengerSeatCharacter(spec.aiPassengerSeatIndex, currentSeat.playerStyle)
                                    end
                                end
                            end
                        end
                    end

                    multiAIPassengerStyle.id = "multiAIPassengerStyle"
                    frame.multiAIPassengerStyle = multiAIPassengerStyle

                    multiAIPassengerStyle.elements[4]:setText(g_i18n:getText("setting_aiPassengerStyle", extension.customEnvironment))
                    multiAIPassengerStyle.elements[6]:setText(g_i18n:getText("toolTip_aiPassengerStyle", extension.customEnvironment))

                    multiAIPassengerStyle:reloadFocusHandling(true)
                    multiAIPassengerStyle:setState(extension.currentStyleId)

                    local checkAIPassengerInput = frame.checkDirt:clone(boxLayout)

                    function checkAIPassengerInput.onClickCallback(_, state)
                        local aiPassengerExtension = g_enterablePassengerAIExtension

                        if aiPassengerExtension ~= nil then
                            aiPassengerExtension.showInputBindings = state == CheckedOptionElement.STATE_CHECKED

                            local vehicle = g_currentMission.controlledVehicle

                            if vehicle ~= nil then
                                vehicle:requestActionEventUpdate()
                            end
                        end
                    end

                    checkAIPassengerInput.id = "checkAIPassengerInput"
                    frame.checkAIPassengerInput = checkAIPassengerInput

                    checkAIPassengerInput.elements[4]:setText(g_i18n:getText("setting_aiPassengerInput", extension.customEnvironment))
                    checkAIPassengerInput.elements[6]:setText(g_i18n:getText("toolTip_aiPassengerInput", extension.customEnvironment))

                    checkAIPassengerInput:reloadFocusHandling(true)
                    checkAIPassengerInput:setIsChecked(extension.showInputBindings)

                    -- Fix linked elements so 'Pause Button' is targeted from bottom and last element from 'Pause Button'
                    local lastElement = boxLayout.elements[#boxLayout.elements]

                    FocusManager:linkElements(lastElement, FocusManager.BOTTOM, frame.buttonPauseGame)
                    FocusManager:linkElements(frame.buttonPauseGame, FocusManager.TOP, lastElement)

                    boxLayout:invalidateLayout()
                end
            else
                frame.checkAIPassenger:setIsChecked(extension.active)

                if frame.multiAIPassengerStyle ~= nil then
                    frame.multiAIPassengerStyle:setState(extension.currentStyleId)
                end

                if frame.checkAIPassengerInput ~= nil then
                    frame.checkAIPassengerInput:setIsChecked(extension.showInputBindings)
                end
            end
        end
    end)

    FSCareerMissionInfo.saveToXMLFile = Utils.appendedFunction(FSCareerMissionInfo.saveToXMLFile, function(missionInfo)
        if missionInfo.isValid and g_enterablePassengerAIExtension ~= nil then
            local xmlFilename = missionInfo.savegameDirectory .. "/passengerExtension.xml"
            local xmlFile = createXMLFile("PassengerExtensionXML", xmlFilename, "passengerExtension")

            if xmlFile ~= nil and xmlFile ~= 0 then
                local extension = g_enterablePassengerAIExtension

                setXMLString(xmlFile, "passengerExtension#version", extension.versionString or "0.0.0.0")
                setXMLFloat(xmlFile, "passengerExtension#buildId", extension.buildId or 0)

                setXMLBool(xmlFile, "passengerExtension.aiPassenger#active", Utils.getNoNil(extension.active, false))
                setXMLInt(xmlFile, "passengerExtension.aiPassenger#styleIndex", Utils.getNoNil(extension.currentStyleId, 1))
                setXMLBool(xmlFile, "passengerExtension.aiPassenger#showInput", Utils.getNoNil(extension.showInputBindings, false))

                saveXMLFile(xmlFile)
                delete(xmlFile)
            end
        end
    end)

    FSBaseMission.loadMapFinished = Utils.appendedFunction(FSBaseMission.loadMapFinished, function(mission)
        local missionInfo = mission.missionInfo
        local extension = g_enterablePassengerAIExtension

        if missionInfo ~= nil and missionInfo.isValid and missionInfo.savegameDirectory ~= nil and extension ~= nil then
            local xmlFilename = missionInfo.savegameDirectory .. "/passengerExtension.xml"

            if fileExists(xmlFilename) then
                local xmlFile = loadXMLFile("PassengerExtensionXML", xmlFilename)

                if xmlFile ~= nil and xmlFile ~= 0 then
                    extension.active = Utils.getNoNil(getXMLBool(xmlFile, "passengerExtension.aiPassenger#active"), false)
                    extension.currentStyleId = Utils.getNoNil(getXMLInt(xmlFile, "passengerExtension.aiPassenger#styleIndex"), 1)

                    if extension.styleKeys[extension.currentStyleId] == nil then
                        extension.currentStyleId = 1
                    end

                    extension.showInputBindings = Utils.getNoNil(getXMLBool(xmlFile, "passengerExtension.aiPassenger#showInput"), false)

                    delete(xmlFile)
                end
            end
        end
    end)
end
