--[[
	Author: 	derelky
	Date:		06.03.2019
	Version:	1.0
	
	History:
				v1.0 @ 06.03.2019 - initial implementation in FS 19
]]

Mill = {}
Mill.ModName = g_currentModName

function Mill.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(FillUnit, specializations) and SpecializationUtil.hasSpecialization(Dischargeable, specializations)
end

function Mill.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "additives", Mill.additives)
end

function Mill.registerOverwrittenFunctions(vehicleType)
	--SpecializationUtil.registerOverwrittenFunction(vehicleType, "getDischargeFillType", Mill.getDischargeFillType)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "dischargeToGround", Mill.dischargeToGround)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "dischargeToObject", Mill.dischargeToObject)
end

function Mill.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", Mill)
end


function Mill.initSpecialization()
	local schema = Vehicle.xmlSchema

	schema:setXMLSpecializationType("Mill")
	schema:register(XMLValueType.INT, "vehicle.mill.additives#fillUnitIndex", "Additives fill unit index")
	schema:register(XMLValueType.FLOAT, "vehicle.mill.additives#usage", "Usage per picked up liter", 2.75e-05)
	schema:register(XMLValueType.FLOAT, "vehicle.mill.additives#bonusrate", "Bonus rate for additives", 1)
	schema:register(XMLValueType.STRING, "vehicle.mill.additives#fillTypes", "Fill types to apply additives")
	schema:setXMLSpecializationType()

end


function Mill.getSpecByName(self, specName, currentModName)
    local spec = self["spec_" .. Utils.getNoNil(currentModName, Mill.ModName) .. "." .. specName];

	if spec ~= nil then
        return spec;
    end;

    return self["spec_" .. specName];
end;


function Mill:onLoad(savegame)
	local spec_Mill = Mill.getSpecByName(self, "Mill");

	spec_Mill.additives = {
		fillUnitIndex = self.xmlFile:getValue("vehicle.mill.additives#fillUnitIndex")
	}
	spec_Mill.additives.available = self:getFillUnitByIndex(spec_Mill.additives.fillUnitIndex) ~= nil
	spec_Mill.additives.usage = self.xmlFile:getValue("vehicle.mill.additives#usage", 2.75e-05)
	spec_Mill.additives.bonusrate = self.xmlFile:getValue("vehicle.mill.additives#bonusrate", 1)
	local additivesFillTypeNames = self.xmlFile:getValue("vehicle.mill.additives#fillTypes")
	spec_Mill.additives.fillTypes = g_fillTypeManager:getFillTypesByNames(additivesFillTypeNames, "Warning: '" .. self.xmlFile:getFilename() .. "' has invalid fillType '%s'.")
	
end


function Mill:additives(FillType, origdroppedLiters, factor)
	local spec = Mill.getSpecByName(self, "Mill");
	local droppedLiters = origdroppedLiters * factor

	if self.isServer and spec.additives.available then
		local fillTypeSupported = false

		for i = 1, #spec.additives.fillTypes do
			if FillType == spec.additives.fillTypes[i] then
				fillTypeSupported = true

				break
			end
		end

		if fillTypeSupported then
			local additivesFillLevel = self:getFillUnitFillLevel(spec.additives.fillUnitIndex)

			if additivesFillLevel > 0 then
				local usage = spec.additives.usage * droppedLiters

				if usage > 0 then 
					local availableUsage = math.min(additivesFillLevel / usage, 1)
					droppedLiters = droppedLiters * (spec.additives.bonusrate * availableUsage)
					factor = droppedLiters/origdroppedLiters
					self:addFillUnitFillLevel(self:getOwnerFarmId(), spec.additives.fillUnitIndex, -usage, self:getFillUnitFillType(spec.additives.fillUnitIndex), ToolType.UNDEFINED)
				end
			end
		end
	end
	return droppedLiters, factor
end



function Mill:dischargeToGround(superFunc, dischargeNode, emptyLiters)
	if emptyLiters == 0 then
		return 0, false, false
	end

	local fillType, factor = self:getDischargeFillType(dischargeNode)


	local fillLevel = self:getFillUnitFillLevel(dischargeNode.fillUnitIndex)
	local minLiterToDrop = g_densityMapHeightManager:getMinValidLiterValue(fillType)
	dischargeNode.litersToDrop = math.min(dischargeNode.litersToDrop + emptyLiters, math.max(dischargeNode.emptySpeed * 250, minLiterToDrop))
	local minDropReached = minLiterToDrop < dischargeNode.litersToDrop
	local hasMinDropFillLevel = minLiterToDrop < fillLevel
	local info = dischargeNode.info
	local dischargedLiters = 0
	local sx, sy, sz = localToWorld(info.node, -info.width, 0, info.zOffset)
	local ex, ey, ez = localToWorld(info.node, info.width, 0, info.zOffset)
	sy = sy + info.yOffset
	ey = ey + info.yOffset

	if info.limitToGround then
		sy = math.max(getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, sx, 0, sz) + 0.1, sy)
		ey = math.max(getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, ex, 0, ez) + 0.1, ey)
	end
	local toDrop = dischargeNode.litersToDrop
	toDrop, factor = self:additives(fillType,dischargeNode.litersToDrop, factor)
	local dropped, lineOffset = DensityMapHeightUtil.tipToGroundAroundLine(self, toDrop, fillType, sx, sy, sz, ex, ey, ez, info.length, nil, dischargeNode.lineOffset, true, nil, true)
	dropped = dropped / factor
	dischargeNode.lineOffset = lineOffset
	dischargeNode.litersToDrop = dischargeNode.litersToDrop - dropped

	if dropped > 0 then
		local unloadInfo = self:getFillVolumeUnloadInfo(dischargeNode.unloadInfoIndex)
		dischargedLiters = self:addFillUnitFillLevel(self:getOwnerFarmId(), dischargeNode.fillUnitIndex, -dropped, self:getFillUnitFillType(dischargeNode.fillUnitIndex), ToolType.UNDEFINED, unloadInfo)
	end

	fillLevel = self:getFillUnitFillLevel(dischargeNode.fillUnitIndex)

	if fillLevel > 0 and fillLevel <= minLiterToDrop then
		dischargeNode.litersToDrop = minLiterToDrop
	end

	return dischargedLiters, minDropReached, hasMinDropFillLevel
end

function Mill:dischargeToObject(superFunc, dischargeNode, emptyLiters, object, targetFillUnitIndex)
	local fillType, factor = self:getDischargeFillType(dischargeNode)
	local supportsFillType = object:getFillUnitSupportsFillType(targetFillUnitIndex, fillType)
	local dischargedLiters = 0

	if supportsFillType then
		local allowFillType = object:getFillUnitAllowsFillType(targetFillUnitIndex, fillType)

		if allowFillType then
			dischargeNode.currentDischargeObject = object
			local toDrop = emptyLiters
			toDrop, factor = self:additives(fillType,emptyLiters, factor)
			local delta = object:addFillUnitFillLevel(self:getActiveFarm(), targetFillUnitIndex, toDrop, fillType, dischargeNode.toolType, dischargeNode.info)
			delta = delta / factor
			local unloadInfo = self:getFillVolumeUnloadInfo(dischargeNode.unloadInfoIndex)
			dischargedLiters = self:addFillUnitFillLevel(self:getOwnerFarmId(), dischargeNode.fillUnitIndex, -delta, self:getFillUnitFillType(dischargeNode.fillUnitIndex), ToolType.UNDEFINED, unloadInfo)
		end
	end

	return dischargedLiters
end