--
-- FS22 - FieldLease - InGameMenuFieldLease
--
-- @Interface: 1.2.0.0 b14651
-- @Author: KR-Softwares
-- @Date: 24.01.2022
-- @Version: 1.0.0.0
--
-- @Support: kr-softwares.com
--
-- Changelog:
-- 	v1.0.0.0 (24.01.2022):
--      - Convert from from FS19 to FS22
--

InGameMenuFieldLease = {}
InGameMenuFieldLease._mt = Class(InGameMenuFieldLease, TabbedMenuFrameElement)

InGameMenuFieldLease.LEASE_MULTIPLIER = 0.01
InGameMenuFieldLease.PROVISION = 0.1

InGameMenuFieldLease.MODE_BUY = 1
InGameMenuFieldLease.MODE_SELL = 2
InGameMenuFieldLease.MODE_LEASE = 3
InGameMenuFieldLease.MODE_LEASESTOP = 4

InGameMenuFieldLease.CONTROLS = {
	MAIN_BOX = "mainBox",
	TABLE_SLIDER = "tableSlider",
	HEADER_BOX = "tableHeaderBox",
	TABLE = "fieldLeaseTable",
	TABLE_TEMPLATE = "fieldLeaseRowTemplate",
}

function InGameMenuFieldLease.new(i18n, messageCenter)
	local self = InGameMenuFieldLease:superClass().new(nil, InGameMenuFieldLease._mt)

    self.name = "InGameMenuFieldLease"
    self.i18n = i18n
    self.messageCenter = messageCenter
    
    self:registerControls(InGameMenuFieldLease.CONTROLS)

    self.backButtonInfo = {
		inputAction = InputAction.MENU_BACK
	}
	self.btnBuySell = {
		text = self.i18n:getText("ui_btn_buy"),
		inputAction = InputAction.MENU_EXTRA_1,
		callback = function ()
			self:buySell()
		end
	}    
	self.btnLeaseEndlease = {
		text = self.i18n:getText("ui_btn_lease"),
		inputAction = InputAction.MENU_ACTIVATE,
		callback = function ()
			self:leaseEndlease()
		end
	}    
  
    self:setMenuButtonInfo({
        self.backButtonInfo,
        self.btnBuySell,
        self.btnLeaseEndlease
    })

    g_currentMission.inGameMenu.frameFieldLease = self

    return self
end

function InGameMenuFieldLease:delete()
	InGameMenuFieldLease:superClass().delete(self)
end

function InGameMenuFieldLease:copyAttributes(src)
    InGameMenuFieldLease:superClass().copyAttributes(self, src)
    self.i18n = src.i18n
end

function InGameMenuFieldLease:onGuiSetupFinished()
	InGameMenuFieldLease:superClass().onGuiSetupFinished(self)
	self.fieldLeaseTable:setDataSource(self)
	self.fieldLeaseTable:setDelegate(self)
end

function InGameMenuFieldLease:initialize()
end

function InGameMenuFieldLease:onFrameOpen()
	InGameMenuFieldLease:superClass().onFrameOpen(self)   
    self:updateContent()
	FocusManager:setFocus(self.fieldLeaseTable)
end

function InGameMenuFieldLease:onFrameClose()
	InGameMenuFieldLease:superClass().onFrameClose(self)   
end

function InGameMenuFieldLease:updateContent()  
	self.farmlandData = {}    
	for _, farmland in pairs(g_farmlandManager.farmlands) do    
		local farmlandO = {
            farmlandId = farmland.id,
            farmlandOwner = g_farmlandManager:getFarmlandOwner(farmland.id),
            area = farmland.areaInHa,
            price = farmland.price/ g_currentMission.environment.daysPerPeriod ,        
            leasePrice = (farmland.price * InGameMenuFieldLease.LEASE_MULTIPLIER) / g_currentMission.environment.daysPerPeriod,
            leasePriceStart = farmland.price * InGameMenuFieldLease.PROVISION,
            isLeased = farmland.isLeased,
            fields = self:getFieldsAsString(farmland.id)
		}

		table.insert(self.farmlandData, farmlandO)
	end

	self:sortList()
	self.fieldLeaseTable:reloadData()    
end

function InGameMenuFieldLease:getFieldsAsString(farmlandId)
    local fieldStr = ""
    for _,field in pairs(g_fieldManager.fields) do
        if field.farmland.id == farmlandId then
            if fieldStr == "" then
                fieldStr = field.fieldId
            else
                fieldStr = fieldStr .. ", " .. field.fieldId
            end        
        end        
    end
    return fieldStr
end

function InGameMenuFieldLease:sortList()
	local tmpList = {
		{
			title = g_i18n:getText("ui_own"),
			farmlands = {}
		},
		{
			title = g_i18n:getText("ui_ownLeased"),
			farmlands = {}
		},
		{
			title = g_i18n:getText("ui_free"),
			farmlands = {}
		},
		{
			title = g_i18n:getText("ui_sold"),
			farmlands = {}
		},
		{
			title = g_i18n:getText("ui_leased"),
			farmlands = {}
		}
	}

    local currentFarmId = -1
    local farm = g_farmManager:getFarmByUserId(g_currentMission.playerUserId)
    if farm ~= nil then
        currentFarmId = farm.farmId
    end
    
	for _, farmlandData in ipairs(self.farmlandData) do
        if farmlandData.farmlandOwner == FarmlandManager.NO_OWNER_FARM_ID then
            table.insert(tmpList[3].farmlands, farmlandData)     
        else
            if farmlandData.farmlandOwner == currentFarmId then
                if farmlandData.isLeased then
                    table.insert(tmpList[2].farmlands, farmlandData)                
                else
                    table.insert(tmpList[1].farmlands, farmlandData)
                end  
            else
                if farmlandData.isLeased then
                    table.insert(tmpList[5].farmlands, farmlandData)                
                else
                    table.insert(tmpList[4].farmlands, farmlandData)
                end  
            end 
        end       
    end
  
    self.sectionFarmlands = {}

    for _,e in pairs(tmpList) do
        if #e.farmlands > 0 then
            table.insert(self.sectionFarmlands, e)
        end
    end   
end

function InGameMenuFieldLease:getNumberOfSections()
	return #self.sectionFarmlands
end

function InGameMenuFieldLease:getNumberOfItemsInSection(list, section)
	return #self.sectionFarmlands[section].farmlands
end

function InGameMenuFieldLease:getTitleForSectionHeader(list, section)
	return self.sectionFarmlands[section].title
end

function InGameMenuFieldLease:populateCellForItemInSection(list, section, index, cell)
	local farmlandData = self.sectionFarmlands[section].farmlands[index]    
	cell:getAttribute("farmland"):setText(farmlandData.farmlandId)
	cell:getAttribute("size"):setText(g_i18n:formatNumber(farmlandData.area, 2))	
	cell:getAttribute("fields"):setText(farmlandData.fields)
	cell:getAttribute("price"):setText(g_i18n:formatMoney(farmlandData.price))	
	cell:getAttribute("leasePrice"):setText(g_i18n:formatMoney(farmlandData.leasePrice))	
end

function InGameMenuFieldLease:onListSelectionChanged(list, section, index)
	local sectionFarmlands = self.sectionFarmlands[section]    
	if sectionFarmlands ~= nil and sectionFarmlands.farmlands[index] ~= nil then   
        self.currentFarmland = sectionFarmlands.farmlands[index]        

        local farmland = g_farmlandManager:getFarmlandById(self.currentFarmland.farmlandId)
        local farmlandOwner = g_farmlandManager:getFarmlandOwner(self.currentFarmland.farmlandId)

        local currentFarmId = -1
        local farm = g_farmManager:getFarmByUserId(g_currentMission.playerUserId)
        if farm ~= nil then
            currentFarmId = farm.farmId
        end
        
        if farmlandOwner == FarmlandManager.NOT_BUYABLE_FARM_ID then
            self.btnBuySell.disabled = true
            self.btnLeaseEndlease.disabled = true
        elseif farmlandOwner == FarmlandManager.NO_OWNER_FARM_ID  then
            self.btnBuySell.disabled = false
            self.btnLeaseEndlease.disabled = false
            self.btnBuySell.text = g_i18n:getText("ui_btn_buy")
            self.btnLeaseEndlease.text = g_i18n:getText("ui_btn_lease")
        else
            if farmlandOwner == currentFarmId then
                if farmland.isLeased then
                    self.btnBuySell.disabled = true
                    self.btnLeaseEndlease.disabled = false
                    self.btnLeaseEndlease.text = g_i18n:getText("ui_btn_endlease")
                else
                    self.btnBuySell.disabled = false
                    self.btnLeaseEndlease.disabled = true
                    self.btnBuySell.text = g_i18n:getText("ui_btn_sell")
                end
            else
                self.btnBuySell.disabled = true
                self.btnLeaseEndlease.disabled = true
            end
        end
    else 
        self.btnBuySell.disabled = true
        self.btnLeaseEndlease.disabled = true
    end

	local showButtons = not g_currentMission.missionDynamicInfo.isMultiplayer or g_currentMission.inGameMenu.playerFarmId ~= FarmManager.SPECTATOR_FARM_ID

    self.btnBuySell.disabled = self.btnBuySell.disabled or not showButtons  
    self.btnLeaseEndlease.disabled = self.btnLeaseEndlease.disabled or not showButtons  
    self:setMenuButtonInfoDirty()
end

function InGameMenuFieldLease:buySell()
    local farmland = g_farmlandManager:getFarmlandById(self.currentFarmland.farmlandId)
    local farmlandOwner = g_farmlandManager:getFarmlandOwner(self.currentFarmland.farmlandId)

    if farmlandOwner == FarmlandManager.NO_OWNER_FARM_ID then 
		local money = g_farmManager:getFarmById(g_currentMission:getFarmId()):getBalance()
        if money >= self.currentFarmland.price then
            self.currentMode = self.MODE_BUY
            local text = string.format(g_i18n:getText("ui_fieldLease_buyQuestion"), 
            self.currentFarmland.farmlandId, 
            g_i18n:formatMoney(self.currentFarmland.price))
            g_gui:showYesNoDialog({text = text, title = g_i18n:getText("ui_fieldLease_buyQuestionHeader"), callback = self.onConfirm, target = self})
        else
            g_gui:showInfoDialog({
				title = g_i18n:getText(InGameMenuMapFrame.L10N_SYMBOL.DIALOG_BUY_FARMLAND_TITLE),
				text = g_i18n:getText(InGameMenuMapFrame.L10N_SYMBOL.DIALOG_BUY_FARMLAND_NOT_ENOUGH_MONEY)
			})
        end
    elseif farmlandOwner == g_currentMission:getFarmId() then    
        self.currentMode = self.MODE_SELL
        local text = string.format(g_i18n:getText("ui_fieldLease_sellQuestion"), 
        self.currentFarmland.farmlandId, 
        g_i18n:formatMoney(self.currentFarmland.price))
        g_gui:showYesNoDialog({text = text, title = g_i18n:getText("ui_fieldLease_sellQuestionHeader"), callback = self.onConfirm, target = self})
    end   
end

function InGameMenuFieldLease:leaseEndlease()
    local farmland = g_farmlandManager:getFarmlandById(self.currentFarmland.farmlandId)
    local farmlandOwner = g_farmlandManager:getFarmlandOwner(self.currentFarmland.farmlandId)

    if farmlandOwner == FarmlandManager.NO_OWNER_FARM_ID then        
        self.currentMode = self.MODE_LEASE
        local text = string.format(g_i18n:getText("ui_fieldLease_leaseQuestion"), 
        self.currentFarmland.farmlandId, 
        g_i18n:formatMoney(self.currentFarmland.leasePriceStart),
        g_i18n:formatMoney(self.currentFarmland.leasePrice))
        g_gui:showYesNoDialog({text = text, title = g_i18n:getText("ui_fieldLease_leaseQuestionHeader"), callback = self.onConfirm, target = self})
    elseif farmlandOwner == g_currentMission:getFarmId() then    
        self.currentMode = self.MODE_LEASESTOP
        local text = string.format(g_i18n:getText("ui_fieldLease_leaseEndQuestion"), 
        self.currentFarmland.farmlandId)
        g_gui:showYesNoDialog({text = text, title = g_i18n:getText("ui_fieldLease_leaseEndQuestionHeader"), callback = self.onConfirm, target = self})
    end    
end

function InGameMenuFieldLease:onConfirm(confirm)
	if confirm then
        if self.currentMode == self.MODE_BUY then
            g_client:getServerConnection():sendEvent(FarmlandLeaseStateEvent.new(self.currentFarmland.farmlandId, g_currentMission:getFarmId(), self.currentFarmland.price, false))       
        elseif self.currentMode == self.MODE_SELL then
            g_client:getServerConnection():sendEvent(FarmlandLeaseStateEvent.new(self.currentFarmland.farmlandId, FarmlandManager.NO_OWNER_FARM_ID, self.currentFarmland.price, false))       
        elseif self.currentMode == self.MODE_LEASE then
            g_client:getServerConnection():sendEvent(FarmlandLeaseStateEvent.new(self.currentFarmland.farmlandId, g_currentMission:getFarmId(), self.currentFarmland.leasePriceStart, true))       
        elseif self.currentMode == self.MODE_LEASESTOP then
            g_client:getServerConnection():sendEvent(FarmlandLeaseStateEvent.new(self.currentFarmland.farmlandId, FarmlandManager.NO_OWNER_FARM_ID, 0, false))       
        end       

        self:updateContent()
    end
end

function InGameMenuFieldLease:farmlandManagerSaveToXmlFile(xmlFilename)
	local xmlFile = loadXMLFile("farmlandsXML", xmlFilename, "farmlands")
    if xmlFile ~= nil then
        local index = 0
		for farmlandId, farmId in pairs(g_farmlandManager.farmlandMapping) do
			if g_farmlandManager.farmlands[farmlandId] ~= nil then
				local farmlandKey = string.format("farmlands.leasedFarmlands(%d)", index)
				setXMLInt(xmlFile, farmlandKey.."#id", farmlandId)
				setXMLBool(xmlFile, farmlandKey.."#isLeased", Utils.getNoNil(g_farmlandManager.farmlands[farmlandId].isLeased, false))
				index = index + 1
			end
        end
        saveXMLFile(xmlFile)
        delete(xmlFile)
    end
end

function InGameMenuFieldLease:farmlandManagerLoadFromXMLFile(xmlFilename)
	if xmlFilename == nil then
        return false
    end
    local xmlFile = loadXMLFile("farmlandXML", xmlFilename)
    if xmlFile == 0 then
        return false
    end
    local farmlandCounter = 0
    while true do
        local key = string.format("farmlands.leasedFarmlands(%d)", farmlandCounter)
        local farmlandId = getXMLInt(xmlFile, key .. "#id")
        if farmlandId == nil then
            break
        end
        g_farmlandManager.farmlands[farmlandId].isLeased = getXMLBool(xmlFile, key .. "#isLeased")
        farmlandCounter = farmlandCounter + 1
    end
    delete(xmlFile)
end

FarmlandManager.saveToXMLFile = KrSoftwareUtils.appendedFunction2(FarmlandManager.saveToXMLFile, InGameMenuFieldLease.farmlandManagerSaveToXmlFile)
FarmlandManager.loadFromXMLFile = KrSoftwareUtils.appendedFunction2(FarmlandManager.loadFromXMLFile, InGameMenuFieldLease.farmlandManagerLoadFromXMLFile)