--
-- FS22 - FieldCalculator - InGameMenuFieldCalculator
--
-- @Interface: 1.2.0.0 b14651
-- @Author: KR-Softwares
-- @Date: 23.01.2022
-- @Version: 1.0.0.0
--
-- @Support: kr-softwares.com
--
-- Changelog:
-- 	v1.0.0.0 (23.01.2022):
--      - Convert from from FS19 to FS22
--

InGameMenuFieldCalculator = {}
InGameMenuFieldCalculator._mt = Class(InGameMenuFieldCalculator, TabbedMenuFrameElement)

InGameMenuFieldCalculator.CONTROLS = {
	MAIN_BOX = "mainBox",
	TABLE_SLIDER = "tableSlider",
	HEADER_BOX = "tableHeaderBox",
	TABLE = "fieldCalculatorTable",
	TABLE_TEMPLATE = "fieldCalculatorRowTemplate",
}

function InGameMenuFieldCalculator.new(i18n, messageCenter)
	local self = InGameMenuFieldCalculator:superClass().new(nil, InGameMenuFieldCalculator._mt)

    self.name = "InGameMenuFieldCalculator"
    self.i18n = i18n
    self.messageCenter = messageCenter
    
	self.dataBindings = {}

    self:registerControls(InGameMenuFieldCalculator.CONTROLS)

    self.backButtonInfo = {
		inputAction = InputAction.MENU_BACK
	}
	self.btnShowSeedUi = {
		text = self.i18n:getText("ui_seed_btn"),
		inputAction = InputAction.MENU_ACTIVATE,
        disabled = true,
		callback = function ()
			self:showSeedUi()
		end
	}    
  
    self:setMenuButtonInfo({
        self.backButtonInfo,
        self.btnShowSeedUi
    })

    return self
end

function InGameMenuFieldCalculator:delete()
	InGameMenuFieldCalculator:superClass().delete(self)
end

function InGameMenuFieldCalculator:copyAttributes(src)
    InGameMenuFieldCalculator:superClass().copyAttributes(self, src)
    self.i18n = src.i18n
end

function InGameMenuFieldCalculator:onGuiSetupFinished()
	InGameMenuFieldCalculator:superClass().onGuiSetupFinished(self)
	self.fieldCalculatorTable:setDataSource(self)
	self.fieldCalculatorTable:setDelegate(self)
end

function InGameMenuFieldCalculator:initialize()
end

function InGameMenuFieldCalculator:onFrameOpen()
	InGameMenuFieldCalculator:superClass().onFrameOpen(self)   
    self:updateContent()
	FocusManager:setFocus(self.fieldCalculatorTable)
end

function InGameMenuFieldCalculator:onFrameClose()
	InGameMenuFieldCalculator:superClass().onFrameClose(self)   
end

function InGameMenuFieldCalculator:updateContent()  
    local function addField(field)
        local field = {
            fieldId = field.fieldId,
            farmId = field.farmland and g_farmlandManager:getFarmlandOwner(field.farmland.id) or g_currentMission.player.farmId,
            fieldArea = field.fieldArea,
            litersPerSecondLime = g_sprayTypeManager:getSprayTypeByName("LIME").litersPerSecond,
            litersPerSecondFertilizer = g_sprayTypeManager:getSprayTypeByName("FERTILIZER").litersPerSecond,
            litersPerSecondLiquideFertilizer = g_sprayTypeManager:getSprayTypeByName("LIQUIDFERTILIZER").litersPerSecond,
            litersPerSecondHerbicide = g_sprayTypeManager:getSprayTypeByName("HERBICIDE").litersPerSecond,
            litersPerSecondManure = g_sprayTypeManager:getSprayTypeByName("MANURE").litersPerSecond,
            litersPerSecondLiquidManure = g_sprayTypeManager:getSprayTypeByName("LIQUIDMANURE").litersPerSecond,
            litersPerSecondDigestate = g_sprayTypeManager:getSprayTypeByName("DIGESTATE").litersPerSecond
        }
        table.insert(self.fieldData, field)
    end

    self.fieldData = {}    
    for _, field in pairs(g_fieldManager.fields) do        
        addField(field)
    end

    --- Adds custom fields, if courseplay is used.
    if g_fieldManager.cpCustomFields then
        for _, field in pairs(g_fieldManager.cpCustomFields) do        
            addField(field)
        end
    end
        
	self:sortList()
	self.fieldCalculatorTable:reloadData()    
end

function InGameMenuFieldCalculator:sortList()
	local tmpList = {
		{
			title = g_i18n:getText("ui_listHeader_free"),
            farmlandId = FarmlandManager.NO_OWNER_FARM_ID,
			fields = {}
		}
	}

	for _, fieldData in ipairs(self.fieldData) do
        local add = false
        for _, data in pairs(tmpList) do
            if data.farmlandId == fieldData.farmId then  
                table.insert(data.fields, fieldData)
                add = true
                break
            end            
        end
        
        if not add then
            local newSection = {
                title = string.format(g_i18n:getText("ui_listHeader_farm"), g_farmManager:getFarmById(fieldData.farmId).name),
                farmlandId = fieldData.farmId,
                fields = {}
            }      
            table.insert(newSection.fields, fieldData)
            table.insert(tmpList, newSection)     
        end
    end

    local currentFarmId = -1
    local farm = g_farmManager:getFarmByUserId(g_currentMission.playerUserId)
    if farm ~= nil then
        currentFarmId = farm.farmId
    end
    
    self.sectionFields = {}

    for _,e in pairs(tmpList) do
        if e.farmlandId == currentFarmId then
            table.insert(self.sectionFields, e)
            break
        end
    end

    for _,e in pairs(tmpList) do
        if e.farmlandId == FarmlandManager.NO_OWNER_FARM_ID then
            table.insert(self.sectionFields, e)
            break
        end
    end

    for _,e in pairs(tmpList) do
        if e.farmlandId ~= FarmlandManager.NO_OWNER_FARM_ID and e.farmlandId ~= currentFarmId then
            table.insert(self.sectionFields, e)
        end
    end
end

function InGameMenuFieldCalculator:getNumberOfSections()
	return #self.sectionFields
end

function InGameMenuFieldCalculator:getNumberOfItemsInSection(list, section)
	return #self.sectionFields[section].fields
end

function InGameMenuFieldCalculator:getTitleForSectionHeader(list, section)
	return self.sectionFields[section].title
end

function InGameMenuFieldCalculator:populateCellForItemInSection(list, section, index, cell)
	local fieldData = self.sectionFields[section].fields[index]    
	cell:getAttribute("field"):setText(fieldData.fieldId)
	cell:getAttribute("size"):setText(g_i18n:formatNumber(fieldData.fieldArea, 2))
	cell:getAttribute("lime"):setText(g_i18n:formatNumber(fieldData.litersPerSecondLime * fieldData.fieldArea * 36000, 2))
	cell:getAttribute("fertilizer"):setText(g_i18n:formatNumber(fieldData.litersPerSecondFertilizer * fieldData.fieldArea * 36000, 2))
	cell:getAttribute("liquideFertilizer"):setText(g_i18n:formatNumber(fieldData.litersPerSecondLiquideFertilizer * fieldData.fieldArea * 36000, 2))
	cell:getAttribute("herbicide"):setText(g_i18n:formatNumber(fieldData.litersPerSecondHerbicide * fieldData.fieldArea * 36000, 2))
	cell:getAttribute("manure"):setText(g_i18n:formatNumber(fieldData.litersPerSecondManure * fieldData.fieldArea * 36000, 2))
	cell:getAttribute("slurry"):setText(g_i18n:formatNumber(fieldData.litersPerSecondLiquidManure * fieldData.fieldArea * 36000, 2))
	cell:getAttribute("digestate"):setText(g_i18n:formatNumber(fieldData.litersPerSecondDigestate * fieldData.fieldArea * 36000, 2))    
end

function InGameMenuFieldCalculator:onListSelectionChanged(list, section, index)
	local sectionFields = self.sectionFields[section]    
	if sectionFields ~= nil and sectionFields.fields[index] ~= nil then        
        self.currentField = sectionFields.fields[index]
        self.btnShowSeedUi.disabled = false
        self:playSample(GuiSoundPlayer.SOUND_SAMPLES.HOVER)
    else 
        self.btnShowSeedUi.disabled = true
    end
    self:setMenuButtonInfoDirty()
end

function InGameMenuFieldCalculator:showSeedUi()
    local dialog = g_gui:showDialog("SeedFrame")
    if dialog ~= nil then
        dialog.target:setFieldData(self.currentField)
    end
end