-- Name: EnhancedAnimalSystem
-- Author: Chissel

local modName = g_currentModName
local modDirectory = g_currentModDirectory

EnhancedAnimalSystem = {}
EnhancedAnimalSystem.Settings = g_easUtils.loadXml()
EnhancedAnimalSystem.SellOvercrowding = true
EnhancedAnimalSystem.ModName = modName..".EnhancedAnimalSystem"
EnhancedAnimalSystem.NUM_BITS_eas_numOvercrowdingHours = 5

-- ################################# Lifecycle methods ####################################

function EnhancedAnimalSystem.prerequisitesPresent(specializations)
	g_easUtils:logText("prerequisitesPresent")
    return SpecializationUtil.hasSpecialization(PlaceableHusbandryAnimals, specializations)
end

function EnhancedAnimalSystem.registerEventListeners(placeableType)
	g_easUtils:logText("Register event listeners for", placeableType.name)
	SpecializationUtil.registerEventListener(placeableType, "onLoad", EnhancedAnimalSystem)
    SpecializationUtil.registerEventListener(placeableType, "onHourChanged", EnhancedAnimalSystem)
    SpecializationUtil.registerEventListener(placeableType, "onReadStream", EnhancedAnimalSystem)
    SpecializationUtil.registerEventListener(placeableType, "onWriteStream", EnhancedAnimalSystem)
	SpecializationUtil.registerEventListener(placeableType, "onReadUpdateStream", EnhancedAnimalSystem)
	SpecializationUtil.registerEventListener(placeableType, "onWriteUpdateStream", EnhancedAnimalSystem)
end

function EnhancedAnimalSystem.registerOverwrittenFunctions(placeableType)
    SpecializationUtil.registerOverwrittenFunction(placeableType, "updateOutput", EnhancedAnimalSystem.updateOutput)
end

-- ################################# onLoad ####################################

function EnhancedAnimalSystem:onLoad(savegame)
    local spec = self.spec_husbandryAnimals
    spec.allowOvercrowding = false
    spec.automaticInsemination = true
    self.eas_numOvercrowdingHours = 0
    self.dirtyFlagOvercrowding = self:getNextDirtyFlag()
end

-- ################################# OnHusbandryAnimalsUpdate ####################################

function EnhancedAnimalSystem.numOfFreeAnimalSlots(spec)
    local totalNumAnimals = spec:getNumOfAnimals()
    for clusterToAdd, _ in pairs(spec.clusterSystem.clustersToAdd) do
        totalNumAnimals = totalNumAnimals + clusterToAdd.numAnimals
    end

    if spec.allowOvercrowding then
        local maxNumAnimals = spec.maxNumAnimals + math.floor(EnhancedAnimalSystem.Settings.MaxOvercrowdingPercentage * spec.maxNumAnimals)
        return math.max(maxNumAnimals - totalNumAnimals, 0)
    else
        return math.max(spec.maxNumAnimals - totalNumAnimals, 0)
    end
end

function EnhancedAnimalSystem:calculateReproduction(spec, cluster, numNewAnimals, freeSpace, isServer)
    local totalOffspring = EnhancedAnimalSystem.calculateTotalOffspring(cluster.numAnimals, spec.animalTypeIndex)
    local farmId = spec:getOwnerFarmId()

    if totalOffspring == 0 then
        g_easUtils:logText("No offspring")
    elseif freeSpace >= totalOffspring then
        g_easUtils:logText("Set cluster to totalOffspring "..totalOffspring)
        freeSpace = freeSpace - totalOffspring
        EnhancedAnimalSystem.addOffspring(spec, cluster, totalOffspring)
    elseif freeSpace == 0 then
        g_easUtils:logText("Overcrowing of: "..totalOffspring)
        local subType = g_currentMission.animalSystem:getSubTypeByIndex(cluster:getSubTypeIndex())
        local sellPrice = subType.sellPrice:get(0)

        EnhancedAnimalSystem.sellAnimalsIfAllowed(totalOffspring, farmId, sellPrice)
    else
        local overcrowding = totalOffspring - freeSpace
        g_easUtils:logText("Overcrowing of: "..overcrowding)

        EnhancedAnimalSystem.addOffspring(spec, cluster, freeSpace)

        local subType = g_currentMission.animalSystem:getSubTypeByIndex(cluster:getSubTypeIndex())
        local sellPrice = subType.sellPrice:get(0)
        EnhancedAnimalSystem.sellAnimalsIfAllowed(overcrowding, farmId, sellPrice, isServer)
        freeSpace = 0
    end

    local subType = g_currentMission.animalSystem:getSubTypeByIndex(cluster.subTypeIndex)
    local animalType = g_currentMission.animalSystem:getTypeByIndex(subType.typeIndex)

    if g_farmManager:getFarmById(farmId) ~= nil then
        local stats = g_currentMission:farmStats(spec:getOwnerFarmId())
        stats:updateStats(animalType.statsBreedingName, totalOffspring)
    end
end

function EnhancedAnimalSystem.addOffspring(spec, cluster, offspring)
    local subType = g_currentMission.animalSystem:getSubTypeByIndex(cluster.subTypeIndex)
    local numMale, numFemale = EnhancedAnimalSystem.calculateMaleFemalePart(subType.malePart, offspring)
    if numMale > 0 then
        local maleSubTypeIndex = g_currentMission.animalSystem:getSubTypeIndexByName(subType.malePart)
        local maleCluster = EnhancedAnimalSystem.createCluster(spec, numMale, maleSubTypeIndex)
        spec.clusterSystem:addPendingAddCluster(maleCluster)
    end

    local femaleCluster = EnhancedAnimalSystem.createCluster(spec, numFemale, cluster.subTypeIndex)
    spec.clusterSystem:addPendingAddCluster(femaleCluster)
end

function EnhancedAnimalSystem.calculateMaleFemalePart(malePart, offspring)
    if malePart == nil then
        return 0, offspring
    else
        local maleParts = 0
        local femaleParts = 0
        for i=1, offspring do
            if math.random(2) == 1 then
                maleParts = maleParts + 1
            else
                femaleParts = femaleParts + 1
            end
        end

        g_easUtils:logText("Male parts: "..maleParts.." Female parts: "..femaleParts)

        return maleParts, femaleParts
    end
end

function EnhancedAnimalSystem.calculateTotalOffspring(numAnimals, animalTypeIndex)
    local totalOffspring = 0

    for i=1, numAnimals do
        local multiplier = EnhancedAnimalSystem.reproductionMultiplierForAnimalType(animalTypeIndex)
        totalOffspring = totalOffspring + multiplier
    end

    return totalOffspring
end

function EnhancedAnimalSystem.sellAnimalsIfAllowed(numAnimals, farmId, sellPrice, isServer)
    if EnhancedAnimalSystem.SellOvercrowding then
        local pricePerAnimal = sellPrice * EnhancedAnimalSystem.Settings.SellingPricePercentage
        local amount = pricePerAnimal * numAnimals

        if isServer then
            g_currentMission:addMoneyChange(amount, farmId, MoneyType.SOLD_ANIMALS, true)

            g_easUtils:logText("Add "..amount.." money to farm with farmId: "..farmId)
        else
            g_client:getServerConnection():sendEvent(MoneyChangeEvent.new(amount, MoneyType.SOLD_ANIMALS, farmId))
            g_easUtils:logText("Send event for "..amount.." money for farm: "..farmId)
        end

        local farm = g_farmManager:getFarmById(farmId)

        if farm ~= nil then
            farm:changeBalance(amount, MoneyType.SOLD_ANIMALS)
            g_easUtils:logText("Sold "..numAnimals.." for price of "..pricePerAnimal)
        end
    end
end

function EnhancedAnimalSystem.createCluster(spec, numAnimals, subTypeIndex)
    local cluster = g_currentMission.animalSystem:createClusterFromSubTypeIndex(subTypeIndex)
    cluster.numAnimals = numAnimals
    cluster.age = 0
    cluster.health = 100
    return cluster
end

function EnhancedAnimalSystem.reproductionMultiplierForAnimalType(animalTypeIndex)
    if animalTypeIndex == AnimalType.PIG then
        return EnhancedAnimalSystem.Settings.PigletProbabilities[math.random(#EnhancedAnimalSystem.Settings.PigletProbabilities)]
    elseif animalTypeIndex == AnimalType.COW then
        return EnhancedAnimalSystem.Settings.CalfProbabilities[math.random(#EnhancedAnimalSystem.Settings.CalfProbabilities)]
    elseif animalTypeIndex == AnimalType.SHEEP then
        return EnhancedAnimalSystem.Settings.LambProbabilities[math.random(#EnhancedAnimalSystem.Settings.LambProbabilities)]
    elseif animalTypeIndex == AnimalType.HORSE then
        return EnhancedAnimalSystem.Settings.FoalProbabilities[math.random(#EnhancedAnimalSystem.Settings.FoalProbabilities)]
    elseif animalTypeIndex == AnimalType.CHICKEN then
        return EnhancedAnimalSystem.Settings.ChickenProbabilities[math.random(#EnhancedAnimalSystem.Settings.ChickenProbabilities)]
    end

    return 1
end

-- ################################# OnPeriodChanged ####################################

function EnhancedAnimalSystem.onPeriodChanged(self, superFunc)
    if self.isServer then
        local spec = self.spec_husbandryAnimals
        local clusters = spec.clusterSystem:getClusters()
        for _, cluster in ipairs(clusters) do
            EnhancedAnimalSystem.increaseMonthSinceLastBirthIfNeeded(cluster)
            EnhancedAnimalSystem.updateInseminatedIfNeeded(spec, cluster, clusters)

            cluster:onPeriodChanged()
            local numNewAnimals = cluster:updateReproduction()
            if numNewAnimals > 0 then
                EnhancedAnimalSystem:calculateReproduction(spec, cluster, numNewAnimals, EnhancedAnimalSystem.numOfFreeAnimalSlots(spec), self.isServer)
            end
        end

        EnhancedAnimalSystem:calculateOldAnimalsDies(self)
        self:raiseActive()
    end
end

PlaceableHusbandryAnimals.onPeriodChanged = Utils.overwrittenFunction(PlaceableHusbandryAnimals.onPeriodChanged, EnhancedAnimalSystem.onPeriodChanged)

function EnhancedAnimalSystem.updateInseminatedIfNeeded(spec, cluster, clusters)
    if not cluster:getCanReproduce() or cluster.isInseminated then
        return
    end

    if EnhancedAnimalSystem.hasMalePartInHusbandry(cluster, clusters) then
        cluster.isInseminated = true
        cluster:setDirty()
    elseif spec.automaticInsemination then
        cluster.isInseminated = true
        cluster:setDirty()

        EnhancedAnimalSystem.collectInseminationCost(spec, cluster:getNumAnimals())
    end
end

function EnhancedAnimalSystem.hasMalePartInHusbandry(cluster, clusters)
	local clusterSubType = g_currentMission.animalSystem:getSubTypeByIndex(cluster:getSubTypeIndex())
    if clusterSubType.malePart ~= nil then
        local malePartSubType = g_currentMission.animalSystem:getSubTypeByName(clusterSubType.malePart)

        for _, subCluster in ipairs(clusters) do
            if subCluster:getSubTypeIndex() == malePartSubType.subTypeIndex and subCluster:getAge() >= malePartSubType.reproductionMinAgeMonth then
                return true
            end
        end
    end

    return false
end

function EnhancedAnimalSystem.collectInseminationCost(spec, numAnimals)
    local farmId = spec:getOwnerFarmId()
    local farm = g_farmManager:getFarmById(farmId)

    if farm == nil then
		Logging.warning("Husbandry has no farm")
        return
    end

    local price = EnhancedAnimalSystem.priceForInseminated(spec.animalTypeIndex)
    local totalCost = price * numAnimals
    local moneyType = MoneyType.ANIMAL_UPKEEP

    if g_currentMission:getIsServer() then
        farm:changeBalance(-totalCost, moneyType)
        g_currentMission:addMoneyChange(-totalCost, farmId, moneyType, true)
    else
        g_client:getServerConnection():sendEvent(EAS_RemoveMoneyEvent.new(farmId, totalCost, moneyType))
    end
end

function EnhancedAnimalSystem.priceForInseminated(animalTypeIndex)
    if animalTypeIndex == AnimalType.PIG then
        return EnhancedAnimalSystem.Settings.PigInseminationCost
    elseif animalTypeIndex == AnimalType.COW then
        return EnhancedAnimalSystem.Settings.CowInseminationCost
    elseif animalTypeIndex == AnimalType.HORSE then
        return EnhancedAnimalSystem.Settings.HorseInseminationCost
    elseif animalTypeIndex == AnimalType.SHEEP then
        return EnhancedAnimalSystem.Settings.SheepInseminationCost
    elseif animalTypeIndex == AnimalType.CHICKEN then
        return EnhancedAnimalSystem.Settings.ChickenInseminationCost
    end

    return 15
end

function EnhancedAnimalSystem.increaseMonthSinceLastBirthIfNeeded(cluster)
	local subType = g_currentMission.animalSystem:getSubTypeByIndex(cluster:getSubTypeIndex())
    local neededAge = subType.reproductionMinAgeMonth + subType.reproductionDurationMonth
    if subType.supportsReproduction and cluster.age >= neededAge and cluster.hadABirth then
        cluster.monthsSinceLastBirth = cluster.monthsSinceLastBirth + 1
    end

    cluster:setDirty()
end

-- ################################# Animals die without health ####################################

function EnhancedAnimalSystem:calculateNumOfDiedAnimalsByHealth(spec)
    if g_easSettingsMenuExtension.animalsCanDie == false then
        return
    end

    g_easUtils:logText("calculateNumOfDiedAnimals")
    local husbandrySpec = spec.spec_husbandryAnimals
    local clusters = husbandrySpec.clusterSystem.clusters

    for _, cluster in ipairs(clusters) do
        local healthValue = cluster.health
        if healthValue == 0 then
            local numOfDeadAnimals = EnhancedAnimalSystem.numOfDeadAnimals(cluster.numAnimals, EnhancedAnimalSystem.Settings.AnimalDiedPercentageWhenHealthIsEmpty)
            cluster.numAnimals = cluster.numAnimals - numOfDeadAnimals
            g_easUtils:logText(numOfDeadAnimals.." died by health")

            if cluster.numAnimals <= 0 then
                g_easUtils:removeItemFromTable(cluster, husbandrySpec.clusterSystem.clusters)
            end
        end
    end

    spec:updateVisualAnimals()
end

-- ################################# Overcrowding ####################################

function EnhancedAnimalSystem:onHourChanged(currentHour)
    if self.isServer then
        EnhancedAnimalSystem:calculateNumOfDiedAnimalsByHealth(self)

        g_easUtils:logText("onHourChanged")
    end
end

function EnhancedAnimalSystem.reduceHealthOnOvercrowding(husbandrySpec, totalNumAnimals)
    local clusters = husbandrySpec.clusterSystem.clusters
    local overcrowdingInPercentage = (totalNumAnimals - husbandrySpec.maxNumAnimals) / 100
    local healthDeduction = math.floor(EnhancedAnimalSystem.Settings.MaxHealthOvercrowdingDeduction * overcrowdingInPercentage)

    for _, cluster in ipairs(clusters) do
        cluster.eas_isOvercrowing = true
        local healthValue = cluster.health
        if healthValue >= healthDeduction then
            cluster.health = healthValue - healthDeduction
        else
            cluster.health = 0
        end

        cluster:setDirty()
    end

    husbandrySpec:raiseActive()
end

function EnhancedAnimalSystem:updateOutput(superFunc, foodFactor, productionFactor, globalProductionFactor)
    if self.isServer then
        local husbandrySpec = self.spec_husbandryAnimals
        local totalNumAnimals = husbandrySpec:getNumOfAnimals()

        if husbandrySpec.maxNumAnimals < totalNumAnimals then
            if self.eas_numOvercrowdingHours > EnhancedAnimalSystem.Settings.NumHoursOfOvercrowdingBeforReduceHealth then
                EnhancedAnimalSystem.reduceHealthOnOvercrowding(husbandrySpec, totalNumAnimals)
            else
                self.eas_numOvercrowdingHours = self.eas_numOvercrowdingHours + 1
                self:raiseDirtyFlags(self.dirtyFlagOvercrowding)
            end
        else
            self.eas_numOvercrowdingHours = 0
            self:raiseDirtyFlags(self.dirtyFlagOvercrowding)
        end
    end

    superFunc(self, foodFactor, productionFactor, globalProductionFactor)
end

function EnhancedAnimalSystem.changeHealth(self, superFunc, delta)
    if self.eas_isOvercrowing == nil then
        superFunc(self, delta)
    end

    self.eas_isOvercrowing = nil
end

AnimalCluster.changeHealth = Utils.overwrittenFunction(AnimalCluster.changeHealth, EnhancedAnimalSystem.changeHealth)

-- ################################# Old animals die ####################################

function EnhancedAnimalSystem:calculateOldAnimalsDies(spec)
    local husbandrySpec = spec.spec_husbandryAnimals
    local clusters = husbandrySpec.clusterSystem.clusters

    for _, cluster in ipairs(clusters) do
        if cluster.age > EnhancedAnimalSystem.maxAgeForAnimal(husbandrySpec.animalTypeIndex) then
            g_easUtils:logText("Cluster is too old. Maybe some animals will die.")

            local numOfDeadAnimals = EnhancedAnimalSystem.numOfDeadAnimals(cluster.numAnimals, EnhancedAnimalSystem.Settings.AnimalDiedPercentageOnMaxAge)
            cluster.numAnimals = cluster.numAnimals - numOfDeadAnimals

            if cluster.numAnimals <= 0 then
                g_easUtils:removeItemFromTable(cluster, husbandrySpec.clusterSystem.clusters)
            end
        end
    end

    spec:updateVisualAnimals()
end

function EnhancedAnimalSystem.numOfDeadAnimals(numAnimals, percentage)
    local deadAnimals = 0

    for i=1, numAnimals do
        local randomNumber = math.random(100)
        if randomNumber <= percentage then
            deadAnimals = deadAnimals + 1
        end
    end

    g_easUtils:logText(deadAnimals.." animals have died")

    return deadAnimals
end

function EnhancedAnimalSystem.maxAgeForAnimal(animalTypeIndex)
    if animalTypeIndex == AnimalType.PIG then
        return EnhancedAnimalSystem.Settings.PigMaxAge
    elseif animalTypeIndex == AnimalType.COW then
        return EnhancedAnimalSystem.Settings.CowMaxAge
    elseif animalTypeIndex == AnimalType.HORSE then
        return EnhancedAnimalSystem.Settings.HorseMaxAge
    elseif animalTypeIndex == AnimalType.SHEEP then
        return EnhancedAnimalSystem.Settings.SheepMaxAge
    elseif animalTypeIndex == AnimalType.CHICKEN then
        return EnhancedAnimalSystem.Settings.ChickenMaxAge
    end

    return 120
end

-- ################################# Save ###################################

function EnhancedAnimalSystem.registerSavegameXMLPaths(schema, basePath)
    schema:register(XMLValueType.BOOL, basePath.."#allowOvercrowding", "Allow overcrowding for husbandry", false)
    schema:register(XMLValueType.BOOL, basePath.."#automaticInsemination", "Allow automatic insemination for husbandry", true)
    schema:register(XMLValueType.INT, basePath.."#eas_numOvercrowdingHours", "Number of overcrowding hours", 0)
    g_easUtils:logText("registerSavegameXMLPaths")
end

function EnhancedAnimalSystem:saveToXMLFile(xmlFile, key, usedModNames)
    xmlFile:setBool(key.."#allowOvercrowding", self.spec_husbandryAnimals.allowOvercrowding)
    xmlFile:setBool(key.."#automaticInsemination", self.spec_husbandryAnimals.automaticInsemination)
	xmlFile:setInt(key .. "#eas_numOvercrowdingHours", self.eas_numOvercrowdingHours or 0)

    g_easUtils:logText("saveToXMLFile")
end

function EnhancedAnimalSystem:loadFromXMLFile(xmlFile, key)
    g_easUtils:logText("loadFromXMLFile for husbandry id: "..self.id)
    local allowOvercrowding = xmlFile:getBool(key.."#allowOvercrowding")
    if allowOvercrowding ~= nil then
        self.spec_husbandryAnimals.allowOvercrowding = allowOvercrowding
    end

    local automaticInsemination = xmlFile:getBool(key.."#automaticInsemination")
    if automaticInsemination ~= nil then
        self.spec_husbandryAnimals.automaticInsemination = automaticInsemination
    end

	self.eas_numOvercrowdingHours = xmlFile:getInt(key .. "#eas_numOvercrowdingHours", self.eas_numOvercrowdingHours or 0)
end

function EnhancedAnimalSystem:onReadStream(streamId, connection)
    self.spec_husbandryAnimals.allowOvercrowding = streamReadBool(streamId)
    self.spec_husbandryAnimals.automaticInsemination = streamReadBool(streamId)
    self.eas_numOvercrowdingHours = streamReadUIntN(streamId, EnhancedAnimalSystem.NUM_BITS_eas_numOvercrowdingHours)
end

function EnhancedAnimalSystem:onWriteStream(streamId, connection)
    streamWriteBool(streamId, self.spec_husbandryAnimals.allowOvercrowding)
    streamWriteBool(streamId, self.spec_husbandryAnimals.automaticInsemination)
    streamWriteUIntN(streamId, self.eas_numOvercrowdingHours or 0, EnhancedAnimalSystem.NUM_BITS_eas_numOvercrowdingHours)
end

function EnhancedAnimalSystem:onReadUpdateStream(streamId, connection)
    self.spec_husbandryAnimals.allowOvercrowding = streamReadBool(streamId)
    self.spec_husbandryAnimals.automaticInsemination = streamReadBool(streamId)
	self.eas_numOvercrowdingHours = streamReadUIntN(streamId, EnhancedAnimalSystem.NUM_BITS_eas_numOvercrowdingHours)
end

function EnhancedAnimalSystem:onWriteUpdateStream(streamId, connection)
    streamWriteBool(streamId, self.spec_husbandryAnimals.allowOvercrowding)
    streamWriteBool(streamId, self.spec_husbandryAnimals.automaticInsemination)
    streamWriteUIntN(streamId, self.eas_numOvercrowdingHours or 0, EnhancedAnimalSystem.NUM_BITS_eas_numOvercrowdingHours)
end