-- Name: EAS_LoadAnimals
-- Author: Chissel

EAS_LoadAnimals = {}

local modName = g_currentModName
local modDirectory = g_currentModDirectory

local supportedAnimalTypes = {["COW"] = true, ["PIG"] = true, ["SHEEP"] = true, ["HORSE"] = true, ["CHICKEN"] = true}

function EAS_LoadAnimals.loadAnimals(self, superFunc, xmlFile, baseDirectory)
    if xmlFile:hasProperty("animals#doNotOverrideAnimals") and xmlFile:getBool("animals#doNotOverrideAnimals", false) then
        self.doNotOverrideAnimals = true
        local mapAnimalsReturnValue = superFunc(self, xmlFile, baseDirectory)
        return mapAnimalsReturnValue
    end

    local path = EAS_LoadAnimals.getAnimalsXMLPath()
    local animalsXML = XMLFile.load("animals", path)

    local mapEnvironment = self.customEnvironment
    self.customEnvironment = modName

    local returnValue = superFunc(self, animalsXML, modDirectory)

    local pathToRemove = {}

    xmlFile:iterate("animals.animal", function (_, key)
        local typeName = xmlFile:getString(key .. "#type")

        if supportedAnimalTypes[typeName] then
            table.insert(pathToRemove, key)
        end
    end)

    for _, animalPath in EAS_Utils:rpairs(pathToRemove) do
        xmlFile:removeProperty(animalPath)
    end

    self.customEnvironment = mapEnvironment
    superFunc(self, xmlFile, baseDirectory)

    return returnValue
end

function EAS_LoadAnimals.getAnimalsXMLPath()
    local hofBergmannMap = g_modManager:getModByName("FS22_HofBergmann")
    if hofBergmannMap ~= nil and g_modIsLoaded["FS22_HofBergmann"] then
        return Utils.getFilename("xml/hof_bergmann_animals.xml", modDirectory)
    else
        return Utils.getFilename("xml/eas_animals.xml", modDirectory)
    end
end

AnimalSystem.loadAnimals = Utils.overwrittenFunction(AnimalSystem.loadAnimals, EAS_LoadAnimals.loadAnimals)

function EAS_LoadAnimals.loadVisualData(self, superFunc, animalType, xmlFile, key, baseDirectory)
	local returnValue = superFunc(self, animalType, xmlFile, key, baseDirectory)

    local textureIndexes = {}

    xmlFile:iterate(key .. ".textureIndexes", function (_, textureIndexesKey)
        xmlFile:iterate(textureIndexesKey .. ".value", function (_, valueKey)
            local textureIndex = xmlFile:getInt(valueKey)
            table.insert(textureIndexes, textureIndex)
        end)
    end)

    if #textureIndexes > 0 then
        returnValue.textureIndexes = textureIndexes
    end

    return returnValue
end

AnimalSystem.loadVisualData = Utils.overwrittenFunction(AnimalSystem.loadVisualData, EAS_LoadAnimals.loadVisualData)

function EAS_LoadAnimals.loadSubType(self, superFunc, animalType, subType, xmlFile, subTypeKey, baseDirectory)
    local value = superFunc(self, animalType, subType, xmlFile, subTypeKey, baseDirectory)

    local malePart = xmlFile:getString(subTypeKey .. "#malePart")

    if malePart ~= nil then
        subType.malePart = malePart
    end

    for _, visual in pairs(subType.visuals) do
        if visual.textureIndexes == nil then
            break
        end

        local visualAnimal = EAS_Utils:deepcopy(visual.visualAnimal)
        local variations = {}

        for _, v in pairs(visual.textureIndexes) do
            table.insert(variations, visualAnimal.variations[v])
        end

        if #variations > 0 then
            visualAnimal.variations = variations
            visual.visualAnimal = visualAnimal
        end
    end

    return value
end

AnimalSystem.loadSubType = Utils.overwrittenFunction(AnimalSystem.loadSubType, EAS_LoadAnimals.loadSubType)

function EAS_LoadAnimals.updateVisuals(self, superFunc)
    superFunc(self)

    local animalIdToCluster = self.animalIdToCluster or {}

    for animalId, cluster in pairs(animalIdToCluster) do
        local subTypeIndex = cluster:getSubTypeIndex()
        local age = cluster:getAge()
        local visualData = self.animalSystem:getVisualByAge(subTypeIndex, age)
        local variations = visualData.visualAnimal.variations

        if #variations > 1 then
            local variation = variations[math.random(1, #variations)]
            setAnimalTextureTile(self.husbandry, animalId, variation.tileUIndex, variation.tileVIndex)
        else
            local variation = variations[1]
            setAnimalTextureTile(self.husbandry, animalId, variation.tileUIndex, variation.tileVIndex)
        end

        local animalRootNode = getAnimalRootNode(self.husbandry, animalId)
		local x, y, z, w = getAnimalShaderParameter(self.husbandry, animalId, "atlasInvSizeAndOffsetUV")
		I3DUtil.setShaderParameterRec(animalRootNode, "atlasInvSizeAndOffsetUV", x, y, z, w)
    end
end

AnimalClusterHusbandry.updateVisuals = Utils.overwrittenFunction(AnimalClusterHusbandry.updateVisuals, EAS_LoadAnimals.updateVisuals)

function EAS_LoadAnimals.create(self, superFunc, xmlFilename, navigationNode, raycastDistance, collisionMask)
    if self.animalSystem.doNotOverrideAnimals then
        local returnValue = superFunc(self, xmlFilename, navigationNode, raycastDistance, collisionMask)
        return returnValue
    end

    local animalType = self.animalSystem:getTypeByName(self.animalTypeName)
    local eas_xmlFilename = xmlFilename

    if supportedAnimalTypes[animalType.name] then
        eas_xmlFilename = Utils.getFilename(animalType.configFilename, modDirectory)
    end

    local returnValue = superFunc(self, eas_xmlFilename, navigationNode, raycastDistance, collisionMask)

	return returnValue
end


AnimalClusterHusbandry.create = Utils.overwrittenFunction(AnimalClusterHusbandry.create, EAS_LoadAnimals.create)
