-- Name: EAS_AnimalClusterExtension
-- Author: Chissel

EAS_AnimalClusterExtension = {}

function EAS_AnimalClusterExtension.changeReproduction(self, superFunc, delta)
    if self.reproduction == 0 then
        EAS_AnimalClusterExtension.calculateSuccessfulInsemination(self)
    end

    if self.isInseminated then
        superFunc(self, delta)

        if self.reproduction == 100 then
            self.isInseminated = false
            self.monthsSinceLastBirth = 0
            self.hadABirth = true
        end
    end
end

function EAS_AnimalClusterExtension.calculateSuccessfulInsemination(cluster)
    local numOfSuccessfulInsemination = EAS_AnimalClusterExtension.calculateNumOfSuccessfulInsemination(cluster)
    local numOfFailedInsemination = cluster.numAnimals - numOfSuccessfulInsemination

    if numOfSuccessfulInsemination == 0 then
        cluster.isInseminated = false
    elseif numOfFailedInsemination > 0 then
        cluster.numAnimals = numOfSuccessfulInsemination
        local failedCluster = cluster:clone()
        failedCluster.isInseminated = false
        failedCluster.numAnimals = numOfFailedInsemination
        cluster.clusterSystem:addPendingAddCluster(failedCluster)
    end
end

function EAS_AnimalClusterExtension.calculateNumOfSuccessfulInsemination(cluster)
	local subType = g_currentMission.animalSystem:getSubTypeByIndex(cluster.subTypeIndex)
    local inseminationProbability = EAS_AnimalClusterExtension.inseminationProbabilityForAnimalType(subType.typeIndex)
    local probability = 100

    if cluster.hadABirth then
        local monthsSinceLastBirth = cluster.monthsSinceLastBirth

        if monthsSinceLastBirth < 1 then
            monthsSinceLastBirth = 1
        end

        if monthsSinceLastBirth <= #inseminationProbability then
            probability = tonumber(inseminationProbability[monthsSinceLastBirth])
        end
    end

    local numOfSuccessfulInsemination = 0
    for i=1, cluster.numAnimals, 1 do
        local randomNumber = math.random(100)
        if randomNumber <= probability then
            numOfSuccessfulInsemination = numOfSuccessfulInsemination + 1
        end
    end

    return numOfSuccessfulInsemination
end

function EAS_AnimalClusterExtension.inseminationProbabilityForAnimalType(animalTypeIndex)
    if animalTypeIndex == AnimalType.PIG then
        return EAS_Utils.Settings.PigInseminationProbabilities
    elseif animalTypeIndex == AnimalType.COW then
        return EAS_Utils.Settings.CowInseminationProbabilities
    elseif animalTypeIndex == AnimalType.HORSE then
        return EAS_Utils.Settings.HorseInseminationProbabilities
    elseif animalTypeIndex == AnimalType.SHEEP then
        return EAS_Utils.Settings.SheepInseminationProbabilities
    elseif animalTypeIndex == AnimalType.CHICKEN then
        return EAS_Utils.Settings.ChickenInseminationProbabilities
    end

    return {}
end

AnimalCluster.changeReproduction = Utils.overwrittenFunction(AnimalCluster.changeReproduction, EAS_AnimalClusterExtension.changeReproduction)

function EAS_AnimalClusterExtension.new(self, superFunc, customMt)
    local returnValue = superFunc(self, customMt)
    returnValue.isInseminated = false
    returnValue.monthsSinceLastBirth = 0
    returnValue.hadABirth = false
    return returnValue
end

AnimalCluster.new = Utils.overwrittenFunction(AnimalCluster.new, EAS_AnimalClusterExtension.new)

function EAS_AnimalClusterExtension.clone(self, superFunc)
    local returnValue = superFunc(self)
    returnValue.isInseminated = self.isInseminated
    returnValue.monthsSinceLastBirth = self.monthsSinceLastBirth
    returnValue.hadABirth = self.hadABirth
    return returnValue
end

AnimalCluster.clone = Utils.overwrittenFunction(AnimalCluster.clone, EAS_AnimalClusterExtension.clone)

function EAS_AnimalClusterExtension.getHash(self, superFunc)
    local returnValue = superFunc(self)
    local monthsSinceLastBirth = 1000000000000 * (100 + self.monthsSinceLastBirth)
    local isInseminated = 1000000000000000 * 1
    local hadABirth = 1000000000000000 * 1

    if self.isInseminated then
        isInseminated = 1000000000000000 * 2
    end

    if self.hadABirth then
        hadABirth = 1000000000000000 * 2
    end

    local hash = returnValue + monthsSinceLastBirth + isInseminated + hadABirth
    return hash
end

AnimalCluster.getHash = Utils.overwrittenFunction(AnimalCluster.getHash, EAS_AnimalClusterExtension.getHash)


function EAS_AnimalClusterExtension:getCanReproduce(superFunc)
	local subType = g_currentMission.animalSystem:getSubTypeByIndex(self:getSubTypeIndex())

	if subType.supportsReproduction then
		return subType.reproductionMinAgeMonth <= self.age
	end

	return false
end

AnimalCluster.getCanReproduce = Utils.overwrittenFunction(AnimalCluster.getCanReproduce, EAS_AnimalClusterExtension.getCanReproduce)

-- ################################# Load / Save / Update ####################################

function EAS_AnimalClusterExtension:saveToXMLFile(superFunc, xmlFile, key, usedModNames)
    superFunc(self, xmlFile, key, usedModNames)
	xmlFile:setBool(key .. "#isInseminated", self.isInseminated)
	xmlFile:setInt(key .. "#monthsSinceLastBirth", self.monthsSinceLastBirth)
	xmlFile:setBool(key .. "#hadABirth", self.hadABirth)
end

AnimalCluster.saveToXMLFile = Utils.overwrittenFunction(AnimalCluster.saveToXMLFile, EAS_AnimalClusterExtension.saveToXMLFile)

function EAS_AnimalClusterExtension:loadFromXMLFile(superFunc, xmlFile, key)
    local returnValue = superFunc(self, xmlFile, key)
	self.isInseminated = xmlFile:getBool(key .. "#isInseminated")
	self.hadABirth = xmlFile:getBool(key .. "#hadABirth")

    if self.isInseminated == nil then
        self.isInseminated = self.reproduction > 0
    end

	self.monthsSinceLastBirth = xmlFile:getInt(key .. "#monthsSinceLastBirth")

    local subType = g_currentMission.animalSystem:getSubTypeByIndex(self:getSubTypeIndex())
    local neededAge = subType.reproductionMinAgeMonth + subType.reproductionDurationMonth
    if self.monthsSinceLastBirth == nil then
        self.monthsSinceLastBirth = math.ceil(subType.reproductionDurationMonth * (self.reproduction / 100))
    end

    if self.hadABirth == nil then
        if self.age >= neededAge then
            self.hadABirth = true
        else
            self.hadABirth = false
        end
    end

	return returnValue
end

AnimalCluster.loadFromXMLFile = Utils.overwrittenFunction(AnimalCluster.loadFromXMLFile, EAS_AnimalClusterExtension.loadFromXMLFile)

function EAS_AnimalClusterExtension:readStream(superFunc, streamId, connection)
    superFunc(self, streamId, connection)
    self.isInseminated = streamReadBool(streamId)
    self.monthsSinceLastBirth = streamReadInt32(streamId)
    self.hadABirth = streamReadBool(streamId)
end

AnimalCluster.readStream = Utils.overwrittenFunction(AnimalCluster.readStream, EAS_AnimalClusterExtension.readStream)

function EAS_AnimalClusterExtension:writeStream(superFunc, streamId, connection)
    superFunc(self, streamId, connection)
    streamWriteBool(streamId, self.isInseminated)
    streamWriteInt32(streamId, self.monthsSinceLastBirth)
    streamWriteBool(streamId, self.hadABirth)
end

AnimalCluster.writeStream = Utils.overwrittenFunction(AnimalCluster.writeStream, EAS_AnimalClusterExtension.writeStream)

function EAS_AnimalClusterExtension:readUpdateStream(superFunc, streamId, connection)
    superFunc(self, streamId, connection)
    self.isInseminated = streamReadBool(streamId)
    self.monthsSinceLastBirth = streamReadInt32(streamId)
    self.hadABirth = streamReadBool(streamId)
end

AnimalCluster.readUpdateStream = Utils.overwrittenFunction(AnimalCluster.readUpdateStream, EAS_AnimalClusterExtension.readUpdateStream)

function EAS_AnimalClusterExtension:writeUpdateStream(superFunc, streamId, connection)
    superFunc(self, streamId, connection)
    streamWriteBool(streamId, self.isInseminated)
    streamWriteInt32(streamId, self.monthsSinceLastBirth)
    streamWriteBool(streamId, self.hadABirth)
end

AnimalCluster.writeUpdateStream = Utils.overwrittenFunction(AnimalCluster.writeUpdateStream, EAS_AnimalClusterExtension.writeUpdateStream)

-- ################################# Additional methods ####################################

function AnimalCluster:inseminate(numAnimals)
    if self.clusterSystem.isServer then
        if numAnimals == self:getNumAnimals() then
            self.isInseminated = true
            self:setDirty()
        else
            local inseminatedCluster = self:clone()
            inseminatedCluster.isInseminated = true
            inseminatedCluster:changeNumAnimals(numAnimals)
            self:changeNumAnimals(-numAnimals)
            self.clusterSystem:addPendingAddCluster(inseminatedCluster)
        end
	else
		g_client:getServerConnection():sendEvent(EAS_InseminateClusterEvent.new(self.clusterSystem.owner, self.id, numAnimals))
    end
end