--
-- AdvancedFarmManager - Vehicle Page
--

AIMGuiVehicleFrame = {}

local AIMGuiVehicleFrame_mt = Class(AIMGuiVehicleFrame, TabbedMenuFrameElement)

AIMGuiVehicleFrame.CONTROLS = {
    "vehicleList",
    "vehicleIcon",
    "vehicleDetail",
    "miniMap",
    "miniMapBG",
    "aimInfoSubVeh"
}


function AIMGuiVehicleFrame:new(l10n)
    local self = TabbedMenuFrameElement.new(nil,AIMGuiVehicleFrame_mt)

    self.messageCenter      = g_messageCenter
    self.l10n               = l10n
    self.vehicles           = {}

    self:registerControls(AIMGuiVehicleFrame.CONTROLS)

    return self
end


function AIMGuiVehicleFrame:copyAttributes(src)
    AIMGuiVehicleFrame:superClass().copyAttributes(self, src)

    self.ui   = src.ui
    self.l10n = src.l10n
end


function AIMGuiVehicleFrame:initialize()
    self.backButtonInfo = {inputAction = InputAction.MENU_BACK}

    self.activateButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_ACTIVATE,
        text        = self.l10n:getText("button_enterVehicle"),
        callback    = function ()
            self:onButtonEnterVehicle()
        end
    }
    self.acceptButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_EXTRA_2,
        text        = self.l10n:getText("button_clean"),
        callback    = function ()
            self:onButtonCleanVehicle()
        end
    }
    self.cancelButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_CANCEL,
        text        = self.l10n:getText("button_sell"),
        callback    = function ()
            self:onButtonSellVehicle()
        end
    }
    self.extra1ButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_EXTRA_1,
        text        = self.l10n:getText("button_repair"),
        callback    = function ()
            self:onButtonRepairVehicle()
        end
    }
    self.extra2ButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_ACCEPT,
        text        = self.l10n:getText("button_repaint"),
        callback    = function ()
            self:onButtonRepaintVehicle()
        end
    }
    self.parkButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.TOGGLE_STORE,
        text        = self.l10n:getText("aim_park"),
        callback    = function ()
            self:onButtonParkVehicle()
        end
    }
    self.renameButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.ACTIVATE_OBJECT,
        text        = self.l10n:getText("button_rename"),
        callback    = function ()
            self:onButtonRenameVehicle()
        end
    }
    self.refuelButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.TOGGLE_LIGHTS,
        text        = self.l10n:getText("action_refuel"),
        callback    = function ()
            self:onButtonRefuelVehicle()
        end
    }
end


function AIMGuiVehicleFrame:onGuiSetupFinished()
    AIMGuiVehicleFrame:superClass().onGuiSetupFinished(self)
    self.vehicleList:setDataSource(self)
    self.vehicleDetail:setDataSource(self)
end


function AIMGuiVehicleFrame:delete()
    AIMGuiVehicleFrame:superClass().delete(self)
    self.messageCenter:unsubscribeAll(self)
end


function AIMGuiVehicleFrame:updateMenuButtons()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    self.menuButtonInfo = {
        {
            inputAction = InputAction.MENU_BACK
        }
    }

    if thisVehicle ~= nil then
        local repairPrice   = thisVehicle:getRepairPrice()
        local repaintPrice  = thisVehicle:getRepaintPrice()
        local isParked      = thisVehicle:getHKParkVehicleState()
        local cleanPrice    = self:vehicleCleanCost(thisVehicle)
        local isLeased      = thisVehicle.propertyState == Vehicle.PROPERTY_STATE_LEASED
        local isBorrowed    = thisVehicle.propertyState == Vehicle.PROPERTY_STATE_MISSION
        local needsDiesel   = self:getDiesel(thisVehicle)
        local isMPGame      = g_currentMission.missionDynamicInfo.isMultiplayer

        table.insert(self.menuButtonInfo, self.activateButtonInfo)

        if g_currentMission:getHasPlayerPermission("farmManager") and not isBorrowed then
            if not isLeased then
                self.cancelButtonInfo.text = g_i18n:getText("button_sell")
            else
                self.cancelButtonInfo.text  = g_i18n:getText("button_return")
            end

            table.insert(self.menuButtonInfo, self.cancelButtonInfo)

            if repairPrice >= 1 then
                self.extra1ButtonInfo.text = g_i18n:getText("button_repair")
                table.insert(self.menuButtonInfo, self.extra1ButtonInfo)
            end

            if repaintPrice >= 1 then
                self.extra2ButtonInfo.text = g_i18n:getText("button_repaint")
                table.insert(self.menuButtonInfo, self.extra2ButtonInfo)
            end

            if isMPGame ~= true and cleanPrice >= 1 then
                self.acceptButtonInfo.text = g_i18n:getText("button_clean")
                table.insert(self.menuButtonInfo, self.acceptButtonInfo)
            end

            if isMPGame ~= true then
                table.insert(self.menuButtonInfo, self.renameButtonInfo)
            end

            if isMPGame ~= true and needsDiesel ~= false and needsDiesel > 1 then
                table.insert(self.menuButtonInfo, self.refuelButtonInfo)
            end
        end

        if isParked then
            self.parkButtonInfo.text = g_i18n:getText("aim_unpark")
        else
            self.parkButtonInfo.text = g_i18n:getText("aim_park")
        end

        table.insert(self.menuButtonInfo, self.parkButtonInfo)
    end

    self:setMenuButtonInfoDirty()
end


function AIMGuiVehicleFrame:onFrameOpen()
    AIMGuiVehicleFrame:superClass().onFrameOpen(self)

    if AdditionalInfo.debug then
        print("~~ AdvancedFarmManager Debug ... AIMGuiVehicleFrame:onFrameOpen")
    end

    self:rebuildTable()

    self:setSoundSuppressed(true)
    FocusManager:setFocus(self.vehicleList)
    self:setSoundSuppressed(false)

    self.messageCenter:subscribe(SellVehicleEvent, self.onRefreshEvent, self)
    self.messageCenter:subscribe(BuyVehicleEvent, self.onRefreshEvent, self)
    self.messageCenter:subscribe(MessageType.VEHICLE_REPAIRED, self.onRefreshEvent, self)
    self.messageCenter:subscribe(MessageType.VEHICLE_REPAINTED, self.onRefreshEvent, self)
end


function AIMGuiVehicleFrame:onRefreshEvent()
    self:rebuildTable()
end


function AIMGuiVehicleFrame:onFrameClose()
    AIMGuiVehicleFrame:superClass().onFrameClose(self)

    self.vehicles = {}
    self.messageCenter:unsubscribeAll(self)
end


function AIMGuiVehicleFrame:rebuildTable()
    if AdditionalInfo.debug then
        print("~~ AdvancedFarmManager Debug ... AIMGuiVehicleFrame:rebuildTable")
    end

    self.vehicles = {}

    if g_currentMission.player ~= nil then
        for _, vehicle in ipairs(g_currentMission.vehicles) do
            local isSelling        = (vehicle.isDeleted ~= nil and vehicle.isDeleted) or (vehicle.isDeleting ~= nil and vehicle.isDeleting)
            local hasAccess        = g_currentMission.accessHandler:canPlayerAccess(vehicle)
            local hasConned        = vehicle.getIsControlled ~= nil
            local isProperty       = vehicle.propertyState == Vehicle.PROPERTY_STATE_OWNED or vehicle.propertyState == Vehicle.PROPERTY_STATE_LEASED or vehicle.propertyState == Vehicle.PROPERTY_STATE_MISSION
            local isPallet         = vehicle.typeName == "pallet"
            local isTrain          = vehicle.typeName == "locomotive"
            local isBelt           = vehicle.typeName == "conveyorBelt" or vehicle.typeName == "pickupConveyorBelt"
            local isRidable        = SpecializationUtil.hasSpecialization(Rideable, vehicle.specializations)
            local isSteerImplement = vehicle.spec_attachable ~= nil

            local skipable         = isTrain or isBelt or isRidable or isSteerImplement or isPallet

            if not isSelling and not skipable and hasConned and hasAccess and vehicle.getSellPrice ~= nil and vehicle.price ~= nil and isProperty then
                table.insert(self.vehicles, vehicle)
            end
        end
    end

    self.vehicleList:reloadData()
    self.vehicleDetail:reloadData()
end

function AIMGuiVehicleFrame:getNumberOfItemsInSection(list, section)
    if list == self.vehicleList then
        return #self.vehicles
    else
        return 8
    end
end

function AIMGuiVehicleFrame:populateCellForItemInSection(list, section, index, cell)
    if list == self.vehicleList then
        local thisVehicle = self.vehicles[index]
        local name        = thisVehicle:getFullName()
        local thisColor   = { 1, 1, 1, 1 }
        local thisHotKey  = 0
        local isParked    = false

        if thisVehicle.getHotKeyVehicleState ~= nil then
            thisHotKey = thisVehicle:getHotKeyVehicleState()
            isParked   = thisVehicle:getHKParkVehicleState()
        end

        cell:getAttribute("statusRunning"):setVisible(false)
        cell:getAttribute("statusParked"):setVisible(false)
        cell:getAttribute("statusMine"):setVisible(false)

        if g_currentMission.missionDynamicInfo.isMultiplayer then
            if thisVehicle.ownerFarmId == g_currentMission:getFarmId() then
                cell:getAttribute("statusMine"):setVisible(true)
            end
        end

        if thisVehicle.getIsMotorStarted ~= nil and thisVehicle:getIsMotorStarted() then
            cell:getAttribute("statusRunning"):setVisible(true)
            thisColor = { 0.159, 0.440, 0.287, 1}
        end

        if isParked then
            cell:getAttribute("statusParked"):setVisible(true)
            thisColor = { 0.2195, 0.2233, 0.2310, 1}
        end

        if thisVehicle.getDamageAmount ~= nil and thisVehicle:getDamageAmount() > 0.12 then
            thisColor = { 0.413, 0.070, 0.078, 1}
        end

        if thisVehicle.propertyState == Vehicle.PROPERTY_STATE_MISSION then
            thisColor = { 0.223, 0.503, 0.807, 1 }
        end

        if thisHotKey ~= nil and thisHotKey > 0 then
            cell:getAttribute("hotkey"):setText(tostring(thisHotKey))
        else
            cell:getAttribute("hotkey"):setText("")
        end

        cell:getAttribute("title"):setText(name)
        cell:getAttribute("title").textColor = thisColor
        cell:getAttribute("power"):setText(self:powerString(thisVehicle, true))

        cell:getAttribute("location"):setText(self:getLocation(thisVehicle))
    else
        local selectedIndex      = self.vehicleList.selectedIndex
        local thisVehicle        = self.vehicles[selectedIndex]
        local attachedImplements = {}
        local thisRawAmount      = 0

        if thisVehicle ~= nil then
            -- (re)set visible.
            self.miniMapBG:setVisible(true)
            self.vehicleIcon:setVisible(true)
            self.vehicleDetail:setVisible(true)
            self.aimInfoSubVeh:setVisible(true)

            if thisVehicle.getAttachedImplements ~= nil then
                attachedImplements = thisVehicle:getAttachedImplements()
            end

            if index == 1 then
                self:setDetailText(
                    cell, true,
                    "aim_occupant",
                    AIMGuiVehicleFrame:vehicleOccupant(thisVehicle)
                )
            elseif index == 2 then -- FUEL LEVEL
                local fuelLevel = self:getFuel(thisVehicle)
                if fuelLevel[1] ~= false then
                    self:setStatusBarValue(cell:getAttribute("detailBar"), fuelLevel[2], 1-fuelLevel[2], 0.3, 0.6)
                    self:setDetailText(
                        cell, false,
                        fuelLevel[1],
                        AIMGuiVehicleFrame:rawToPerc(fuelLevel[2], false)
                    )
                end
            elseif index == 3 then  -- DEF WHEN LOADED AND EXISTS.
                local defLevel = self:getDEF(thisVehicle)
                if defLevel ~= nil then
                    self:setStatusBarValue(cell:getAttribute("detailBar"), defLevel, 1-defLevel, 0.3, 0.6)
                    self:setDetailText(
                        cell, false,
                        "fillType_def",
                        AIMGuiVehicleFrame:rawToPerc(defLevel, false)
                    )
                else
                    self:setDetailText(cell, true, "", "")
                end
            elseif index == 4 then
                if thisVehicle.getDamageAmount ~= nil then
                    thisRawAmount = thisVehicle:getDamageAmount()
                end

                self:setStatusBarValue(cell:getAttribute("detailBar"), thisRawAmount, thisRawAmount, 0.1, 0.2)
                self:setDetailText(
                    cell, false,
                    "infohud_damage",
                    AIMGuiVehicleFrame:rawToPerc(thisRawAmount, false)
                )
            elseif index == 5 then
                if thisVehicle.getWearTotalAmount ~= nil then
                    thisRawAmount = thisVehicle:getWearTotalAmount()
                end

                self:setStatusBarValue(cell:getAttribute("detailBar"), 1-thisRawAmount, thisRawAmount, 0.3, 0.6)
                self:setDetailText(
                    cell, false,
                    "ui_paintCondition",
                    AIMGuiVehicleFrame:rawToPerc(thisRawAmount, true)
                )
            elseif index == 6 then
                if thisVehicle.getDirtAmount ~= nil then
                    thisRawAmount = thisVehicle:getDirtAmount()
                end

                self:setStatusBarValue(cell:getAttribute("detailBar"), thisRawAmount, thisRawAmount, 0.3, 0.6)
                self:setDetailText(
                    cell, false,
                    "setting_dirt",
                    AIMGuiVehicleFrame:rawToPerc(thisRawAmount, false)
                )
            elseif index == 7 then
                cell:setVisible(false)
                if #attachedImplements >= 1 then
                    local thisImp   = attachedImplements[1]
                    if thisImp.object ~= nil then
                        local thisName  = thisImp.object:getFullName()

                        self:setDetailText(cell, true, "aim_tool1", thisName)
                        cell:setVisible(true)
                    end
                end
            elseif index == 8 then
                cell:setVisible(false)
                if #attachedImplements >= 2 then
                    local thisImp   = attachedImplements[2]
                    if thisImp.object ~= nil then
                        local thisName  = thisImp.object:getFullName()

                        self:setDetailText(cell, true, "aim_tool2", thisName)
                        cell:setVisible(true)
                    end
                end
            end
        else
            -- hide UI, no vehicle
            self.miniMapBG:setVisible(false)
            self.vehicleIcon:setVisible(false)
            self.vehicleDetail:setVisible(false)
            self.aimInfoSubVeh:setVisible(false)
        end
    end
end


function AIMGuiVehicleFrame:setDetailText(cell, nukeBar, title, level)
    cell:getAttribute("detailBarBG"):setVisible(true)
    cell:getAttribute("detailBar"):setVisible(true)

    if title == "" then
        cell:getAttribute("detailTitle"):setText("")
    else
        cell:getAttribute("detailTitle"):setText(g_i18n:getText(title))
    end

    cell:getAttribute("detailLevel"):setText(level)

    if nukeBar then
        cell:getAttribute("detailLevel"):applyProfile("aimMenuVehicleDetailLevelNoBar")
        cell:getAttribute("detailTitle"):applyProfile("aimMenuVehicleDetailTitleNoBar")
        cell:getAttribute("detailBarBG"):setVisible(false)
        cell:getAttribute("detailBar"):setVisible(false)
    end
end

function AIMGuiVehicleFrame:setStatusBarValue(statusBarElement, value, rawValue, levelGood, levelWarn)

    if rawValue < levelGood then
        statusBarElement:applyProfile("aimMenuVehicleDetailBar")
    elseif rawValue < levelWarn then
        statusBarElement:applyProfile("aimMenuVehicleDetailBarWarning")
    else
        statusBarElement:applyProfile("aimMenuVehicleDetailBarDanger")
    end

    local fullWidth = statusBarElement.parent.absSize[1] - statusBarElement.margin[1] * 2
    local minSize = 0

    if statusBarElement.startSize ~= nil then
        minSize = statusBarElement.startSize[1] + statusBarElement.endSize[1]
    end

    statusBarElement:setSize(math.max(minSize, fullWidth * math.min(1, value)), nil)
end


function AIMGuiVehicleFrame:powerString(vehicle, noBrace)
    local storeItem   = g_storeManager:getItemByXMLFilename(vehicle.configFileName)

    if vehicle.configurations == nil or vehicle.configurations.motor == nil then
        return ""
    end

    local boughtMotor = vehicle.configurations.motor
    local motorPower  = storeItem.configurations.motor[boughtMotor].power

    if motorPower == nil then return "" end

    local hp, _ = g_i18n:getPower(motorPower)

    local returnText = string.format(g_i18n:getText("shop_maxPowerValueSingle"), math.floor(hp))

    if noBrace == nil or noBrace == false then
        return " [" .. returnText .. "]"
    end

    return returnText
end


function AIMGuiVehicleFrame:rawToPerc(value, invert)
    if not invert then
        return math.ceil((value)*100) .. " %"
    end
    return math.ceil((1 - value)*100) .. " %"
end


function AIMGuiVehicleFrame:getDEF(vehicle)
    local defFillUnitIndex = vehicle:getConsumerFillUnitIndex(FillType.DEF)

    if defFillUnitIndex ~= nil then
        local fillLevel = vehicle:getFillUnitFillLevel(defFillUnitIndex)
        local capacity  = vehicle:getFillUnitCapacity(defFillUnitIndex)
        return fillLevel / capacity
    end
    return nil
end

function AIMGuiVehicleFrame:getDiesel(vehicle)
    if vehicle.getConsumerFillUnitIndex == nil then
        return false
    end

    local fillIndex = vehicle:getConsumerFillUnitIndex(FillType.DIESEL)

    if fillIndex == nil then
        return false
    end

    local fuelLevel  = vehicle:getFillUnitFillLevel(fillIndex)
    local capacity   = vehicle:getFillUnitCapacity(fillIndex)
    local toFill     = capacity - fuelLevel

    return toFill
end


function AIMGuiVehicleFrame:getFuel(vehicle)
    local fuelTypeList = {
        {
            FillType.DIESEL,
            "fillType_diesel",
        }, {
            FillType.ELECTRICCHARGE,
            "fillType_electricCharge",
        }, {
            FillType.METHANE,
            "fillType_methane",
        }
    }
    if vehicle.getConsumerFillUnitIndex ~= nil then
        -- This should always pass, unless it's a very odd custom vehicle type.
        for _, fuelType in pairs(fuelTypeList) do
            local fillUnitIndex = vehicle:getConsumerFillUnitIndex(fuelType[1])
            if ( fillUnitIndex ~= nil ) then
                local fuelLevel  = vehicle:getFillUnitFillLevel(fillUnitIndex)
                local capacity   = vehicle:getFillUnitCapacity(fillUnitIndex)
                local percentage = fuelLevel / capacity
                return { fuelType[2], percentage }
            end
        end
    end
    return { false } -- unknown fuel type, should not be possible.
end

function AIMGuiVehicleFrame:vehicleCleanCost(vehicle)
    local thisRawAmount = 0
    if vehicle.getDirtAmount ~= nil then
        thisRawAmount = vehicle:getDirtAmount()
    end

    if thisRawAmount == 0 then
        return 0
    elseif thisRawAmount < 0.25 then
        return 150
    elseif thisRawAmount < 0.5 then
        return 250
    elseif thisRawAmount > 0.9 then
        return 500
    else
        return 400
    end
end

function AIMGuiVehicleFrame:vehicleRunning(vehicle)
    if vehicle.getIsMotorStarted ~= nil and vehicle:getIsMotorStarted() then
        return g_i18n:getText("ui_yes")
    end

    return g_i18n:getText("ui_no")
end


function AIMGuiVehicleFrame:vehicleParked(vehicle)
    if vehicle.getIsTabbable ~= nil and ( not vehicle:getIsTabbable() ) then
        return g_i18n:getText("ui_yes")
    end

    return g_i18n:getText("ui_no")
end


function AIMGuiVehicleFrame:vehicleOccupant(vehicle)
    if vehicle.getIsControlled ~= nil and vehicle:getIsControlled() then
        if vehicle.getControllerName ~= nil then
            return vehicle:getControllerName()
        else
            return g_i18n:getText("aim_you")
        end
    end

    if vehicle.getIsAIActive ~= nil and vehicle:getIsAIActive() then
        if vehicle.ad ~= nil and vehicle.ad.stateModule ~= nil and vehicle.ad.stateModule:isActive() then
            return g_i18n:getText("aim_con_ad")
        elseif vehicle.getCpStatus ~= nil then
            local cpStatus = vehicle:getCpStatus()
            if cpStatus:getIsActive() then
                return g_i18n:getText("aim_con_cp")
            end
        else
            return g_i18n:getText("aim_con_ai")
        end
    end

    return g_i18n:getText("ui_none")
end


function AIMGuiVehicleFrame:onListSelectionChanged(list, section, index)
    if list == self.vehicleList then
        local thisVehicle = self.vehicles[index]
        local storeItem   = g_storeManager:getItemByXMLFilename(thisVehicle.configFileName)

        self.vehicleIcon:setImageFilename(storeItem.imageFilename)

        local vehicleInfoExtra = {}

        if g_currentMission.missionDynamicInfo.isMultiplayer then
            local thisFarmID   = thisVehicle.ownerFarmId
            local thisFarm     = g_farmManager:getFarmById(thisFarmID)
            local thisFarmName = thisFarm.name
            if thisFarmName ~= nil then
                table.insert(vehicleInfoExtra, thisFarmName)
            end
        end

        local powerString = self:powerString(thisVehicle, true)

        if powerString ~= "" then
            table.insert(vehicleInfoExtra, powerString)
        end
        if thisVehicle.propertyState == Vehicle.PROPERTY_STATE_MISSION then
            table.insert(vehicleInfoExtra, g_i18n:getText("fieldJob_contract"))
        end
        if thisVehicle.propertyState == Vehicle.PROPERTY_STATE_LEASED then
            table.insert(vehicleInfoExtra, g_i18n:getText("button_lease"))
        end
        if thisVehicle.getIsMotorStarted ~= nil and thisVehicle:getIsMotorStarted() then
            table.insert(vehicleInfoExtra, g_i18n:getText("ui_production_status_running"))
        end
        if thisVehicle:getHKParkVehicleState() then
            table.insert(vehicleInfoExtra, g_i18n:getText("aim_parked"))
        end


        self.aimInfoSubVeh:setText(table.concat(vehicleInfoExtra, ", "))

        local imageFile = g_currentMission.mapImageFilename

        if imageFile ~= self.miniMap.overlay.filename then
            self.miniMap.imageSize = {g_currentMission.terrainSize*2, g_currentMission.terrainSize*2}
            self.miniMap:setImageFilename(imageFile)
        end

        local imageSizeTable      = self.miniMap.imageSize
        local centerFactor        = g_currentMission.terrainSize
        local centerU, _, centerV = getWorldTranslation(thisVehicle.rootNode)

        local startU              = tostring(math.floor(centerFactor + centerU - 200)) .. "px"
        local startV              = tostring(math.floor(centerFactor + centerV - 200)) .. "px"

        local UVString            = startU .. " " .. startV .. " 400px 400px"

        local theseUVs            = GuiUtils.getUVs(UVString, imageSizeTable)

        self.miniMap:setImageUVs(nil, unpack(theseUVs))

        if AdditionalInfo.debug then
            print("~~ AdvancedFarmManager Debug ... newMapPosition: " .. UVString)
        end

        self.vehicleDetail:reloadData()
    end

    self:updateMenuButtons()
end


function AIMGuiVehicleFrame:getLocation(vehicle)
    local fieldNumber = 0
    local isField     = false
    local wx, wy, wz  = 0, 0, 0

    local function getIsOnField()
        if vehicle.components == nil then return false end

        for _, component in pairs(vehicle.components) do
            wx, wy, wz = localToWorld(component.node, getCenterOfMass(component.node))

            local h = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, wx, wy, wz)

            if h-1 > wy then
                break
            end

            local isOnField, _ = FSDensityMapUtil.getFieldDataAtWorldPosition(wx, wy, wz)
            if isOnField then
                isField = true
                return true
            end
        end
        return false
    end
    if getIsOnField() then
        local farmlandId = g_farmlandManager:getFarmlandIdAtWorldPosition(wx, wz)
        if farmlandId ~= nil then

            local foundField = false

            for f=1, #g_fieldManager.fields do

                if foundField then break end

                local field = g_fieldManager.fields[f]

                if field ~= nil and field.farmland ~= nil and field.farmland.id == farmlandId then
                    local fieldId = field.fieldId

                    -- set this as a "fall back" if we don't get a "real" field number below
                    -- this is likely to happen on any enlarged fields, and at the borders of a lot
                    -- of the base game maps.
                    fieldNumber = fieldId

                    for a=1, #field.setFieldStatusPartitions do
                        local b                    = field.setFieldStatusPartitions[a]
                        local x, z, wX, wZ, hX, hZ = b.x0, b.z0, b.widthX, b.widthZ, b.heightX, b.heightZ
                        local distanceMax          = math.max(wX,wZ,hX,hZ)
                        local distance             = MathUtil.vector2Length(wx-x,wz-z);
                        if distance <= distanceMax then
                            fieldNumber = fieldId
                            foundField  = true
                            break
                        end
                    end
                end
            end
        end
    end
    if isField then
        return string.format("F-%03d", fieldNumber)
    else
        return "--"
    end
end


function AIMGuiVehicleFrame:onButtonRefuelVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]
    local needsDiesel   = self:getDiesel(thisVehicle)
    local fuelCost      = g_currentMission.economyManager:getPricePerLiter(FillType.DIESEL) * needsDiesel * 1.2

    g_gui:showYesNoDialog({
        text      = string.format(g_i18n:getText("aim_refuel_ask"), g_i18n:formatMoney(fuelCost)),
        callback  = self.onYesNoRefuelDialog,
        target    = self,
        yesButton = g_i18n:getText("button_yes"),
        noButton  = g_i18n:getText("button_cancel")
    })
end


function AIMGuiVehicleFrame:onYesNoRefuelDialog(yes)
    if yes then
        local selectedIndex = self.vehicleList.selectedIndex
        local thisVehicle   = self.vehicles[selectedIndex]
        local needsDiesel   = self:getDiesel(thisVehicle)
        local fillUnitIndex = thisVehicle:getConsumerFillUnitIndex(FillType.DIESEL)
        local farmId        = thisVehicle:getActiveFarm()

        if needsDiesel == false then
            return
        end

        local delta = thisVehicle:addFillUnitFillLevel(farmId, fillUnitIndex, needsDiesel, FillType.DIESEL, ToolType.TRIGGER, nil)

        if delta > 0 then
            local price = delta * g_currentMission.economyManager:getPricePerLiter(FillType.DIESEL) * 1.2
            g_farmManager:updateFarmStats(farmId, "expenses", price)
            g_currentMission:addMoney(-price, farmId, MoneyType.PURCHASE_FUEL, true)
        end
        self:rebuildTable()
    end
end


function AIMGuiVehicleFrame:onButtonParkVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    thisVehicle:setHKParkVehicleState(not thisVehicle:getHKParkVehicleState())

    self:rebuildTable()
end


function AIMGuiVehicleFrame:onButtonEnterVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]
    local isConned      = thisVehicle.getIsControlled ~= nil and thisVehicle:getIsControlled()

    if not isConned then
        -- available, switch to it
        g_currentMission:requestToEnterVehicle(thisVehicle)
        g_gui:showGui("")
    elseif thisVehicle.spec_enterable.controllerUserId ==  g_currentMission.playerUserId then
        -- already in it, just close GUI
        g_gui:showGui("")
    else
        -- occupied, refuse request
        g_gui:showInfoDialog({
            text = g_i18n:getText("aim_occupied")
        })
    end
end


function AIMGuiVehicleFrame:onButtonSellVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]
    local isLeased      = thisVehicle.propertyState == Vehicle.PROPERTY_STATE_LEASED
    local l10nText      = "ui_youWantToSellVehicle"

    if isLeased then
        l10nText = "ui_youWantToReturnVehicle"
    end

    g_gui:showYesNoDialog({
        text      = g_i18n:getText(l10nText),
        callback  = self.onYesNoSellDialog,
        target    = self,
        yesButton = g_i18n:getText("button_yes"),
        noButton  = g_i18n:getText("button_cancel")
    })
end


function AIMGuiVehicleFrame:onYesNoSellDialog(yes)
    if yes then
        local selectedIndex = self.vehicleList.selectedIndex
        local thisVehicle   = self.vehicles[selectedIndex]

        g_client:getServerConnection():sendEvent(SellVehicleEvent.new(thisVehicle, EconomyManager.DIRECT_SELL_MULTIPLIER, true))
    end
end


function AIMGuiVehicleFrame:onButtonRepairVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if thisVehicle ~= nil and thisVehicle:getRepairPrice(true) >= 1 then
        g_gui:showYesNoDialog({
            text     = string.format(g_i18n:getText("ui_repairDialog"), g_i18n:formatMoney(thisVehicle:getRepairPrice(true))),
            callback = self.onYesNoRepairDialog,
            target   = self,
            yesSound = GuiSoundPlayer.SOUND_SAMPLES.CONFIG_WRENCH
        })

        return true
    else
        return false
    end
end


function AIMGuiVehicleFrame:onButtonRepaintVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if thisVehicle ~= nil and thisVehicle:getRepaintPrice() >= 1 then
        g_gui:showYesNoDialog({
            text     = string.format(g_i18n:getText("ui_repaintDialog"), g_i18n:formatMoney(thisVehicle:getRepaintPrice())),
            callback = self.onYesNoRepaintDialog,
            target   = self,
            yesSound = GuiSoundPlayer.SOUND_SAMPLES.CONFIG_SPRAY
        })

        return true
    else
        return false
    end
end


function AIMGuiVehicleFrame:onButtonCleanVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]
    local thisCost      = self:vehicleCleanCost(thisVehicle)

    if thisVehicle ~= nil and thisCost >= 1 then
        g_gui:showYesNoDialog({
            text     = string.format(g_i18n:getText("aim_cleanDialog"), g_i18n:formatMoney(thisCost)),
            callback = self.onYesNoCleanDialog,
            target   = self,
            yesSound = GuiSoundPlayer.SOUND_SAMPLES.CONFIG_SPRAY
        })

        return true
    else
        return false
    end
end


function AIMGuiVehicleFrame:onYesNoCleanDialog(yes)
    if yes then
        local selectedIndex = self.vehicleList.selectedIndex
        local thisVehicle   = self.vehicles[selectedIndex]
        local farmId        = thisVehicle:getActiveFarm()
        local thisCost      = self:vehicleCleanCost(thisVehicle)

        if g_currentMission:getMoney() <  thisCost then
            g_gui:showInfoDialog({
                text = g_i18n:getText("shop_messageNotEnoughMoneyToBuy")
            })
        else
            g_currentMission:addMoney(-1 * thisCost, farmId, MoneyType.OTHER, true)
            self:cleanVehicle(thisVehicle)
        end

        self:rebuildTable()
    end
end


function AIMGuiVehicleFrame:cleanVehicle(vehicle)
    if vehicle ~= nil and vehicle.spec_washable ~= nil and vehicle.spec_washable.washableNodes ~= nil then
        for _, nodeData in ipairs(vehicle.spec_washable.washableNodes) do
            vehicle.spec_washable:setNodeDirtAmount(nodeData, 0, true)
        end
    end

    if vehicle.getAttachedImplements ~= nil then
        local attachedImplements = vehicle:getAttachedImplements();
        for _, implement in pairs(attachedImplements) do
            if implement.object ~= nil then
                self:cleanVehicle(implement)
            end
        end
    end
end


function AIMGuiVehicleFrame:onYesNoRepairDialog(yes)
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if yes then
        if g_currentMission:getMoney() < thisVehicle:getRepairPrice() then
            g_gui:showInfoDialog({
                text = g_i18n:getText("shop_messageNotEnoughMoneyToBuy")
            })
        else
            g_client:getServerConnection():sendEvent(WearableRepairEvent.new(thisVehicle, true))
        end
    end
end


function AIMGuiVehicleFrame:onYesNoRepaintDialog(yes)
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if yes then
        if g_currentMission:getMoney() < thisVehicle:getRepaintPrice() then
            g_gui:showInfoDialog({
                text = g_i18n:getText("shop_messageNotEnoughMoneyToBuy")
            })
        else
            g_client:getServerConnection():sendEvent(WearableRepaintEvent.new(thisVehicle, true))
        end
    end
end


function AIMGuiVehicleFrame:onButtonRenameVehicle()
    local selectedIndex  = self.vehicleList.selectedIndex
    local thisVehicle    = self.vehicles[selectedIndex]
    local defaultNewName = ""

    if thisVehicle.spec_hotKeyVehicle.nickname ~= nil then
        defaultNewName = thisVehicle.spec_hotKeyVehicle.nickname
    end

    g_gui:showTextInputDialog({
        text        = g_i18n:getText("button_changeName"),
        defaultText = defaultNewName,
        callback    = function (result, yes)
            if yes then
                if result:len() == 0 then
                    result = ""
                end

                local sIndex               = self.vehicleList.selectedIndex
                local thisChangeVehicle    = self.vehicles[sIndex]
                thisChangeVehicle:setHotKeyNickName(result)
                self:rebuildTable()
            end
        end,
        maxCharacters = 40,
        dialogPrompt  = g_i18n:getText("ui_enterName"),
        imePrompt     = g_i18n:getText("ui_enterName"),
        confirmText   = g_i18n:getText("button_change")
    })
end