--
-- AdvancedFarmManager - Placeable Page
--

AIMGuiPlaceableFrame = {}

local AIMGuiPlaceableFrame_mt = Class(AIMGuiPlaceableFrame, TabbedMenuFrameElement)

AIMGuiPlaceableFrame.CONTROLS = {
    "placeList",
    "placeIcon",
}


function AIMGuiPlaceableFrame:new(l10n)
    local self = TabbedMenuFrameElement.new(nil,AIMGuiPlaceableFrame_mt)

    self.messageCenter      = g_messageCenter
    self.l10n               = l10n
    self.placables          = {}

    self:registerControls(AIMGuiPlaceableFrame.CONTROLS)

    return self
end


function AIMGuiPlaceableFrame:copyAttributes(src)
    AIMGuiPlaceableFrame:superClass().copyAttributes(self, src)

    self.ui   = src.ui
    self.l10n = src.l10n
end


function AIMGuiPlaceableFrame:initialize()
    self.backButtonInfo = {inputAction = InputAction.MENU_BACK}

    self.activateButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_ACTIVATE,
        text        = self.l10n:getText("aim_warp_loc"),
        callback    = function ()
            self:onButtonWarpPlaceable()
        end
    }
    self.cancelButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_CANCEL,
        text        = self.l10n:getText("button_sell"),
        callback    = function ()
            self:onButtonSellPlaceable()
        end
    }
    self.hotspotButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_ACCEPT,
        text        = self.l10n:getText("action_tag"),
        callback    = function ()
            self:onButtonHotspotPlaceable()
        end
    }
end


function AIMGuiPlaceableFrame:onGuiSetupFinished()
    AIMGuiPlaceableFrame:superClass().onGuiSetupFinished(self)
    self.placeList:setDataSource(self)
end


function AIMGuiPlaceableFrame:delete()
    AIMGuiPlaceableFrame:superClass().delete(self)
    self.messageCenter:unsubscribeAll(self)
end


function AIMGuiPlaceableFrame:updateMenuButtons()
    local selectedIndex = self.placeList.selectedIndex
    local thisPlace     = self.placables[selectedIndex]
    local sellPrice     = math.floor(thisPlace:getSellPrice())

    self.menuButtonInfo = {
        {
            inputAction = InputAction.MENU_BACK
        }
    }

    if selectedIndex > 0 then
        table.insert(self.menuButtonInfo, self.activateButtonInfo)

        self.cancelButtonInfo.text = string.format(
            "%s (%s)",
            g_i18n:getText("button_sell"),
            g_i18n:formatMoney(sellPrice, 0, true, true)
        )

        if thisPlace:canBeSold() and g_currentMission:getHasPlayerPermission("farmManager") then
            table.insert(self.menuButtonInfo, self.cancelButtonInfo)
        end

        if self:getHotspot() ~= nil then
            table.insert(self.menuButtonInfo, self.hotspotButtonInfo)
        end
    end

    self:setMenuButtonInfoDirty()
end


function AIMGuiPlaceableFrame:onFrameOpen()
    AIMGuiPlaceableFrame:superClass().onFrameOpen(self)

    if AdditionalInfo.debug then
        print("~~ AdvancedFarmManager Debug ... AIMGuiPlaceableFrame:onFrameOpen")
    end

    self.placeIcon:setVisible(false)

    self:rebuildTable()

    self:setSoundSuppressed(true)
    FocusManager:setFocus(self.placeList)
    self:setSoundSuppressed(false)

    self.messageCenter:subscribe(SellPlaceableEvent, self.onRefreshEvent, self)
    self.messageCenter:subscribe(BuyPlaceableEvent, self.onRefreshEvent, self)
    self.messageCenter:subscribe(MessageType.FARM_PROPERTY_CHANGED, self.onRefreshEvent, self)
end


function AIMGuiPlaceableFrame:onRefreshEvent()
    self:rebuildTable()
end


function AIMGuiPlaceableFrame:onFrameClose()
    AIMGuiPlaceableFrame:superClass().onFrameClose(self)

    self.placables = {}
    self.messageCenter:unsubscribeAll(self)
end


function AIMGuiPlaceableFrame:rebuildTable()
    self.placables = {}

    for _, placeable in pairs(g_currentMission.placeableSystem.placeables) do
        if placeable.ownerFarmId == g_currentMission.player.farmId and placeable.ownerFarmId ~= FarmManager.SPECTATOR_FARM_ID and not placeable.markedForDeletion and not placeable.isDeleted and not placeable.isDeleteing then
            table.insert(self.placables, placeable)
        end
    end

    table.sort(self.placables, function (k1, k2)
        local k1name = k1.storeItem.name
        local k2name = k2.storeItem.name

        if k1.brand ~= nil and k1.brand.title ~= nil and k1.brand.title ~= "None" then
            k1name = k1.brand.title .. " " .. k1name
        end
        if k2.brand ~= nil and k2.brand.title ~= nil and k2.brand.title ~= "None" then
            k2name = k2.brand.title .. " " .. k2name
        end

        return k1name < k2name
    end )

    self.placeList:reloadData()
end

function AIMGuiPlaceableFrame:getNumberOfItemsInSection(list, section)
    if list == self.placeList then
        return #self.placables
    else
        return 0
    end
end

function AIMGuiPlaceableFrame:populateCellForItemInSection(list, section, index, cell)
    if list == self.placeList then
        local thisPlace   = self.placables[index]
        local sellPrice   = math.floor(thisPlace:getSellPrice())
        local name        = thisPlace:getName()
        local wx, _, wz   = getWorldTranslation(thisPlace.rootNode)

        if thisPlace.brand ~= nil and thisPlace.brand.title ~= nil and thisPlace.brand.title ~= "None" then
            name = thisPlace.brand.title .. " " .. name
        end
        if not thisPlace:canBeSold() then
            name = name .. " [" .. g_i18n:getText("aim_nonSell") .. "]"
        end

        cell:getAttribute("title"):setText(name)
        cell:getAttribute("location"):setText(self:locationToText(thisPlace))
        cell:getAttribute("value"):setText(g_i18n:formatMoney(sellPrice, 0, true, true))
    end
end


function AIMGuiPlaceableFrame:onListSelectionChanged(list, section, index)
    if list == self.placeList then
        local thisPlace   = self.placables[index]
        local storeItem   = g_storeManager:getItemByXMLFilename(thisPlace.configFileName)

        if index > 0 then
            self.placeIcon:setVisible(true)
            self.placeIcon:setImageFilename(storeItem.imageFilename)
        else
            self.placeIcon:setVisible(false)
        end
    end

    self:updateMenuButtons()
end


function AIMGuiPlaceableFrame:locationToText(thisPlace)
    local thirdTerrainSize = g_currentMission.terrainSize / 6
    local x, _, z   = getWorldTranslation(thisPlace.rootNode)

    if x == 0 and z == 0 then
        return "--"
    end

    if math.abs(x) < thirdTerrainSize and math.abs(z) < thirdTerrainSize then
        return g_i18n:getText("aim_locate_center")
    end

    if math.abs(x) < thirdTerrainSize then
        if z < 1 then
            return g_i18n:getText("aim_locate_n")
        end
        return g_i18n:getText("aim_locate_s")
    end

    if math.abs(z) < thirdTerrainSize then
        if x < 1 then
            return g_i18n:getText("aim_locate_w")
        end
        return g_i18n:getText("aim_locate_e")
    end

    if x < 1 then
        if z < 1 then
            return(g_i18n:getText("aim_locate_nw"))
        end
        return g_i18n:getText("aim_locate_sw")
    end

    if z < 1 then
        return g_i18n:getText("aim_locate_ne")
    end

    return g_i18n:getText("aim_locate_se")
end

function AIMGuiPlaceableFrame:onButtonWarpPlaceable()
    local dropHeight    = 1.2
    local thisPlace     = self.placables[self.placeList.selectedIndex]
    local foundHotSpot  = nil
    local warpX, warpZ

    if thisPlace.spec_hotspots ~= nil and thisPlace.spec_hotspots.mapHotspots ~= nil then
        for _, mapHotSpot in ipairs(thisPlace.spec_hotspots.mapHotspots) do
            if not foundHotSpot and mapHotSpot.teleportWorldX ~= nil and mapHotSpot.teleportWorldZ ~= nil then
                foundHotSpot = "warpHotSpot"
                warpX        = mapHotSpot.teleportWorldX
                warpZ        = mapHotSpot.teleportWorldZ
            end
        end
    end
    if not foundHotSpot and thisPlace.spec_clearAreas ~= nil and thisPlace.spec_clearAreas.areas ~= nil then
        for _, thisArea in ipairs(thisPlace.spec_clearAreas.areas) do
            if not foundHotSpot and thisArea.start ~= nil then
                foundHotSpot    = "clearArea"
                warpX, _, warpZ = getWorldTranslation(thisArea.start)
            end
        end
    end
    if not foundHotSpot and thisPlace.spec_placement ~= nil and thisPlace.spec_placement.testAreas ~= nil then
        for _, thisArea in ipairs(thisPlace.spec_placement.testAreas) do
            if not foundHotSpot and thisArea.startNode ~= nil then
                foundHotSpot    = "testArea"
                warpX, _, warpZ = getWorldTranslation(thisArea.startNode)
            end
        end
    end
    if not foundHotSpot then
        foundHotSpot    = "fallback"
        warpX, _, warpZ = getWorldTranslation(thisPlace.rootNode)
    end

    if AdditionalInfo.debug then
        print("~~ AdvancedFarmManager Debug ... warpToPlaceable: " .. tostring(foundHotSpot) .. " " .. tostring(math.floor(warpX)) ..  " / " .. tostring(math.floor(warpZ)))
    end


    if not g_currentMission.controlPlayer and g_currentMission.controlledVehicle ~= nil then
        g_currentMission:onLeaveVehicle(warpX, dropHeight, warpZ, false, false)
    else
        g_currentMission.player:moveTo(warpX, dropHeight, warpZ, false, false)
    end

    g_gui:showGui("")
end


function AIMGuiPlaceableFrame:onButtonSellPlaceable()
    local thisPlace   = self.placables[self.placeList.selectedIndex]
    local sellPrice   = math.floor(thisPlace:getSellPrice())
    local name        = thisPlace:getName()

    if thisPlace.brand ~= nil and thisPlace.brand.title ~= nil and thisPlace.brand.title ~= "None" then
        name = thisPlace.brand.title .. " " .. name
    end

    g_gui:showYesNoDialog({
        --text      = g_i18n:getText("ui_constructionSellConfirmation"), -- 2 position args, 1st is what, 2nd is cash
        text      = string.format(g_i18n:getText("ui_constructionSellConfirmation"), name, g_i18n:formatMoney(sellPrice)),
        callback  = self.onYesNoSellDialog,
        target    = self,
        yesButton = g_i18n:getText("button_yes"),
        noButton  = g_i18n:getText("button_cancel")
    })
end


function AIMGuiPlaceableFrame:onYesNoSellDialog(yes)
    if yes then
        local thisPlace = self.placables[self.placeList.selectedIndex]

        g_client:getServerConnection():sendEvent(SellPlaceableEvent.new(thisPlace, false, false))
    end
end

function AIMGuiPlaceableFrame:getHotspot()
    local thisPlace     = self.placables[self.placeList.selectedIndex]
    local foundHotSpot  = nil

    if thisPlace.spec_hotspots ~= nil and thisPlace.spec_hotspots.mapHotspots ~= nil then
        for _, mapHotSpot in ipairs(thisPlace.spec_hotspots.mapHotspots) do
            if not foundHotSpot and mapHotSpot.teleportWorldX ~= nil and mapHotSpot.teleportWorldZ ~= nil then
                foundHotSpot = mapHotSpot
            end
        end
    end

    return foundHotSpot
end

function AIMGuiPlaceableFrame:onButtonHotspotPlaceable()
    local hotspot     = self:getHotspot()

    if hotspot ~= nil then
        if hotspot == g_currentMission.currentMapTargetHotspot then
            g_currentMission:setMapTargetHotspot()
        else
            g_currentMission:setMapTargetHotspot(hotspot)
        end
    end
end
