--
-- AdvancedFarmManager - Implement Page
--

AIMGuiImplementFrame = {}

local AIMGuiImplementFrame_mt = Class(AIMGuiImplementFrame, TabbedMenuFrameElement)

AIMGuiImplementFrame.CONTROLS = {
    "vehicleList",
    "vehicleIcon",
    "vehicleDetail",
    "miniMap",
    "miniMapBG",
    "aimInfoSubImp",
}


function AIMGuiImplementFrame:new(l10n)
    local self = TabbedMenuFrameElement.new(nil,AIMGuiImplementFrame_mt)

    self.messageCenter      = g_messageCenter
    self.l10n               = l10n
    self.vehicles           = {}

    self:registerControls(AIMGuiImplementFrame.CONTROLS)

    return self
end


function AIMGuiImplementFrame:copyAttributes(src)
    AIMGuiImplementFrame:superClass().copyAttributes(self, src)

    self.ui   = src.ui
    self.l10n = src.l10n
end


function AIMGuiImplementFrame:initialize()
    self.backButtonInfo = {inputAction = InputAction.MENU_BACK}

    self.activateButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_ACTIVATE,
        text        = self.l10n:getText("aim_warp_tool"),
        callback    = function ()
            self:onButtonWarpVehicle()
        end
    }
    self.acceptButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_EXTRA_2,
        text        = self.l10n:getText("button_clean"),
        callback    = function ()
            self:onButtonCleanVehicle()
        end
    }
    self.cancelButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_CANCEL,
        text        = self.l10n:getText("button_sell"),
        callback    = function ()
            self:onButtonSellVehicle()
        end
    }
    self.extra1ButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_EXTRA_1,
        text        = self.l10n:getText("button_repair"),
        callback    = function ()
            self:onButtonRepairVehicle()
        end
    }
    self.extra2ButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_ACCEPT,
        text        = self.l10n:getText("button_repaint"),
        callback    = function ()
            self:onButtonRepaintVehicle()
        end
    }
end


function AIMGuiImplementFrame:onGuiSetupFinished()
    AIMGuiImplementFrame:superClass().onGuiSetupFinished(self)
    self.vehicleList:setDataSource(self)
    self.vehicleDetail:setDataSource(self)
end


function AIMGuiImplementFrame:delete()
    AIMGuiImplementFrame:superClass().delete(self)
    self.messageCenter:unsubscribeAll(self)
end


function AIMGuiImplementFrame:updateMenuButtons()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    self.menuButtonInfo = {
        {
            inputAction = InputAction.MENU_BACK
        }
    }

    if thisVehicle ~= nil then
        local repairPrice   = thisVehicle:getRepairPrice()
        local repaintPrice  = thisVehicle:getRepaintPrice()
        local cleanPrice    = self:vehicleCleanCost(thisVehicle)
        local sellPrice     = math.min(math.floor(thisVehicle:getSellPrice() * EconomyManager.DIRECT_SELL_MULTIPLIER), thisVehicle:getPrice())
        local isLeased      = thisVehicle.propertyState == Vehicle.PROPERTY_STATE_LEASED
        local isBorrowed    = thisVehicle.propertyState == Vehicle.PROPERTY_STATE_MISSION
        local isMPGame      = g_currentMission.missionDynamicInfo.isMultiplayer

        table.insert(self.menuButtonInfo, self.activateButtonInfo)

        if g_currentMission:getHasPlayerPermission("farmManager") and not isBorrowed then
            if not isLeased then
                self.cancelButtonInfo.text = string.format(
                    "%s (%s)",
                    g_i18n:getText("button_sell"),
                    g_i18n:formatMoney(sellPrice, 0, true, true)
                )
            else
                self.cancelButtonInfo.text  = g_i18n:getText("button_return")
            end

            table.insert(self.menuButtonInfo, self.cancelButtonInfo)

            if repairPrice >= 1 then
                self.extra1ButtonInfo.text = g_i18n:getText("button_repair")
                table.insert(self.menuButtonInfo, self.extra1ButtonInfo)
            end

            if repaintPrice >= 1 then
                self.extra2ButtonInfo.text = g_i18n:getText("button_repaint")
                table.insert(self.menuButtonInfo, self.extra2ButtonInfo)
            end

            if isMPGame ~= true and cleanPrice >= 1 then
                self.acceptButtonInfo.text = g_i18n:getText("button_clean")
                table.insert(self.menuButtonInfo, self.acceptButtonInfo)
            end
        end
    end

    self:setMenuButtonInfoDirty()
end


function AIMGuiImplementFrame:onFrameOpen()
    AIMGuiImplementFrame:superClass().onFrameOpen(self)

    if AdditionalInfo.debug then
        print("~~ AdvancedFarmManager Debug ... AIMGuiImplementFrame:onFrameOpen")
    end

    self:rebuildTable()

    self:setSoundSuppressed(true)
    FocusManager:setFocus(self.vehicleList)
    self:setSoundSuppressed(false)

    self.messageCenter:subscribe(SellVehicleEvent, self.onRefreshEvent, self)
    self.messageCenter:subscribe(BuyVehicleEvent, self.onRefreshEvent, self)
    self.messageCenter:subscribe(MessageType.VEHICLE_REPAIRED, self.onRefreshEvent, self)
    self.messageCenter:subscribe(MessageType.VEHICLE_REPAINTED, self.onRefreshEvent, self)
end


function AIMGuiImplementFrame:onRefreshEvent()
    self:rebuildTable()
end


function AIMGuiImplementFrame:onFrameClose()
    AIMGuiImplementFrame:superClass().onFrameClose(self)

    self.vehicles = {}
    self.messageCenter:unsubscribeAll(self)
end


function AIMGuiImplementFrame:rebuildTable()
    self.vehicles = {}

    if g_currentMission.player ~= nil then
        for _, vehicle in ipairs(g_currentMission.vehicles) do
            local isSelling        = (vehicle.isDeleted ~= nil and vehicle.isDeleted) or (vehicle.isDeleting ~= nil and vehicle.isDeleting)
            local hasAccess        = g_currentMission.accessHandler:canPlayerAccess(vehicle)
            local isProperty       = vehicle.propertyState == Vehicle.PROPERTY_STATE_OWNED or vehicle.propertyState == Vehicle.PROPERTY_STATE_LEASED or vehicle.propertyState == Vehicle.PROPERTY_STATE_MISSION
            local isPallet         = vehicle.typeName == "pallet"
            local isTrain          = vehicle.typeName == "locomotive"
            local isBelt           = vehicle.typeName == "conveyorBelt" or vehicle.typeName == "pickupConveyorBelt"
            local isRidable        = SpecializationUtil.hasSpecialization(Rideable, vehicle.specializations)
            local isSteerImplement = vehicle.spec_attachable ~= nil

            local skipable         = isTrain or isBelt or isRidable or isPallet

            if not isSelling and not skipable and isSteerImplement and hasAccess and vehicle.getSellPrice ~= nil and vehicle.price ~= nil and isProperty then
                table.insert(self.vehicles, vehicle)
            end
        end
    end

    table.sort(self.vehicles, function (k1, k2)
        local k1name  = k1:getFullName()
        local k2name  = k2:getFullName()

        return k1name < k2name
    end )

    self.vehicleList:reloadData()
    self.vehicleDetail:reloadData()
end

function AIMGuiImplementFrame:getNumberOfItemsInSection(list, section)
    if list == self.vehicleList then
        return #self.vehicles
    else
        return 3
    end
end

function AIMGuiImplementFrame:populateCellForItemInSection(list, section, index, cell)
    if list == self.vehicleList then
        local thisVehicle = self.vehicles[index]
        local name        = thisVehicle:getFullName()
        local thisColor   = { 1, 1, 1, 1 }

        if thisVehicle.getDamageAmount ~= nil and thisVehicle:getDamageAmount() > 0.12 then
            thisColor = { 0.413, 0.070, 0.078, 1}
        end

        if thisVehicle.propertyState == Vehicle.PROPERTY_STATE_MISSION then
            thisColor = { 0.223, 0.503, 0.807, 1 }
        end

        cell:getAttribute("statusMine"):setVisible(false)

        if g_currentMission.missionDynamicInfo.isMultiplayer then
            if thisVehicle.ownerFarmId == g_currentMission:getFarmId() then
                cell:getAttribute("statusMine"):setVisible(true)
            end
        end

        cell:getAttribute("title"):setText(name)
        cell:getAttribute("title").textColor = thisColor

        cell:getAttribute("location"):setText(self:getLocation(thisVehicle))
    else
        local selectedIndex      = self.vehicleList.selectedIndex
        local thisVehicle        = self.vehicles[selectedIndex]
        local thisRawAmount      = 0

        if thisVehicle ~= nil then
            -- (re)set visible.
            self.miniMapBG:setVisible(true)
            self.vehicleIcon:setVisible(true)
            self.vehicleDetail:setVisible(true)
            self.aimInfoSubImp:setVisible(true)

            if index == 1 then
                if thisVehicle.getDamageAmount ~= nil then
                    thisRawAmount = thisVehicle:getDamageAmount()
                end

                self:setStatusBarValue(cell:getAttribute("detailBar"), thisRawAmount, thisRawAmount, 0.1, 0.2)
                self:setDetailText(
                    cell, false,
                    "infohud_damage",
                    AIMGuiImplementFrame:rawToPerc(thisRawAmount, false)
                )
            elseif index == 2 then
                if thisVehicle.getWearTotalAmount ~= nil then
                    thisRawAmount = thisVehicle:getWearTotalAmount()
                end

                self:setStatusBarValue(cell:getAttribute("detailBar"), 1-thisRawAmount, thisRawAmount, 0.3, 0.6)
                self:setDetailText(
                    cell, false,
                    "ui_paintCondition",
                    AIMGuiImplementFrame:rawToPerc(thisRawAmount, true)
                )
            elseif index == 3 then
                if thisVehicle.getDirtAmount ~= nil then
                    thisRawAmount = thisVehicle:getDirtAmount()
                end

                self:setStatusBarValue(cell:getAttribute("detailBar"), thisRawAmount, thisRawAmount, 0.3, 0.6)
                self:setDetailText(
                    cell, false,
                    "setting_dirt",
                    AIMGuiImplementFrame:rawToPerc(thisRawAmount, false)
                )
            end
        else
            -- hide UI, no implement
            self.miniMapBG:setVisible(false)
            self.vehicleIcon:setVisible(false)
            self.vehicleDetail:setVisible(false)
            self.aimInfoSubImp:setVisible(false)
        end
    end
end


function AIMGuiImplementFrame:setDetailText(cell, nukeBar, title, level)
    cell:getAttribute("detailTitle"):setText(g_i18n:getText(title))
    cell:getAttribute("detailLevel"):setText(level)

    if nukeBar then
        cell:getAttribute("detailLevel"):applyProfile("aimMenuVehicleDetailLevelNoBar")
        cell:getAttribute("detailTitle"):applyProfile("aimMenuVehicleDetailTitleNoBar")
        cell:getAttribute("detailBarBG"):setVisible(false)
        cell:getAttribute("detailBar"):setVisible(false)
    end
end

function AIMGuiImplementFrame:setStatusBarValue(statusBarElement, value, rawValue, levelGood, levelWarn)

    if rawValue < levelGood then
        statusBarElement:applyProfile("aimMenuVehicleDetailBar")
    elseif rawValue < levelWarn then
        statusBarElement:applyProfile("aimMenuVehicleDetailBarWarning")
    else
        statusBarElement:applyProfile("aimMenuVehicleDetailBarDanger")
    end

    local fullWidth = statusBarElement.parent.absSize[1] - statusBarElement.margin[1] * 2
    local minSize   = 0

    if statusBarElement.startSize ~= nil then
        minSize = statusBarElement.startSize[1] + statusBarElement.endSize[1]
    end

    statusBarElement:setSize(math.max(minSize, fullWidth * math.min(1, value)), nil)
end


function AIMGuiImplementFrame:needPowerString(vehicle, noBraces)
    local storeItem   = g_storeManager:getItemByXMLFilename(vehicle.configFileName)

    StoreItemUtil.loadSpecsFromXML(storeItem)

    if storeItem.specs == nil or storeItem.specs.neededPower == nil then
        return ""
    end

    local powerConfig = 0
    local neededPower = 0

    if vehicle.configurations ~= nil and vehicle.configurations.powerConsumer ~= nil then
        powerConfig = vehicle.configurations.powerConsumer
    end

    if powerConfig == 0 and storeItem.specs.neededPower.base ~= nil then
        neededPower = storeItem.specs.neededPower.base
    else
        if storeItem.specs.neededPower.config[powerConfig] ~= nil then
            neededPower = storeItem.specs.neededPower.config[powerConfig]
        end
    end

    if neededPower > 0 then
        local returnString = string.format(g_i18n:getText("shop_maxPowerValueSingle"), math.floor(neededPower))
        if noBraces == nil or noBraces == false then
            return " [" .. returnString .. "]"
        end
        return returnString
    end

    return ""
end


function AIMGuiImplementFrame:rawToPerc(value, invert)
    if not invert then
        return math.ceil((value)*100) .. " %"
    end
    return math.ceil((1 - value)*100) .. " %"
end


function AIMGuiImplementFrame:vehicleCleanCost(vehicle)
    local thisRawAmount = 0
    if vehicle.getDirtAmount ~= nil then
        thisRawAmount = vehicle:getDirtAmount()
    end

    if thisRawAmount == 0 then
        return 0
    elseif thisRawAmount < 0.25 then
        return 150
    elseif thisRawAmount < 0.5 then
        return 250
    elseif thisRawAmount > 0.9 then
        return 500
    else
        return 400
    end
end


function AIMGuiImplementFrame:onListSelectionChanged(list, section, index)
    if list == self.vehicleList then
        local thisVehicle = self.vehicles[index]
        local storeItem   = g_storeManager:getItemByXMLFilename(thisVehicle.configFileName)

        self.vehicleIcon:setImageFilename(storeItem.imageFilename)

        local vehicleInfoExtra = {}

        if g_currentMission.missionDynamicInfo.isMultiplayer then
            local thisFarmID   = thisVehicle.ownerFarmId
            local thisFarm     = g_farmManager:getFarmById(thisFarmID)
            local thisFarmName = thisFarm.name
            if thisFarmName ~= nil then
                table.insert(vehicleInfoExtra, thisFarmName)
            end
        end

        local powerString = self:needPowerString(thisVehicle, true)

        if powerString ~= "" then
            table.insert(vehicleInfoExtra, powerString)
        end
        if thisVehicle.propertyState == Vehicle.PROPERTY_STATE_MISSION then
            table.insert(vehicleInfoExtra, g_i18n:getText("fieldJob_contract"))
        end
        if thisVehicle.propertyState == Vehicle.PROPERTY_STATE_LEASED then
            table.insert(vehicleInfoExtra, g_i18n:getText("button_lease"))
        end

        self.aimInfoSubImp:setText(table.concat(vehicleInfoExtra, ", "))
        self.vehicleDetail:reloadData()

        local imageFile = g_currentMission.mapImageFilename

        if imageFile ~= self.miniMap.overlay.filename then
            self.miniMap.imageSize = {g_currentMission.terrainSize*2, g_currentMission.terrainSize*2}
            self.miniMap:setImageFilename(imageFile)
        end

        local imageSizeTable      = self.miniMap.imageSize
        local centerFactor        = g_currentMission.terrainSize
        local centerU             = 0
        local centerV             = 0

        if thisVehicle.rootNode ~= nil then
            centerU, _, centerV = getWorldTranslation(thisVehicle.rootNode)
        end

        local startU              = tostring(math.floor(centerFactor + centerU - 200)) .. "px"
        local startV              = tostring(math.floor(centerFactor + centerV - 200)) .. "px"

        local UVString            = startU .. " " .. startV .. " 400px 400px"

        local theseUVs            = GuiUtils.getUVs(UVString, imageSizeTable)

        self.miniMap:setImageUVs(nil, unpack(theseUVs))

        if AdditionalInfo.debug then
            print("~~ AdvancedFarmManager Debug ... newMapPosition: " .. UVString)
        end
    end

    self:updateMenuButtons()
end


function AIMGuiImplementFrame:getLocation(vehicle)
    local fieldNumber = 0
    local isField     = false
    local wx, wy, wz  = 0, 0, 0

    local function getIsOnField()
        if vehicle.components == nil then return false end

        for _, component in pairs(vehicle.components) do
            wx, wy, wz = localToWorld(component.node, getCenterOfMass(component.node))

            local h = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, wx, wy, wz)

            if h-1 > wy then
                break
            end

            local isOnField, _ = FSDensityMapUtil.getFieldDataAtWorldPosition(wx, wy, wz)
            if isOnField then
                isField = true
                return true
            end
        end
        return false
    end
    if getIsOnField() then
        local farmlandId = g_farmlandManager:getFarmlandIdAtWorldPosition(wx, wz)
        if farmlandId ~= nil then

            local foundField = false

            for f=1, #g_fieldManager.fields do

                if foundField then break end

                local field = g_fieldManager.fields[f]

                if field ~= nil and field.farmland ~= nil and field.farmland.id == farmlandId then
                    local fieldId = field.fieldId

                    -- set this as a "fall back" if we don't get a "real" field number below
                    -- this is likely to happen on any enlarged fields, and at the borders of a lot
                    -- of the base game maps.
                    fieldNumber = fieldId

                    for a=1, #field.setFieldStatusPartitions do
                        local b                    = field.setFieldStatusPartitions[a]
                        local x, z, wX, wZ, hX, hZ = b.x0, b.z0, b.widthX, b.widthZ, b.heightX, b.heightZ
                        local distanceMax          = math.max(wX,wZ,hX,hZ)
                        local distance             = MathUtil.vector2Length(wx-x,wz-z);
                        if distance <= distanceMax then
                            fieldNumber = fieldId
                            foundField  = true
                            break
                        end
                    end
                end
            end
        end
    end
    if isField then
        return string.format("F-%03d", fieldNumber)
    else
        return "--"
    end
end

function AIMGuiImplementFrame:onButtonWarpVehicle()
    local dropHeight    = 1.2
    local period        = g_currentMission.environment.currentPeriod
    local dayOfMonth    = g_currentMission.environment.currentDayInPeriod
    local isSouthern    = g_currentMission.environment.daylight.latitude < 0
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if thisVehicle.rootNode == nil then
        return
    end

    local maxSize       = math.max(thisVehicle.size.length, thisVehicle.size.width)
    local offset        = (maxSize / 2) + 1
    local wx, _, wz     = getWorldTranslation(thisVehicle.rootNode)

    local month = period + 2

    if isSouthern then
        month = month + 6
    end

    month = (month - 1) % 12 + 1

    if month == 4 and dayOfMonth == 1 then
        dropHeight = 250
    end

    if not g_currentMission.controlPlayer and g_currentMission.controlledVehicle ~= nil then
        g_currentMission:onLeaveVehicle(wx - offset, dropHeight, wz - offset, false, false)
    else
        g_currentMission.player:moveTo(wx - offset, dropHeight, wz - offset, false, false)
    end

    g_gui:showGui("")
end


function AIMGuiImplementFrame:onButtonSellVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]
    local isLeased      = thisVehicle.propertyState == Vehicle.PROPERTY_STATE_LEASED
    local l10nText      = "ui_youWantToSellVehicle"

    if isLeased then
        l10nText = "ui_youWantToReturnVehicle"
    end

    g_gui:showYesNoDialog({
        text      = g_i18n:getText(l10nText),
        callback  = self.onYesNoSellDialog,
        target    = self,
        yesButton = g_i18n:getText("button_yes"),
        noButton  = g_i18n:getText("button_cancel")
    })
end


function AIMGuiImplementFrame:onYesNoSellDialog(yes)
    if yes then
        local selectedIndex = self.vehicleList.selectedIndex
        local thisVehicle   = self.vehicles[selectedIndex]

        g_client:getServerConnection():sendEvent(SellVehicleEvent.new(thisVehicle, EconomyManager.DIRECT_SELL_MULTIPLIER, true))
    end
end


function AIMGuiImplementFrame:onButtonRepairVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if thisVehicle ~= nil and thisVehicle:getRepairPrice(true) >= 1 then
        g_gui:showYesNoDialog({
            text     = string.format(g_i18n:getText("ui_repairDialog"), g_i18n:formatMoney(thisVehicle:getRepairPrice(true))),
            callback = self.onYesNoRepairDialog,
            target   = self,
            yesSound = GuiSoundPlayer.SOUND_SAMPLES.CONFIG_WRENCH
        })

        return true
    else
        return false
    end
end


function AIMGuiImplementFrame:onButtonRepaintVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if thisVehicle ~= nil and thisVehicle:getRepaintPrice() >= 1 then
        g_gui:showYesNoDialog({
            text     = string.format(g_i18n:getText("ui_repaintDialog"), g_i18n:formatMoney(thisVehicle:getRepaintPrice())),
            callback = self.onYesNoRepaintDialog,
            target   = self,
            yesSound = GuiSoundPlayer.SOUND_SAMPLES.CONFIG_SPRAY
        })

        return true
    else
        return false
    end
end


function AIMGuiImplementFrame:onButtonCleanVehicle()
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]
    local thisCost      = self:vehicleCleanCost(thisVehicle)

    if thisVehicle ~= nil and thisCost >= 1 then
        g_gui:showYesNoDialog({
            text     = string.format(g_i18n:getText("aim_cleanDialog"), g_i18n:formatMoney(thisCost)),
            callback = self.onYesNoCleanDialog,
            target   = self,
            yesSound = GuiSoundPlayer.SOUND_SAMPLES.CONFIG_SPRAY
        })

        return true
    else
        return false
    end
end


function AIMGuiImplementFrame:onYesNoCleanDialog(yes)
    if yes then
        local selectedIndex = self.vehicleList.selectedIndex
        local thisVehicle   = self.vehicles[selectedIndex]

        if g_currentMission:getMoney() <  self:vehicleCleanCost(thisVehicle) then
            g_gui:showInfoDialog({
                text = g_i18n:getText("shop_messageNotEnoughMoneyToBuy")
            })
        else
            self:cleanVehicle(thisVehicle)
        end

        self:rebuildTable()
    end
end


function AIMGuiImplementFrame:cleanVehicle(vehicle)
    if vehicle ~= nil and vehicle.spec_washable ~= nil and vehicle.spec_washable.washableNodes ~= nil then
        for _, nodeData in ipairs(vehicle.spec_washable.washableNodes) do
            vehicle.spec_washable:setNodeDirtAmount(nodeData, 0, true)
        end
    end
end


function AIMGuiImplementFrame:onYesNoRepairDialog(yes)
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if yes then
        if g_currentMission:getMoney() < thisVehicle:getRepairPrice() then
            g_gui:showInfoDialog({
                text = g_i18n:getText("shop_messageNotEnoughMoneyToBuy")
            })
        else
            g_client:getServerConnection():sendEvent(WearableRepairEvent.new(thisVehicle, true))
        end
    end
end


function AIMGuiImplementFrame:onYesNoRepaintDialog(yes)
    local selectedIndex = self.vehicleList.selectedIndex
    local thisVehicle   = self.vehicles[selectedIndex]

    if yes then
        if g_currentMission:getMoney() < thisVehicle:getRepaintPrice() then
            g_gui:showInfoDialog({
                text = g_i18n:getText("shop_messageNotEnoughMoneyToBuy")
            })
        else
            g_client:getServerConnection():sendEvent(WearableRepaintEvent.new(thisVehicle, true))
        end
    end
end