--
-- AdvancedFarmManager - Placeable Page
--

AIMGuiFieldFrame = {}

local AIMGuiFieldFrame_mt = Class(AIMGuiFieldFrame, TabbedMenuFrameElement)

AIMGuiFieldFrame.CONTROLS = {
    "placeList",
}


function AIMGuiFieldFrame:new(l10n)
    local self = TabbedMenuFrameElement.new(nil,AIMGuiFieldFrame_mt)

    self.messageCenter      = g_messageCenter
    self.l10n               = l10n
    self.fields             = {}

    self:registerControls(AIMGuiFieldFrame.CONTROLS)

    return self
end


function AIMGuiFieldFrame:copyAttributes(src)
    AIMGuiFieldFrame:superClass().copyAttributes(self, src)

    self.ui   = src.ui
    self.l10n = src.l10n
end


function AIMGuiFieldFrame:initialize()
    self.backButtonInfo = {inputAction = InputAction.MENU_BACK}

    self.activateButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_ACTIVATE,
        text        = self.l10n:getText("aim_warp_loc"),
        callback    = function ()
            self:onButtonWarpField()
        end
    }
    self.hotspotButtonInfo = {
        profile     = "buttonActivate",
        inputAction = InputAction.MENU_ACCEPT,
        text        = self.l10n:getText("action_tag"),
        callback    = function ()
            self:onButtonHotspotField()
        end
    }
end


function AIMGuiFieldFrame:onGuiSetupFinished()
    AIMGuiFieldFrame:superClass().onGuiSetupFinished(self)
    self.placeList:setDataSource(self)
end


function AIMGuiFieldFrame:delete()
    AIMGuiFieldFrame:superClass().delete(self)
    self.messageCenter:unsubscribeAll(self)
end


function AIMGuiFieldFrame:updateMenuButtons()
    self.menuButtonInfo = {
        {
            inputAction = InputAction.MENU_BACK
        }
    }

    if self.placeList.selectedIndex > 0 then
        table.insert(self.menuButtonInfo, self.activateButtonInfo)
        table.insert(self.menuButtonInfo, self.hotspotButtonInfo)
    end

    self:setMenuButtonInfoDirty()
end


function AIMGuiFieldFrame:onFrameOpen()
    AIMGuiFieldFrame:superClass().onFrameOpen(self)

    if AdditionalInfo.debug then
        print("~~ AdvancedFarmManager Debug ... AIMGuiFieldFrame:onFrameOpen")
    end

    self:rebuildTable()

    self:setSoundSuppressed(true)
    FocusManager:setFocus(self.placeList)
    self:setSoundSuppressed(false)

    self.messageCenter:subscribe(MessageType.FARM_PROPERTY_CHANGED, self.onRefreshEvent, self)
end


function AIMGuiFieldFrame:onRefreshEvent()
    self:rebuildTable()
end


function AIMGuiFieldFrame:onFrameClose()
    AIMGuiFieldFrame:superClass().onFrameClose(self)

    self.fields = {}
    self.messageCenter:unsubscribeAll(self)
end


function AIMGuiFieldFrame:rebuildTable()
    self.fields = {}

    for _, field in ipairs(g_fieldManager:getFields()) do
        table.insert(self.fields, field)
    end

    self.placeList:reloadData()
end

function AIMGuiFieldFrame:getNumberOfItemsInSection(list, section)
    if list == self.placeList then
        return #self.fields
    else
        return 0
    end
end

function AIMGuiFieldFrame:populateCellForItemInSection(list, section, index, cell)
    if list == self.placeList then
        local isMPGame          = g_currentMission.missionDynamicInfo.isMultiplayer
        local currentMissions   = g_missionManager:getMissionsList(g_currentMission:getFarmId())
        local thisField         = self.fields[index]
        local thisFieldFarmland = thisField.farmland.id
        local ownerFarmId       = g_farmlandManager:getFarmlandOwner(thisFieldFarmland)
        local fruitDesc         = g_fruitTypeManager:getFruitTypeByIndex(thisField.fruitType)
        local farmColor         = { 1, 1, 1, 1 }
        local farmName
        local fruitName         = "--"

        if ownerFarmId == g_currentMission:getFarmId() and ownerFarmId ~= FarmManager.SPECTATOR_FARM_ID then
            farmName = g_i18n:getText("fieldInfo_ownerYou")
            if g_currentMission.missionDynamicInfo.isMultiplayer then
                local farm = g_farmManager:getFarmById(ownerFarmId)

                if farm ~= nil then
                    farmName  = farm.name
                    farmColor = Farm.COLORS[farm.color]
                end
            end
        elseif ownerFarmId == AccessHandler.EVERYONE or ownerFarmId == AccessHandler.NOBODY then
            local farmland = g_farmlandManager:getFarmlandById(thisFieldFarmland)
            farmColor = { 0.5, 0.5, 0.5, 1 }
            if farmland == nil then
                farmName = g_i18n:getText("fieldInfo_ownerNobody")
            else
                local npc = farmland:getNPC()
                farmName = npc ~= nil and npc.title or "Unknown"
            end
        else
            local farm = g_farmManager:getFarmById(ownerFarmId)

            if farm ~= nil then
                farmName  = farm.name
                farmColor = Farm.COLORS[farm.color]
            else
                farmName = "Unknown"
                farmColor = { 0.5, 0.5, 0.5, 1 }
            end
        end

        if isMPGame ~= true and fruitDesc ~= nil and fruitDesc.fillType ~= nil then
            fruitName = fruitDesc.fillType.title

            local minHarvest  = fruitDesc.minHarvestingGrowthState
            local maxHarvest  = fruitDesc.maxHarvestingGrowthState
            local maxGrowing  = fruitDesc.minHarvestingGrowthState - 1
            local withered    = fruitDesc.maxHarvestingGrowthState + 1
            local cutState    = fruitDesc.cutState
            local maxAskState = math.max(withered, cutState)

            local maxGrowthState = 0
            local maxArea = 0

            if fruitDesc.minPreparingGrowthState >= 0 then
                maxGrowing = math.min(maxGrowing, fruitDesc.minPreparingGrowthState - 1 )
            end

            if fruitDesc.maxPreparingGrowthState >= 0 then
                withered = fruitDesc.maxPreparingGrowthState + 1
            end

            local x, z = FieldUtil.getMeasurementPositionOfField(thisField)

            for i = 0, maxAskState do
                local area, _ = FieldUtil.getFruitArea(x - 1, z - 1, x + 1, z - 1, x - 1, z + 1, FieldUtil.FILTER_EMPTY, FieldUtil.FILTER_EMPTY, thisField.fruitType, i, i, 0, 0, 0, false)

                if maxArea < area then
                    maxGrowthState = i
                    maxArea = area
                end
            end

            if AdditionalInfo.debug then
                print("~~ AdvancedFarmManager Debug ... Field Update: " .. tostring(thisField.fieldId) .. " FruitState:" .. tostring(maxGrowthState))
            end

            local text = ""

            if maxGrowthState == fruitDesc.cutState then
                text = g_i18n:getText("ui_growthMapCut")
            elseif maxGrowthState == withered then
                text = g_i18n:getText("ui_growthMapWithered")
            elseif maxGrowthState > 0 and maxGrowthState <= maxGrowing then
                text = g_i18n:getText("ui_growthMapGrowing")
            elseif fruitDesc.minPreparingGrowthState >= 0 and fruitDesc.minPreparingGrowthState <= maxGrowthState and maxGrowthState <= fruitDesc.maxPreparingGrowthState then
                text = g_i18n:getText("ui_growthMapReadyToPrepareForHarvest")
            elseif minHarvest <= maxGrowthState and maxGrowthState <= maxHarvest then
                text = g_i18n:getText("ui_growthMapReadyToHarvest")
            end

            if text ~= "" then
                fruitName = text .. ", " .. fruitName
            end
        end

        for _, thisMission in ipairs(currentMissions) do
            if thisMission.field ~= nil and thisMission.field.fieldId ~= nil and thisMission.field.fieldId == thisField.fieldId then
                local missionType = "Unknown"
                if thisMission.type.name == "fertilize" then
                    missionType = g_i18n:getText("fieldJob_jobType_fertilizing")
                elseif thisMission.type.name == "spray" then
                    missionType = g_i18n:getText("fieldJob_jobType_spraying")
                elseif thisMission.type.name == "cultivate" then
                    missionType = g_i18n:getText("fieldJob_jobType_cultivating")
                elseif thisMission.type.name == "sow" then
                    missionType = g_i18n:getText("fieldJob_jobType_sowing")
                elseif thisMission.type.name == "harvest" then
                    missionType = g_i18n:getText("fieldJob_jobType_harvesting")
                elseif thisMission.type.name == "mow_bale" then
                    missionType = g_i18n:getText("fieldJob_jobType_baling")
                elseif thisMission.type.name == "plow" then
                    missionType = g_i18n:getText("fieldJob_jobType_plowing")
                elseif thisMission.type.name == "weed" then
                    missionType = g_i18n:getText("fieldJob_jobType_weeding")
                end

                if thisMission.farmId == nil then
                    farmName = farmName .. " - " .. string.format(g_i18n:getText("aim_contract_avail"), missionType)
                else
                    farmName = farmName .. " - " .. string.format(g_i18n:getText("aim_contract_active"), missionType)
                end
            end
            end

        cell:getAttribute("title"):setText(tostring(thisField.fieldId))
        cell:getAttribute("owner"):setText(farmName)
        cell:getAttribute("owner").textColor = farmColor
        cell:getAttribute("location"):setText(self:locationToText(thisField) .. ", " .. tostring(math.floor(thisField.fieldArea*100)/100) .. " ha")
        cell:getAttribute("crop"):setText(fruitName)
    end
end


function AIMGuiFieldFrame:onListSelectionChanged(list, section, index)
    self:updateMenuButtons()
end


function AIMGuiFieldFrame:locationToText(thisPlace)
    local thirdTerrainSize = g_currentMission.terrainSize / 6
    local x, _, z   = getWorldTranslation(thisPlace.nameIndicator)

    if x == 0 and z == 0 then
        return "--"
    end

    if math.abs(x) < thirdTerrainSize and math.abs(z) < thirdTerrainSize then
        return g_i18n:getText("aim_locate_center")
    end

    if math.abs(x) < thirdTerrainSize then
        if z < 1 then
            return g_i18n:getText("aim_locate_n")
        end
        return g_i18n:getText("aim_locate_s")
    end

    if math.abs(z) < thirdTerrainSize then
        if x < 1 then
            return g_i18n:getText("aim_locate_w")
        end
        return g_i18n:getText("aim_locate_e")
    end

    if x < 1 then
        if z < 1 then
            return(g_i18n:getText("aim_locate_nw"))
        end
        return g_i18n:getText("aim_locate_sw")
    end

    if z < 1 then
        return g_i18n:getText("aim_locate_ne")
    end

    return g_i18n:getText("aim_locate_se")
end

function AIMGuiFieldFrame:onButtonWarpField()
    local dropHeight       = 1.2
    local thisPlace        = self.fields[self.placeList.selectedIndex]
    local warpX, _,  warpZ = getWorldTranslation(thisPlace.nameIndicator)

    if not g_currentMission.controlPlayer and g_currentMission.controlledVehicle ~= nil then
        g_currentMission:onLeaveVehicle(warpX, dropHeight, warpZ, false, false)
    else
        g_currentMission.player:moveTo(warpX, dropHeight, warpZ, false, false)
    end

    g_gui:showGui("")
end

function AIMGuiFieldFrame:onButtonHotspotField()
    local thisPlace        = self.fields[self.placeList.selectedIndex]
    local hotspot          = thisPlace.mapHotspot

    if hotspot == g_currentMission.currentMapTargetHotspot then
        g_currentMission:setMapTargetHotspot()
    else
        g_currentMission:setMapTargetHotspot(hotspot)
    end
end


